/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *     
 *******************************************************************************/
package org.eclipse.vtp.editor.actions;

import org.eclipse.jface.text.BadLocationException;
import org.eclipse.jface.text.IDocument;
import org.eclipse.swt.graphics.Point;
import org.eclipse.wst.sse.core.internal.provisional.IModelManager;
import org.eclipse.wst.sse.core.internal.provisional.IStructuredModel;
import org.eclipse.wst.sse.core.internal.provisional.IndexedRegion;
import org.eclipse.wst.sse.core.internal.provisional.StructuredModelManager;
import org.eclipse.wst.sse.core.internal.provisional.text.IStructuredDocumentRegion;
import org.eclipse.wst.sse.ui.StructuredTextEditor;
import org.eclipse.wst.xml.core.internal.provisional.document.IDOMNode;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;

/**
 * Utility methods for actions that depend on XML context
 */
public final class ActionSupport {

	/**
	 * hide constructor
	 */
	private ActionSupport() {
	}

	/**
	 * This returns true if your cursor is in the body of an element. Specify
	 * the element name, using the "elementName" parameter.
	 * @param viewer	viewer
	 * @param elementName	element name
	 * @return true if cursor is in named element
	 */
	public static boolean isBodyOfElement(StructuredTextEditor textEditor,
			String elementName) {
		char c;
		char cPlus;

		/** Get the character at the current cursor position */
		try {


			Point s = textEditor.getTextViewer().getTextWidget().getSelection();			

			
			
//			Point s = textEditor.getSelectionRange();
			IDocument document = textEditor.getDocumentProvider().getDocument(textEditor.getEditorInput());
		
			c = document.getChar(s.x);
			cPlus = document.getChar(s.x + 1);
		} catch (BadLocationException e) {
			return false;
		}

		IDOMNode xmlNode = currentPosGetXMLNode(textEditor);

		if (xmlNode != null) {
			IDOMNode parentXMLNode = (IDOMNode) xmlNode.getParentNode();
			if (parentXMLNode != null) {
				boolean isTextNode = (xmlNode.getNodeType() == IDOMNode.TEXT_NODE);
				boolean isElementParent = (parentXMLNode.getNodeName()
						.compareTo(elementName) == 0);

				if (!isTextNode && c == '<') {
					isTextNode = true;
					if (cPlus == '/') {
						isElementParent = (xmlNode.getNodeName().compareTo(
								elementName) == 0);
					}
				}
				return (isTextNode && isElementParent);
			} else {
				return false;
			}
		} else {
			return false;
		}
	}

	/**
	 * This returns the Attribute value (based on the position of your cursor)
	 * of the specified Attribute, or <code>null</code> if the node is null,
	 * if the attribute does not exist, if the value is null itself or the empty
	 * string.
	 * 
	 * @param textEditor	viewer
	 * @param attrName	attribute name
	 * @return attribute value or null
	 */
	public static String currentPosGetAttributeValue(
			StructuredTextEditor textEditor, String attrName) {
		IDOMNode xmlNode = currentPosGetXMLNode(textEditor);

		// If the node is null return null
		if (xmlNode == null) {
			return null;
		}
		// verify node has attributes, if not return null
		NamedNodeMap map = xmlNode.getAttributes();
		if (map == null) {
			return null;
		}
		// verify node has the attribute specified by "attrName", if not return
		// null
		Node attrNode = map.getNamedItem(attrName);
		if (attrNode == null) {
			return null;
		}
		// verify if value of attribute is null, if so return null
		if (attrNode.getNodeValue() == null) {
			return null;
		}
		// verify if value of attribute is the empty string, if so return null
		String attrValue = attrNode.getNodeValue().trim();

		if (attrValue.compareTo("") == 0) { //$NON-NLS-1$
			return null; 
		}

		return attrValue;
	}

	/**
	 * This returns the XMLnode from the editor, based on the cursor
	 * position. Please note, the return value could be "null"
	 * @param textEditor viewer
	 * @return xml node under cursor
	 */
	public static IDOMNode currentPosGetXMLNode(StructuredTextEditor textEditor) {

		Point s = textEditor.getTextViewer().getTextWidget().getSelection();			
		
//		Point s = textEditor.getSelectionRange();
		int absoluteStart = s.x;
		IDOMNode xmlNode = (IDOMNode) getNodeAt(textEditor,
				absoluteStart);
		return ((xmlNode == null)) ? null : xmlNode;
	}
	
	public static IndexedRegion getNodeAt(StructuredTextEditor textEditor, int documentOffset) {

		if (textEditor == null)
			return null;

		IndexedRegion node = null;
		IModelManager mm = StructuredModelManager.getModelManager();
		IStructuredModel model = null;
		if (mm != null)
			model = mm.getExistingModelForRead(textEditor.getDocumentProvider().getDocument(textEditor.getEditorInput()));
		try {
			if (model != null) {
				int lastOffset = documentOffset;
				node = model.getIndexedRegion(documentOffset);
				while (node == null && lastOffset >= 0) {
					lastOffset--;
					node = model.getIndexedRegion(lastOffset);
				}
			}
		} finally {
			if (model != null)
				model.releaseFromRead();
		}
		return node;
	}

	/**
	 * This returns true if your cursor is in an element with a src attribute,
	 * and that src attribute is in double quotes (ie <grammar
	 * src="example.gram"/>) Specify the element name, using the "elementName"
	 * parameter.
	 * 
	 * @param viewer		viewer
	 * @param elementName	element name or null if don't care (all elements are ok)
	 * @param attributeName	attribute name
	 * @return true if cursor is in a named alement with given attribute
	 */
	public static boolean isElementWithAttribute(StructuredTextEditor textEditor,
			String elementName, String attributeName) {
		if (textEditor == null) return false;
		// The user is not on a tag
		if (currentPosGetTagName(textEditor) == null) {
			return false;
		}
		if ((elementName == null || currentPosGetTagName(textEditor).compareTo(elementName) == 0)
				&& currentPosGetAttributeValue(textEditor, attributeName) != null) {
			IDOMNode node = currentPosGetXMLNode(textEditor);
			IStructuredDocumentRegion flatNode = (IStructuredDocumentRegion) node
					.getFirstStructuredDocumentRegion();
			String startTag = flatNode.getText();
			int srcIndex = startTag.indexOf(attributeName);

			// There is a first double quote and an ending double quote, return
			// true
			int firstDblQuoteIndex = startTag.indexOf('"', srcIndex);
			if (firstDblQuoteIndex != -1) {
				int secondDblQuoteIndex = startTag.indexOf('"',
						firstDblQuoteIndex + 1);
				return (secondDblQuoteIndex != -1);
			} else {
				return false;
			}
		} else {
			return false;
		}
	}

	/**
	 * This returns current Tag name (based on the position of your cursor), or
	 * null if the node is null
	 * @param viewer viewer
	 * @return vurrent tag name
	 */
	public static String currentPosGetTagName(StructuredTextEditor textEditor) {
		IDOMNode xmlNode = currentPosGetXMLNode(textEditor);

		// If the node is null return null
		return (xmlNode == null) ? null : xmlNode.getNodeName();
	}

}
