/*******************************************************************************
 * Copyright (c) 2005 Tellme Networks, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Tellme Networks, Inc. - Initial implementation
 *******************************************************************************/

package org.eclipse.vtp.internal.dialer;


/**
 * Interface describing connections to a dialer object.
 * 
 * This interface outlines the methods for communicating with an active
 * connection attached to a dialer (an object implementing the
 * IVoipDialer interface). The connection is used to dial out a specific
 * number (or SIP URL), and to disconnect when the user has no further need
 * of it. It also allows for checking of the status of the connection itself.
 * 
 * In the dialer model used by the Voice Tools, a VoIP connection is derived
 * from a dialer, and is used to derive calls. The connection does not use a
 * listener as the dialer does by default. Implementations may choose to add
 * listeners for event messages, or tie the dialer listener to the connection.
 * The purpose of the connection is to exist separately from the controlling
 * dialer, and produce call objects that also exist separately from either of
 * the connection or dialer.
 * 
 * A connection should not call disconnect() as long as there are active call
 * objects in existence, unless the call class is equipped to handle such
 * interruptions.
 * 
 * @author rjray
 */
public interface IVoipConnection {
    /// Stable connection state, no errors
    int CONNECTION_OK      = 1;
    /// Connection object is in the process of dialing
    int CONNECTION_DIALING = CONNECTION_OK << 1;
    /// The last operation on/through the connect failed
    int CONNECTION_ERROR   = CONNECTION_DIALING << 1;

    /**
     * Dial the specified number and return an object representing the call.
     * The IVoipCall interface describes the handles used to track active
     * calls. The number is generally going to be 10 digits in length, though
     * use of these interfaces out side the U.S. would have different formats
     * (as would special sequences that used * characters or other codes). The
     * dialer should also support SIP URLs, and the Skype implementation will
     * accept Skype connection strings as well. This
     * method call blocks until the call is actually connected or it fails.
     * In the case of failure, a VoipDialerException is thrown.
     * 
     * @param number The number to dial, generally 10 digits in length
     * @return An object implementing IVoipCall, representing the active call
     * @throws VoipDialerException
     */
    IVoipCall dial(final String number) throws VoipDialerException;

    /**
     * Retrieve the current status of the connection. This is represented as an
     * integer. The status of the connection may or may not reflect the status
     * of the dialer itself.
     * 
     * @return The current status of the connection as an integer
     */
    int status();

    /**
     * Disconnect this object from the dialer. When there will be no more
     * usage of the connection object for dialing, this should be used. If the
     * dialer is not in a state to handle the disconnect, a
     * VoipDialerException may be thrown.
     * 
     * @throws VoipDialerException
     */
    void disconnect() throws VoipDialerException;
}
