/*******************************************************************************
 * Copyright (c) 2005 Tellme Networks, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Tellme Networks, Inc. - Initial implementation
 *******************************************************************************/

package org.eclipse.vtp.internal.dialer;


/**
 * An exception class for error cases in the dialer, connections or calls.
 * 
 * When an error occurs in a lower-level call, the calling methods in any of
 * the dialer, connection or call classes will need to signal this to the
 * application. Using this class allows for passing in an object instance that
 * implements the IVoipDialerListener interface, which will lead to the error
 * message itself being sent to the listener for the user to respond to.
 *
 * @author rjray
 */
public class VoipDialerException extends Exception {
    /**
     * Construct a new exception using the given message, and log it via the
     * listener. In addition to instantiating the new object, the text of the
     * exception is sent to a listener object being used by the user of the
     * dialer itself.
     * 
     * @param listener An object implementing IVoipDialerListener
     * @param message The String containing the error message
     */
    public VoipDialerException(final IVoipDialerListener listener,
            final String message)
    {
        super(message);
        listener.dialerMessage(message);
    }

    /**
     * Construct a new exception using the given message and existing Throwable
     * object, logging the message via the listener. Creates the exception as
     * per usual with the String and Throwable parameters, but also logs the
     * new message via the listener instance being used by the user by the user
     * of the dialer itself.
     * 
     * @param listener An object implementing IVoipDialerListener
     * @param message String containing the error message
     * @param thrown Existing, lower-level exception that led to this one
     */
    public VoipDialerException(final IVoipDialerListener listener,
            final String message, final Throwable thrown)
    {
        super(message, thrown);
        listener.dialerMessage(message);
    }

    /**
     * Construct a new exception without adding a new message to the running
     * stack, but logging the message from the existing exception to the
     * listener. Creates the exception without any addition text to the
     * cumulative message. The current message content from the given
     * Throwable instance is logged via the listener instance being used by
     * the user of the dialer itself.
     * 
     * @param listener An object implementing IVoipDialerListener
     * @param thrown Existing, lower-level exception that led to this one
     */
    public VoipDialerException(final IVoipDialerListener listener,
            final Throwable thrown)
    {
        super(thrown);
        listener.dialerMessage(thrown.getMessage());
    }
}
