/*******************************************************************************
 * Copyright (c) 2006 Tellme Networks, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Tellme Networks, Inc. - Initial implementation
 *******************************************************************************/

package org.eclipse.vtp.launching.tellme;

import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Locale;

import org.eclipse.debug.core.DebugEvent;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.debug.core.ILaunch;
import org.eclipse.debug.core.model.IProcess;
import org.eclipse.vtp.launching.VoiceXMLBrowserProcess;

/**
 * class CallState provides a shared resource that allows the independent
 * threads watching for call completion and fetching log entries to report their
 * status to the other.
 *
 * The launch is truly done when both threads have finished
 *
 * @author mgreenawalt
 *
 */
public class CallState {
    private static boolean callWatchThreadDone = false;
    private static boolean logFetchThreadDone = false;
    private static boolean logListThreadDone = false;
    private static boolean callWatchSuccessful = true;
    private static boolean logFetchSuccessful = true;
    private static boolean logListSuccessful = true;
    private static String callWatchReason = "";
    private static String logFetchReason = "";
    private static String logListReason = "";
    private static String callID = null; // studio call id for use in finish
    private static StringBuffer logList = null;     // hold logList between threads
    private static boolean stopButtonPushed = false;
    
    final private static SimpleDateFormat time =
        new SimpleDateFormat("mm:ss.SSS", Locale.getDefault());		//$NON-NLS-1$
    private final static boolean DEBUGGING =
        TellmePlugin.getDefault().isDebugging();
    
    private static Object synchObj = new Object();

    public static boolean launchDone() {
        if (DEBUGGING) {
            System.out.println(time.format(new Date()) +
                    ": CallState.launchDone() called");		//$NON-NLS-1$
            System.out.flush();
        }
        synchronized (synchObj) {
            return callWatchThreadDone && logListThreadDone && logFetchThreadDone;
        }
    }

    public static void initCallWatch() {
        if (DEBUGGING) {
            System.out.println(time.format(new Date()) +
                    ": CallState.initCallWatch() called");		//$NON-NLS-1$
            System.out.flush();
        }
        synchronized (synchObj) {
            callWatchThreadDone = false;
            callWatchSuccessful = true;
            callWatchReason = "";
        }
    }

    public static void initLogFetch() {
        if (DEBUGGING) {
            System.out.println(time.format(new Date()) +
                    ": CallState.initLogFetch() called");		//$NON-NLS-1$
            System.out.flush();
        }
        synchronized (synchObj) {
            logFetchThreadDone = false;
            logFetchSuccessful = true;
            logFetchReason = "";
        }
    }

    public static void initLogList() {
        if (DEBUGGING) {
            System.out.println(time.format(new Date()) +
                    ": CallState.initLogList() called");		//$NON-NLS-1$
            System.out.flush();
        }
        synchronized (synchObj) {
            logListThreadDone = false;
            logListSuccessful = true;
            logListReason = "";
        }
    }

    public static void setCallWatchDone() {
        if (DEBUGGING) {
            System.out.println(time.format(new Date()) +
                    ": CallState.setCallWatchDone() called");		//$NON-NLS-1$
            System.out.flush();
        }
        synchronized (synchObj) {
            callWatchThreadDone = true;
        }
    }

    public static void setLogFetchDone() {
        if (DEBUGGING) {
            System.out.println(time.format(new Date()) +
                    ": CallState.setLogFetchDone() called");		//$NON-NLS-1$
            System.out.flush();
        }
        synchronized (synchObj) {
            logFetchThreadDone = true;
        }
    }

    public static void setLogListDone() {
        if (DEBUGGING) {
            System.out.println(time.format(new Date()) +
                    ": CallState.setLogListDone() called");		//$NON-NLS-1$
            System.out.flush();
        }
        synchronized (synchObj) {
            logListThreadDone = true;
        }
    }

    public static Object getSynch() {
        return synchObj;
    }

    public static void setLogList(final StringBuffer loglist) {
        if (DEBUGGING) {
            System.out.println(time.format(new Date()) +
                    ": CallState.setLogList() called");		//$NON-NLS-1$
            System.out.flush();
        }
        logList = loglist;
    }

    public static StringBuffer getLogList() {
        if (DEBUGGING) {
            System.out.println(time.format(new Date()) +
                    ": CallState.getLogList() called");		//$NON-NLS-1$
            System.out.flush();
        }
        return logList;
    }

    public static void reportCallWatchFail(final String reason) {
        if (DEBUGGING) {
            System.out.println(time.format(new Date()) +
                    ": CallState.reportCallWatchFail() called");		//$NON-NLS-1$
            System.out.flush();
        }
        synchronized (synchObj) {
            callWatchSuccessful = false;
            callWatchReason = reason;
        }
    }

    public static void reportCallWatchSucceed(final String reason) {
        if (DEBUGGING) {
            System.out.println(time.format(new Date()) +
                    ": CallState.reportCallWatchSucceed() called");		//$NON-NLS-1$
            System.out.flush();
        }
        synchronized (synchObj) {
            callWatchSuccessful = true;
            callWatchReason = reason;
        }
    }

    public static void reportLogFetchFail(final String reason) {
        if (DEBUGGING) {
            System.out.println(time.format(new Date()) +
                    ": CallState.reportLogFetchFail() called");		//$NON-NLS-1$
            System.out.flush();
        }
        synchronized (synchObj) {
            logFetchSuccessful = false;
            logFetchReason = reason;
        }
    }

    public static void reportLogFetchSucceed(final String reason) {
        if (DEBUGGING) {
            System.out.println(time.format(new Date()) +
                    ": CallState.reportLogFetchSucceed() called");		//$NON-NLS-1$
            System.out.flush();
        }
        synchronized (synchObj) {
            logFetchSuccessful = true;
            logFetchReason = reason;
        }
    }

    public static void reportLogListFail(final String reason) {
        if (DEBUGGING) {
            System.out.println(time.format(new Date()) +
                    ": CallState.reportLogListFail() called");		//$NON-NLS-1$
            System.out.flush();
        }
        synchronized (synchObj) {
            logListSuccessful = false;
            logListReason = reason;
        }
    }

    public static void reportLogListSucceed(final String reason) {
        if (DEBUGGING) {
            System.out.println(time.format(new Date()) +
                    ": CallState.reportLogListSucceed() called");		//$NON-NLS-1$
            System.out.flush();
        }
        synchronized (synchObj) {
            logListSuccessful = true;
            logListReason = reason;
        }
    }

    public static void setCallID(final String callid) {
        if (DEBUGGING) {
            System.out.println(time.format(new Date()) +
                    ": CallState.setCallID() called");		//$NON-NLS-1$
            System.out.flush();
        }
        callID = callid;
    }

    public static void endLaunch() {
        if (DEBUGGING) {
            System.out.println(time.format(new Date()) +
                    ": CallState.endLaunch() called");		//$NON-NLS-1$
            // (new Throwable("from where?")).printStackTrace(System.out);
            System.out.flush();
        }
        // locate this browserprocess
        VoiceXMLBrowserProcess activeBrowser = null;
        final ILaunch[] launches = DebugPlugin.getDefault().getLaunchManager()
            .getLaunches();
        if (launches != null) {
            for (int i = 0; i < launches.length; i++) {
                final IProcess[] threads = launches[i].getProcesses();
                if (threads != null && threads.length > 0
                    && threads[0] instanceof VoiceXMLBrowserProcess) {
                    activeBrowser = (VoiceXMLBrowserProcess) threads[0];
                }
            }
        }

        // we get here if all threads are done
        if (callWatchSuccessful && logFetchSuccessful && logListSuccessful) {
            DisplayActions.showEndOfCall(callID);
        }
        else {
            // to here means some or all threads had problem
            DisplayActions.showError(("OK".equals(callWatchReason)?"":callWatchReason)+(("".equals(callWatchReason)||"OK".equals(callWatchReason))?"":"\n")+logListReason+("".equals(logListReason)?"":"\n")+logFetchReason);
        }
        
        activeBrowser.getVoiceXMLBrowser().stop();

        activeBrowser.setTerminated(true);
        DebugEvent[] event = new DebugEvent[2];
        event[0] = new DebugEvent(activeBrowser, DebugEvent.TERMINATE);
		event[1] = new DebugEvent(activeBrowser.getLaunch(), DebugEvent.CHANGE);
        DebugPlugin.getDefault().fireDebugEventSet(event);
        
    }
    
    public static void setStopping(boolean state){
    	if (DEBUGGING){
    		System.out.println(time.format(new Date()) +": CallState.setStopping() called"); //$NON-NLS-1$
    		System.out.flush();
    	}
    	stopButtonPushed = state;
    }
    
    public static boolean isStopping(){
    	if (DEBUGGING){
    		System.out.println(time.format(new Date()) +": CallState.isStopping() called: value = "+stopButtonPushed); //$NON-NLS-1$
    		System.out.flush();
    	}
    	return stopButtonPushed;
    }
}
