/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *
 *******************************************************************************/
package org.eclipse.vtp.catalog;

import org.eclipse.core.runtime.ILog;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;

/**
 * Convenience class to log messages to the log file.
 */
public final class Logger {
	
	/**
	 * Hide constructor
	 */
	private Logger() { }
	
	/**
	 * Plugin ID
	 */
	private static final String PLUGINID = "org.eclipse.vtp.editor"; //$NON-NLS-1$

	/**
	 * The Platform logger
	 */
	private static ILog logger = Platform.getLog(Platform.getBundle(PLUGINID));

	/**
	 * IStatus.OK, repeated for convenience
	 */
	public static final int OK = IStatus.OK; // 0
	
	/**
	 * IStatus.INFO, repeated for convenience
	 */
	public static final int INFO = IStatus.INFO; // 1
	
	/**
	 * IStatus.WARNING, repeated for convenience
	 */
	public static final int WARNING = IStatus.WARNING; // 2

	/**
	 * IStatus.ERROR, repeated for convenience
	 */
	public static final int ERROR = IStatus.ERROR; // 4

	/**
	 * DEBUG_START (200)
	 */
	private static final int DEBUG_START = 200;
	
	/**
	 * DEBUG_START + OK
	 */
	public static final int OK_DEBUG = DEBUG_START + OK;
	
	/**
	 * DEBUG_START + INFO
	 */
	public static final int INFO_DEBUG = DEBUG_START + INFO;
	
	/**
	 * DEBUG_START + WARNING
	 */
	public static final int WARNING_DEBUG = DEBUG_START + WARNING;
	
	/**
	 * DEBUG_START + ERROR
	 */
	public static final int ERROR_DEBUG = DEBUG_START + ERROR;

	/**
	 * Adds message to log.
	 * @param level severity level of the message 
	 * (OK, INFO, WARNING, ERROR, OK_DEBUG, INFO_DEBUG, WARNING_DEBUG, ERROR_DEBUG)
	 * @param message text to add to the log
	 * @param exception exception thrown
	 */
	private static void iLog(int level, String message, Throwable exception) {
		if (level == OK_DEBUG || level == INFO_DEBUG || level == WARNING_DEBUG || level == ERROR_DEBUG) {
			if (!isDebugging()) {
				return;
			}
		}

		int severity = IStatus.OK;
		switch (level) {
			case INFO_DEBUG :
			case INFO :
				severity = IStatus.INFO;
				break;
			case WARNING_DEBUG :
			case WARNING :
				severity = IStatus.WARNING;
				break;
			case ERROR_DEBUG :
			case ERROR :
				severity = IStatus.ERROR;
				break;
			default: // silence checkStyle
		}
		message = (message != null) ? message : "null"; //$NON-NLS-1$
		Status statusObj = new Status(severity, PLUGINID, severity, message, exception);
		logger.log(statusObj);
	}

	/**
	 * @return true if the plugin for this logger is debugging
	 */
	public static boolean isDebugging() {
		return Platform.inDebugMode();
	}

	/**
	 * log message with level
	 * @param level	level
	 * @param message	message
	 */
	public static void log(int level, String message) {
		iLog(level, message, null);
	}

	/**
	 * log message and stacktrace with level
	 * @param level	level
	 * @param message	message
	 * @param exception	stacktrace
	 */
	public static void log(int level, String message, Throwable exception) {
		iLog(level, message, exception);
	}

	/**
	 * Log the exception stacktrace and message
	 * @param message	message
	 * @param exception	exception
	 */
	public static void logException(String message, Throwable exception) {
		iLog(ERROR, message, exception);
	}

	/**
	 * log the exception's stacktrace
	 * @param exception exception
	 */
	public static void logException(Throwable exception) {
		iLog(ERROR, exception.getMessage(), exception);
	}
}