/*******************************************************************************
 * Copyright (c) 2005 Tellme Networks, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Tellme Networks, Inc. - Initial implementation
 *******************************************************************************/

package org.eclipse.vtp.internal.dialer;


/**
 * Interface describing a single in-progress call through a dialer connection.
 * 
 * This interface outlines the methods for communication with an active call
 * as instantiated by a connection (an object implementing the IVoipConnection
 * interface). A call runs mostly asynchronously from the perspective of a
 * dialer, except for when the user has to send DTMF data.
 * 
 * In the model being used by the Voice Tools, a call is derived from an
 * active connection. It has no direct link to the dialer instance that is
 * controlling everything. It must communicate upwards by means of the
 * connection.
 * 
 * If a connection is abrubtly severed, and the connection object is not able
 * to signal active call objects, calls may be left in an untrackable state.
 *  
 * @author rjray
 */
public interface IVoipCall {
    /// Stable call state, no errors
    int CALL_OK = 1;
    /// Call is in a connected, active state
    int CALL_ACTIVE = CALL_OK << 1;
    /// Call has completed, and this object is no longer useful
    int CALL_COMPLETED = CALL_ACTIVE << 1;
    /// System has determined that there is no active Skype client
    int CALL_NO_PHONE = CALL_COMPLETED <<1;
    /// The last operation on/through the call object failed
    int CALL_ERROR  = CALL_NO_PHONE << 1;

    /**
     * Retrieve the current status of the call. This is represented as an
     * integer. The status of the call may or may not reflect the status
     * of the parent connection or the dialer itself.
     * 
     * @return The current status of the call as an integer
     */
    int status();

    /**
     * Send a DTMF string along the telephony connection. Instruct the
     * controlling connection to send DTMF data through the line. Most
     * soft-phone implementations will send this as a series of single
     * characters, with a 200-millisecond pause in between each.
     * 
     * @param dtmf DTMF sequence of digits and/or * or # characters
     * @throws VoipDialerException
     */
    void sendDTMF(final String dtmf) throws VoipDialerException;

    /**
     * Send a DTMF string along the telephony connection. Instruct the
     * controlling connection to send DTMF data through the line. Most
     * soft-phone implementations will send this as a series of single
     * characters, with a default pause of 200 milliseconds between each.
     * The value of pause should be an explicit timing value (in milliseconds)
     * that the dialer should use when sending the characters in dtmf.
     * 
     * @param dtmf DTMF sequence of digits and/or * or # characters
     * @param pause The number of milliseconds to pause between digits
     * @throws VoipDialerException
     */
    void sendDTMF(final String dtmf, final int pause)
        throws VoipDialerException;

    /**
     * Hang up the current call. Send to the controlling connection/dialer the
     * needed signals to immediately terminate the call. May throw an exception
     * if the connection is no longer active.
     * 
     * @throws VoipDialerException
     */
    void hangUp() throws VoipDialerException;
    
    /**
     * Return the IVoipConnection object that is the parent of this call.
     * The object is given in the constructor and is preserved. 
     * With this method, it will be possible to access connection methods 
     * when only a call object is at hand.
     */
    IVoipConnection getConnection();
    
}
