/*******************************************************************************
 * Copyright (c) 2005 Tellme Networks, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Tellme Networks, Inc. - Initial implementation
 *******************************************************************************/

package org.eclipse.vtp.internal.dialer;


/**
 * Interface describing the management of a dialer (software-phone).
 * 
 * This interface is the primary way for applications to interface with a
 * software-phone as being managed within the VTP. Through this interface,
 * the dialer is initiallized and connections made. This operates on a model
 * of separating the dialer itself from connections and calls
 * (@see org.eclipse.vtp.dialer.IVoipConnection and
 * @see org.eclipse.vtp.dialer.IVoipCall).
 * 
 * @author rjray
 */
public interface IVoipDialer {
    /// Stable dialer state, no errors
    int DIALER_OK         = 1;
    /// Dialer object is instantiated but not initialized
    int DIALER_NO_INIT    = DIALER_OK << 1;
    /// Dialer is in the process of trying to establish a connection
    int DIALER_CONNECTING = DIALER_NO_INIT << 1;
    /// Last operation on/through the dialer failed
    int DIALER_ERROR      = DIALER_CONNECTING << 1;

    /**
     * Initialize the concrete dialer instance. This sets up connection points,
     * starts applications when necessary (such as with Skype) and generally
     * prepares the VoIP environment. If a problem arises, a VoipDialerException
     * is thrown.
     * 
     * @throws VoipDialerException
     */
    void initialize() throws VoipDialerException;

    /**
     * Initialize the concrete dialer instance. This sets up connection points,
     * starts applications when necessary (such as with Skype) and generally
     * prepares the VoIP environment. This version takes a listener object
     * that implements the IVoipDialerListener interface as an argument, and
     * uses it to communicate messages from the underlying software-phone back
     * to the user. If a problems occurs, a VoipDialerException is thrown and
     * the error message is sent to the listener.
     * 
     * @param listener An object implementing the IVoipDialerListener interface
     * @throws VoipDialerException
     */
    void initialize(final IVoipDialerListener listener)
        throws VoipDialerException;

    /**
     * Connect the user to the IVoipDialer instance. In some cases, this may
     * be a no-op, but for services like Skype it is necessary to do
     * pre-dialing set-up. If a problem arises, a VoipDialerException is
     * thrown. If the dialer was initialized with a listener object for the
     * user, the error message is sent to the listener as well.
     * 
     * @return Object implementing the IVoipConnection interface
     * @throws VoipDialerException
     */
    IVoipConnection connect() throws VoipDialerException;

    /**
     * Retrieve the current status of the IVoipDialer instance.
     * 
     * @return The current status of the dialer as an integer
     */
    int status();
}
