/*******************************************************************************
 * Copyright (c) 2005 Tellme Networks, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Tellme Networks, Inc. - Initial implementation
 *******************************************************************************/

package org.eclipse.vtp.launching.internal.tellme;

import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Locale;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.eclipse.debug.core.DebugPlugin;

import org.eclipse.vtp.internal.dialer.IVoipCall;
import org.eclipse.vtp.internal.dialer.IVoipConnection;
import org.eclipse.vtp.internal.dialer.IVoipDialer;
import org.eclipse.vtp.internal.dialer.IVoipDialerFactory;
import org.eclipse.vtp.internal.dialer.VoipDialerException;

/**
 * @author mgreenawalt
 * 
 */
public class CallStudio {

	private static IVoipDialer dialer;

	final private static SimpleDateFormat time = new SimpleDateFormat(
			"mm:ss.SSS", Locale.getDefault());		//$NON-NLS-1$

	final private static boolean DEBUGGING = TellmePlugin.getDefault()
			.isDebugging();

	private CallStudio() {
		super();
		if (DEBUGGING) {
			System.out.println(time.format(new Date()) + ": CallStudio create");		//$NON-NLS-1$
			System.out.flush();
		}

	}

	protected static void initializeCall() throws CoreException,
			VoipDialerException {

		// look up the extending plugin for the VoipDialer
		if (DEBUGGING) {
			System.out.println(time.format(new Date())
					+ ": CallStudio.initializeCall() called.");		//$NON-NLS-1$
			System.out.flush();
		}

		boolean dialerFound = false;
		CallState.setStopping(false);
		final IExtensionPoint iep = Platform
				.getExtensionRegistry()
				.getExtensionPoint(
						"org.eclipse.vtp.launching.tellme.DialerImplementation");		//$NON-NLS-1$
		if (iep != null) {
			final IExtension[] extensions = iep.getExtensions();
			// choosing first phone extension
			// TODO may want to flag occurrence of multiple phone extensions

			final IConfigurationElement[] elements = extensions[0]
					.getConfigurationElements();
			if (elements != null) {
				for (int j = 0; j < elements.length; j++) {
					final Object obj = elements[j]
							.createExecutableExtension("class");		//$NON-NLS-1$
					if (obj instanceof IVoipDialerFactory) {
						dialer = ((IVoipDialerFactory) obj).getDialer();
						dialerFound = true;
						break;
					}
				}
			}
		}
		if (!dialerFound) {
			throw new VoipDialerException(null, Messages.CallStudio_0);
		}
	}

	public static IVoipCall startCall(final String studioNumber,
			final boolean noAutoLogin, final String studioID,
			final String studioPIN) throws VoipDialerException {
		if (DEBUGGING) {
			System.out.println(time.format(new Date())
					+ ": CallStudio.startCall() called.");		//$NON-NLS-1$
			System.out.flush();
		}

		int callstatus = 0;
		IVoipCall callObj = null;
//		dialer.initialize();
		final IVoipConnection accessObj = dialer.connect();

		if (DEBUGGING) {
			System.out.println("ConnectionStatus="		//$NON-NLS-1$
					+ connectStatus(accessObj.status()));
		}
		if (accessObj.status() != IVoipConnection.CONNECTION_OK) {
			throw new VoipDialerException(null, Messages.CallStudio_1);
		}
		callObj = accessObj.dial(studioNumber);

		callstatus = callObj.status();
		if (callstatus == IVoipCall.CALL_ACTIVE) {
			try {
				Thread.sleep(1500);
			} catch (InterruptedException e) {
				DebugPlugin.log(new Status(IStatus.ERROR,
						"org.eclipse.vtp.launching.tellme", IStatus.ERROR,		//$NON-NLS-1$
						Messages.CallStudio_2, e));
			}
			final String DTMFstring = "#" + studioID + "#" + studioPIN + "#";		//$NON-NLS-1$//$NON-NLS-2$//$NON-NLS-3$
			// System.out.println("DTMF string: " + DTMFstring);
			if (noAutoLogin) {
				callObj.sendDTMF(DTMFstring, 300);
			}
		}
		return callObj;
	}

	public static String finishCall(final IVoipCall callObj)
			throws VoipDialerException {
		String retval = "OK";		//$NON-NLS-1$
		if (DEBUGGING) {
			System.out.println(time.format(new Date())
					+ ": CallStudio.finishCall() called.");		//$NON-NLS-1$
			System.out.flush();
		}
		int callstatus = callObj.status();
		if (DEBUGGING) {
			System.out.println(time.format(new Date()) + ": CallProgress="		//$NON-NLS-1$
					+ callStatus(callstatus));
			System.out.flush();
		}
		do {
			try {
				Thread.sleep(2000);
			} catch (InterruptedException e) {
				DebugPlugin.log(new Status(IStatus.ERROR,
						"org.eclipse.vtp.launching.tellme", IStatus.ERROR,		//$NON-NLS-1$
						Messages.CallStudio_3, e));
			}
			callstatus = callObj.status();
		} while (callstatus != IVoipCall.CALL_COMPLETED
				&& callstatus != IVoipCall.CALL_ERROR && !CallState.isStopping());
		
		if (CallState.isStopping()){
			callstatus = IVoipCall.CALL_COMPLETED;
			retval = callStatus(callstatus);
		}
		
		if (callstatus == IVoipCall.CALL_ERROR) {
			retval = callStatus(callstatus);
		}

		return retval;
	}

	public static String connectStatus(final int stat) {
		String result;
		switch (stat) {
		case IVoipConnection.CONNECTION_DIALING:
			result = "CONNECTION_DIALING";		//$NON-NLS-1$
			break;
		case IVoipConnection.CONNECTION_ERROR:
			result = "CONNECTION_ERROR";		//$NON-NLS-1$
			break;
		case IVoipConnection.CONNECTION_OK:
			result = "CONNECTION_OK";		//$NON-NLS-1$
			break;
		default:
			result = "????";		//$NON-NLS-1$
			break;
		}
		return result;
	}

	public static String callStatus(final int stat) {
		String result;
		switch (stat) {
		case IVoipCall.CALL_ACTIVE:
			result = "CALL_ACTIVE";		//$NON-NLS-1$
			break;
		case IVoipCall.CALL_COMPLETED:
			result = "CALL_COMPLETED";		//$NON-NLS-1$
			break;
		case IVoipCall.CALL_ERROR:
			result = "CALL_ERROR";		//$NON-NLS-1$
			break;
		case IVoipCall.CALL_OK:
			result = "CALL_OK";		//$NON-NLS-1$
			break;
		default:
			if (stat == IVoipCall.CALL_ERROR << 1) {
				result = "AWAIT_RESPONSE";		//$NON-NLS-1$
			} else {
				result = Integer.toString(stat);
			}
		}
		return result;
	}
}
