/*******************************************************************************
 * Copyright (c) 2005 Tellme Networks, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Tellme Networks, Inc. - Initial implementation
 *******************************************************************************/

package org.eclipse.vtp.launching.internal.tellme;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.eclipse.vtp.launching.internal.tellme.LogToken;

public class LogItemReader {

	final private BufferedReader myReader;

	public LogItemReader(InputStream httpStream) {
		myReader = new BufferedReader(new InputStreamReader(httpStream));
	}

	// extractToken consumes the input stream, returning a LogToken
	// - any sequence of characters bounded by matching < >
	// (we assume that the <> are not nested)
	// - any sequence of charactes not bounded by <>
	// do not accumulate any line-ending characters

	private LogToken extractToken() {
		final StringBuffer accum = new StringBuffer();
		int ch = 0;

		// consume line-ending characters
		try {
			ch = myReader.read();
			while (ch == '\n' || ch == '\r')
				ch = myReader.read();
		} catch (IOException e) {
			return new LogToken(LogToken.IOE_TOKEN, "", false); //$NON-NLS-1$
		}
		// handle premature end of file
		if (ch < 0) {
			return new LogToken(LogToken.EOF_TOKEN, null, true);
		}
		// accumulate a good character
		accum.append((char) ch);
		// if not at a '<' then do else clause
		if (ch == '<' && ch >= 0) {
			return readHtmlToken(accum, ch);
		} else {
			if (ch < 0){
				return new LogToken(LogToken.EOF_TOKEN, null, false);
			}
			return readTextToken(accum);
		}

	}

	/**
	 * @param accum
	 * @return
	 */
	private LogToken readTextToken(final StringBuffer accum) {
		int ch;
		// note: the character in ch has already been put in the accum
		try {
			myReader.mark(5);
			ch = myReader.read();
			while (ch != '<' && ch >= 0) {
				if (ch != '\n' && ch != '\r') {
					accum.append((char) ch);
				}
				myReader.mark(5);
				ch = myReader.read();
			}
			// if we read the '>' then we need to reposition to read it
			// again
			if (ch >= 0) {
				myReader.reset();
			} else {
				return new LogToken(LogToken.EOF_TOKEN, accum.toString(),
					false);
			}
		} catch (IOException e) {
			return new LogToken(LogToken.IOE_TOKEN, accum.toString(), false);
		}
		return new LogToken(LogToken.TEXT_TOKEN, accum.toString(), true);
	}

	/**
	 * @param accum
	 * @param ch
	 * @return
	 */
	private LogToken readHtmlToken(final StringBuffer accum, final int chr) {
		int ch = chr;
		try {
			ch = myReader.read();
			while (ch >= 0) {
				if (ch != '\n' && ch != '\r') {
					accum.append((char) ch);
				}
				if (ch == '>') {
					break;
				}
				ch = myReader.read();
			}
		} catch (IOException e) {
			return new LogToken(LogToken.IOE_TOKEN, accum.toString(), false);
		}
		// to here either if ch is '>' or EOF reached
		if (ch < 0) {
			return new LogToken(LogToken.EOF_TOKEN, accum.append('>')
				.toString(), false);
		}
		return new LogToken(LogToken.HTML_TOKEN, accum.toString(), true);
	}

	final private static Pattern logLine = Pattern
		.compile("<div class=\"log_\\w+?\">\\s*\\[(.*?)\\]\\s(.*?)</div>"); //$NON-NLS-1$

	public LogItem nextLogItem() {
		String type = null;
		final StringBuffer accum = new StringBuffer();
		LogToken temp = null;
		while (!(temp = extractToken()).isDiv()
			&& temp.getType() != LogToken.EOF_TOKEN
			&& temp.getType() != LogToken.IOE_TOKEN)
			;
		if (temp.getType() == LogToken.EOF_TOKEN
			|| temp.getType() == LogToken.IOE_TOKEN) {
//			System.out.println("Starting log: "+(temp.getType()==LogToken.EOF_TOKEN?"EOF":"IOE")+"\tBody: "+temp.getContent());
//			System.out.flush();
			return null;
		}
		accum.append(temp.getContent()); // accumulate the start <div
		type = temp.getLogType();

		while (!(temp = extractToken()).isEndDiv()
			&& temp.getType() != LogToken.EOF_TOKEN
			&& temp.getType() != LogToken.IOE_TOKEN) {
			accum.append(temp.getContent());
		}
		accum.append(temp.getContent());

		if (temp.getType() == LogToken.EOF_TOKEN
			|| temp.getType() == LogToken.IOE_TOKEN) {
//			System.out.println("Reading log: "+(temp.getType()==LogToken.EOF_TOKEN?"EOF":"IOE")+"\tBody: "+temp.getContent());
//			System.out.flush();
			return null;
		}
		// at this point we have the entire <div> section in the accum buffer
		// extract the timestamp and body of the message
		Matcher matchLog = logLine.matcher(accum.toString());
		if (matchLog.matches()) {
			String logbody = massageLogBody(matchLog.group(2));
			return new LogItem(type, matchLog.group(1), logbody);
		}
		return new LogItem("error", null, accum.toString()); //$NON-NLS-1$
	}

	final Pattern htmlItem = Pattern.compile("<.*?>"); //$NON-NLS-1$
	final Pattern xmlChar = Pattern
		.compile("(&quot;)|(&apos;)|(&amp;)|(&gt;)|(&lt;)"); //$NON-NLS-1$
	final Pattern remTrash = Pattern.compile("\\[show detail *?\\]"); //$NON-NLS-1$

	// method massageLogBody does two things:
	// 1. Removes any HTML entities in <> pairs inside the string
	// 2. Converts XML encodings back to chars
	private String massageLogBody(final String body) {
		final Matcher htmlMatcher = htmlItem.matcher(body);
		final StringBuffer acc = new StringBuffer();
		String interim = htmlMatcher.replaceAll(""); //$NON-NLS-1$
		final Matcher trashMatcher = remTrash.matcher(interim);
		interim = trashMatcher.replaceAll(""); //$NON-NLS-1$
		final Matcher xmlMatcher = xmlChar.matcher(interim);
		while (xmlMatcher.find()) {
			xmlMatcher.appendReplacement(acc,
				xmlMatcher.group(1) != null ? "\"" //$NON-NLS-1$
					: (xmlMatcher.group(2) != null ? "'" //$NON-NLS-1$
						: (xmlMatcher.group(3) != null ? "&" : (xmlMatcher //$NON-NLS-1$
							.group(4) != null ? ">" : "<")))); //$NON-NLS-1$ //$NON-NLS-2$
		}
		xmlMatcher.appendTail(acc);
		return acc.toString();

	}
}
