package org.eclipse.vtp.catalog.internal.ui;
/*******************************************************************************
 * Copyright (c) 2006 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *     
 *******************************************************************************/
import java.util.TreeSet;
import java.util.Vector;

import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.MouseAdapter;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.TableItem;
import org.eclipse.wst.xml.core.internal.XMLCorePlugin;
import org.eclipse.wst.xml.core.internal.catalog.provisional.ICatalog;
import org.eclipse.wst.xml.core.internal.catalog.provisional.ICatalogEntry;
import org.eclipse.wst.xml.core.internal.catalog.provisional.INextCatalog;

/**
 * Dialog to choose XML catalog entry.
 * 
 * @author Brent D. Metz
 */
public class CatalogEntrySelectionDialog extends Dialog {
	protected String title;
	protected String previousValue;
	protected String selectedCatalogEntry;
	protected Table catalogTable;
	
	/**
	 * Creates a dialog for choosing a catalog entry.
	 * 
	 * @param parentShell The shell to use as a base for this modal dialog.
	 * @param title The title to place on the dialog.
	 * @param previous_value The catalog entry used previously before this dialog was opened.
	 */
	protected CatalogEntrySelectionDialog(Shell parentShell, String title, String previousValue) {
		super(parentShell);
		setShellStyle(SWT.DIALOG_TRIM|SWT.RESIZE);
		this.title=title;
		this.previousValue=previousValue;
		this.selectedCatalogEntry=previousValue;
	}
	
	/**
	 * Returns the keys in the XML Catalog suitable for populating the preference page table.
	 * 
	 * @return A Vector of Strings containing the XML Catalog keys.
	 */
	protected Vector getKeys() {
		TreeSet ss = new TreeSet();
		
		getShell().setText(title);
		
		ICatalog xmlCatalog = XMLCorePlugin.getDefault().getDefaultXMLCatalog();
		if (xmlCatalog != null) {
			INextCatalog[] nextCatalogs = xmlCatalog.getNextCatalogs();
			if (nextCatalogs == null) {
				return new Vector();
			}
			for (int i = 0 ; i < nextCatalogs.length ; i++) {
				INextCatalog catalog = nextCatalogs[i];
	            ICatalog referencedCatalog = catalog.getReferencedCatalog();
	            if (referencedCatalog != null) {
	            	ICatalog foundCatalog = null;
	                if (XMLCorePlugin.SYSTEM_CATALOG_ID.equals(referencedCatalog.getId())) {
	                    foundCatalog = referencedCatalog;
	                } else if (XMLCorePlugin.USER_CATALOG_ID.equals(referencedCatalog.getId())) {
	                	foundCatalog = referencedCatalog;
	                }
	                
	                ICatalogEntry entries[] = foundCatalog.getCatalogEntries();
	                if (entries != null) {
	                	for (int j = 0 ; j < entries.length ; j++) {
	                		if (entries[j] != null && entries[j].getEntryType() == ICatalogEntry.ENTRY_TYPE_PUBLIC) {
	                			String key = entries[j].getKey();
	                			if (!ss.contains(key)) {
	                				ss.add(key);
	                			}
	                		}
	                	}
	                }
	            }
			}
		}
		
		return new Vector(ss);
	}

	protected Control createDialogArea(Composite parent) {
	    Composite composite = new Composite(parent, SWT.NULL);
	    composite.setLayout(new GridLayout());
	    composite.setLayoutData(new GridData(GridData.FILL_BOTH));
		
		catalogTable = new Table(composite, SWT.BORDER|SWT.V_SCROLL);
		catalogTable.setLinesVisible(true);
		TableColumn tc = new TableColumn(catalogTable, SWT.NONE);
		tc.setWidth(300);
		
		GridData gd = new GridData(GridData.FILL_BOTH);
		gd.heightHint=200;
		catalogTable.setLayoutData(gd);
		
		Vector keys = getKeys();
		for (int i = 0 ; i < keys.size() ; i++) {
			TableItem ti = new TableItem(catalogTable, SWT.NONE);
			ti.setText(0, (String)keys.get(i));
			if (previousValue.equals(keys.get(i))) {
				catalogTable.setSelection(i);
			}
		}
		catalogTable.addMouseListener(new MouseAdapter() {
			public void mouseDoubleClick(MouseEvent e) {
				int i = catalogTable.getSelectionIndex();
				if (i == -1) {
					return;
				}
				selectedCatalogEntry=catalogTable.getItem(i).getText();
				okPressed();
			}
		});
		catalogTable.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				int i = catalogTable.getSelectionIndex();
				if (i == -1) {
					return;
				}
				selectedCatalogEntry=catalogTable.getItem(i).getText();
			}
		});
		
		return composite;
	}

	public String getSelectedCatalogEntry() {
		return selectedCatalogEntry;
	}
}
