package org.eclipse.vtp.catalog.registry;
/*******************************************************************************
 * Copyright (c) 2005,2006 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *     
 *******************************************************************************/

import java.util.StringTokenizer;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.Platform;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.vtp.catalog.internal.CatalogPlugin;
import org.eclipse.wst.xml.core.internal.XMLCorePlugin;
import org.eclipse.wst.xml.core.internal.catalog.provisional.ICatalog;
import org.eclipse.wst.xml.core.internal.catalog.provisional.ICatalogEntry;
import org.eclipse.wst.xml.core.internal.catalog.provisional.INextCatalog;


/**
 * Identifies the default XML catalog entry for a given content type.
 * 
 * @author Brent D. Metz
 */
public class CatalogDefaultResolver {
	public static final String CCXML_ID = "org.eclipse.vtp.editor.ccxml.core.ccxmlsource"; //$NON-NLS-1$
	public static final String SRGS_ID = "org.eclipse.vtp.editor.srgxml.core.srgxmlsource"; //$NON-NLS-1$
	public static final String VXML_ID = "org.eclipse.vtp.editor.vxml.core.vxmlsource"; //$NON-NLS-1$
	public static final String VERSION = "_version"; //$NON-NLS-1$
	
	/**
	 * Given a public identifier such as "-//IETF//DTD HTML 2.0 Strict//EN" returns the version contained
	 *   within. ("2.0") NOTE: There is no standardized way to identify the version so this method only makes
	 *   a best attempt.
	 *   
	 * @param identifier The public identifier to scan.
	 * @return The version contained in the public identifier or null if none was identified.
	 */
	public static String getVersionFromPublicIdentifier(String identifier) {
		int begin=-1;
		int end=-1;
		
		if (identifier == null || identifier.length() < 7) {
			return null;
		}
		
		begin = identifier.indexOf("//"); //$NON-NLS-1$
		if (begin==-1) {
			return null;
		}
		begin = identifier.indexOf("//", begin+1); //$NON-NLS-1$
		if (begin == -1) {
			return null;
		}
		
		end = identifier.indexOf("//", begin+1); //$NON-NLS-1$
		if (end == -1) {
			return null;
		}
		
		String display = identifier.substring(begin+"//".length(), end); //$NON-NLS-1$
		StringTokenizer displayTokens = new StringTokenizer(display);
		while (displayTokens.hasMoreTokens()) {
			String token = displayTokens.nextToken();
			try {
				new Double(token);
				return token;
			} catch (Exception e) {
				
			}
		}
		
		
		return null;
	}
	
	/**
	 * Given a content ID loads the default public key to use from the preference store.
	 * 
	 * @param contentTypeId The content ID to look up (ie org.eclipse.vtp.editor.ccxml.core.ccxmlsource)
	 * @param The public key of the XML Catalog Entry to use or an empty string if none found.
	 */
	protected static String loadPreferenceStoreDefault(String contentTypeId) {
		IPreferenceStore store = CatalogPlugin.getDefault().getPreferenceStore();
		if (contentTypeId == null || store == null) {
			return new String();
		}
		return store.getString(contentTypeId);
	}
	
	/**
	 * Given a content ID loads the version of the catalog to use from the preference store.
	 * 
	 * @param contentTypeId The content ID to look up (ie org.eclipse.vtp.editor.ccxml.core.ccxmlsource)
	 * @param The version of the XML Catalog Entry to use or an empty string if none found.
	 */
	protected static String loadPreferenceStoreDefaultVersion(String contentTypeId) {
		IPreferenceStore store = CatalogPlugin.getDefault().getPreferenceStore();
		if (store == null) {
			return new String();
		}
		return store.getString(contentTypeId + VERSION);
	}
	
	
	/**
	 * Given a content ID loads the default public identifier to use from the VTP catalog extension point.
	 * 
	 * @param contentTypeId The content ID to look up (ie org.eclipse.vtp.editor.ccxml.core.ccxmlsource)
	 * @param The public key of the XML Catalog Entry to use or an empty string if none found.
	 */
	public static String loadExtensionPointDefaultEntry(String id) {
		IExtensionPoint iep = Platform.getExtensionRegistry().getExtensionPoint("org.eclipse.vtp.catalog.language"); //$NON-NLS-1$
		if (iep == null) {
			return new String();
		}
		IExtension[] extensions = iep.getExtensions();

		for (int i = 0; i < extensions.length; i++) {
			IConfigurationElement[] elements = extensions[i].getConfigurationElements();
			for (int j = 0; j < elements.length; j++) {
				String contentTypeId = elements[j].getAttribute("contentTypeId"); //$NON-NLS-1$
				String publicKey = elements[j].getAttribute("publicKey"); //$NON-NLS-1$
					
				if (contentTypeId.length() > 0 && contentTypeId.equals(id) && publicKey.length() > 0 ) {
					return publicKey;
				}
			}
		}
		return new String();
	}
	
	/**
	 * Given a content ID loads the default version of to use from the VTP catalog extension point.
	 * 
	 * @param contentTypeId The content ID to look up (ie org.eclipse.vtp.editor.ccxml.core.ccxmlsource)
	 * @param The public key of the XML Catalog Entry to use or an empty string if none found.
	 */
	public static String loadExtensionPointDefaultVersion(String id) {
		IExtensionPoint iep = Platform.getExtensionRegistry().getExtensionPoint("org.eclipse.vtp.catalog.language"); //$NON-NLS-1$
		if (iep == null) {
			return new String();
		}
		IExtension[] extensions = iep.getExtensions();

		for (int i = 0; i < extensions.length; i++) {
			IConfigurationElement[] elements = extensions[i].getConfigurationElements();
			for (int j = 0; j < elements.length; j++) {
				String contentTypeId = elements[j].getAttribute("contentTypeId"); //$NON-NLS-1$
				String version = elements[j].getAttribute("version"); //$NON-NLS-1$
					
				if (contentTypeId.length() > 0 && contentTypeId.equals(id) && version != null && version.length() > 0 ) {
					return version;
				}
			}
		}
		return new String();
	}
	
	
	/**
	 * Given a content type id returns an object with information describing the proper entry in the
	 * 	XML Catalog to use.
	 * 
	 * @param id A string representing the type of content.
	 * @return An object representing the default entry in the XML Catalog to use for this content type id or null if one cannot be identified.
	 */
	public static final CatalogEntry getDefaultEntryForContentType(String id) {
		String publicKey;
		
		publicKey = loadPreferenceStoreDefault(id);
		if (publicKey.length() == 0) {
			publicKey = loadExtensionPointDefaultEntry(id);
		}
		if (publicKey.length() == 0) {
			return null;
		}
		
		CatalogEntry result = new CatalogEntry();
		result.setKey(publicKey);
		
		String ver = loadPreferenceStoreDefaultVersion(id);
		if (ver.length() > 0) {
			result.setVersion(ver);
		} else {
			result.setVersion(getVersionFromPublicIdentifier(publicKey));
		}
		
		ICatalog xmlCatalog = XMLCorePlugin.getDefault().getDefaultXMLCatalog();
		if (xmlCatalog != null) {
			INextCatalog[] nextCatalogs = xmlCatalog.getNextCatalogs();
			if (nextCatalogs == null) {
				return null;
			}
			for (int i = 0 ; i < nextCatalogs.length ; i++) {
				INextCatalog catalog = nextCatalogs[i];
	            ICatalog referencedCatalog = catalog.getReferencedCatalog();
	            if (referencedCatalog != null) {
	            	ICatalog foundCatalog = null;
	                if (XMLCorePlugin.SYSTEM_CATALOG_ID.equals(referencedCatalog.getId())) {
	                    foundCatalog = referencedCatalog;
	                } else if (XMLCorePlugin.USER_CATALOG_ID.equals(referencedCatalog.getId())) {
	                	foundCatalog = referencedCatalog;
	                }
	                
	                ICatalogEntry entries[] = foundCatalog.getCatalogEntries();
	                if (entries != null) {
	                	for (int j = 0 ; j < entries.length ; j++) {
	                		if (entries[j] != null && entries[j].getKey().equals(publicKey)) {
	                			result.setURI(entries[j].getURI());
	                			String webURL = entries[j].getAttributeValue("webURL"); //$NON-NLS-1$
	                			if (webURL != null && webURL.length() > 0) {
	                				result.setWebURL(webURL);
	                			}
	                		}
	                	}
	                }
	            }
			}
		}
		
		return result;
	}
}
