/********************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *     
 *******************************************************************************/
package org.eclipse.vtp.editor.internal.wizards;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.Locale;

import org.eclipse.vtp.catalog.registry.CatalogEntry;
import org.eclipse.vtp.editor.internal.nls.VoicetoolsResourceHandler;


/**
 * Default generic content generator for the various new file wizards
 * Configured from plugin.xml manifest's &lt;newfile&gt; element, and
 * optional subclasses can be defined there in the generatorClass attribute
 * 
 * @author Frank Sauer
 */
public class NewFileContentGenerator {

	/**
	 * XML indent (3 spaces)
	 */
	protected static final String INDENT = "   "; //$NON-NLS-1$
	
	private CatalogEntry dtd;

	/**
	 * constructor
	 * @param dtd the vocabulary to use
	 */
	public NewFileContentGenerator(CatalogEntry dtd) {
		this.dtd = dtd;
	}
	
	/**
	 * @return the current vocabulary
	 */
	public CatalogEntry getDTD() {
		return dtd;
	}
	
	/**
	 * @return content for the JSP page contentType directive
	 */
	protected String getMimeType() {
		return "application/voicexml+xml"; //$NON-NLS-1$
	}
	
	/**
	 * prints the INDENT string to p
	 * @param p output goes here
	 */
	protected void indent(PrintWriter p) {
		p.print(INDENT);
	}
	
	/**
	 * Main generator method. Write the entire template from the
	 * root element down. The name of the root element is obtained 
	 * from the dtd, and a version attribute and namespace attribute 
	 * are added. Based on configuration, an optional meta tag is added 
	 * as a child, and finally an XML comment is inserted.
	 * @param p output goes here.
	 * @param writeJSPPageDirective use true to output a page directive
	 */
	public void writeRootElement(PrintWriter p, boolean writeJSPPageDirective, String encodingString, String rootElementName) {
		p.print("<"); //$NON-NLS-1$
		p.print(rootElementName);	
		writeVersionAttribute(p);
		if (rootElementName == "vxml") { //$NON-NLS-1$
			writeNamespaceAttribute(p);
			writeLanguageAttribute(p);
		}	
		addAdditionalAttributes1(p);
		p.println(">"); //$NON-NLS-1$
		if (writeJSPPageDirective) {
			writeJSPPageDirective(p, encodingString);
		}
		if (rootElementName == "vxml") {		 //$NON-NLS-1$
			writeMetaTag(p);
		}	
		writeComment(p);
		addAdditionalAttributes2(p);
		writeClosingTag(p, rootElementName);
	}

	/**
	 * writes xmlns="[dtd.getNamespace()]" to p
	 * @param p	output goes here
	 */
	protected void writeNamespaceAttribute(PrintWriter p) {
		String namespace = "http://www/w3/org/2001/vxml"; //$NON-NLS-1$
//	CN TODO	namespace = dtd.getNamespace();
		if (namespace != null && namespace.length() > 0) {
			p.print(" xmlns=\""); //$NON-NLS-1$
			p.print(namespace);
			p.print("\""); //$NON-NLS-1$
		}
		else {
			p.print(" xmlns=\""); //$NON-NLS-1$
			p.print("\""); //$NON-NLS-1$
		}
	}

	/**
	 * writes version="[getVersion().getName()]" to p
	 * @param p	output goes here
	 */
	protected void writeVersionAttribute(PrintWriter p) {
		p.print(" version=\""); //$NON-NLS-1$
		p.print(dtd.getVersion());
		p.print("\""); //$NON-NLS-1$
	}

	/**
	 * writes xml:lang="[getXmlLang()]" to p
	 * @param p	output goes here
	 */
    protected void writeLanguageAttribute(PrintWriter p) {
        String lang = getXmlLang();
        String country = getXmlCountry();
        if (lang != null && lang.length() > 0) {
               p.print(" xml:lang=\""); //$NON-NLS-1$
               p.print(lang);                     
               p.print("-");  //$NON-NLS-1$
               p.print(country);
               p.print("\""); //$NON-NLS-1$


		}
	}

	/**
	 * If the current DTD has a default language specified,
	 * this method returns language from the currently set locale
	 * otherwise it returns null (some XML vocabularies like CCXML
	 * do not support the xml:lang attribute.
	 * @return 2 letter code for the locale's language or null
	 */
	protected String getXmlLang() {		
		return System.getProperty("user.language"); //$NON-NLS-1$
	}
	
    protected String getXmlCountry() {              
		return System.getProperty("user.country"); //$NON-NLS-1$
 }

	/**
	 * This default implementation returns the 
	 * default locale set by the VM
	 * @return the default Locale
	 */
	protected Locale getLocale() {
		return Locale.getDefault();
	}
	
	/**
	 * Do nothing here, but allow subclasses to add new
	 * attributes to the generated root element
	 * @param p output goes here
	 */
	protected void addAdditionalAttributes1(PrintWriter p) {
	}
	
	
	/**
	 * Do nothing here, but allow subclasses to add new
	 * attributes to the generated root element
	 * @param p output goes here
	 */
	protected void addAdditionalAttributes2(PrintWriter p) {
	}
	
	/**
	 * output the DOCTYPE line to p
	 * @param p output goes here
	 */
	public void writeDocType(PrintWriter p, String rootElementName, String contentType) {
		p.print("<!DOCTYPE "); //$NON-NLS-1$
		p.print(rootElementName);
		p.print(" "); //$NON-NLS-1$
		String publicKey = dtd.getKey(); 
		p.print("PUBLIC \"");  //$NON-NLS-1$
		p.print(publicKey); 
		p.print("\" \"");  //$NON-NLS-1$
		String webUrl = dtd.getWebURL();
		if (webUrl != null && webUrl.length() > 0) {
			p.print(webUrl);
		} else {
			String uri = dtd.getURI();
			String shortName = uri.substring(uri.lastIndexOf('/') + 1);
			p.print(shortName);
		}
		p.println("\">"); //$NON-NLS-1$
		p.println();
	}
	
	/**
	 * print the closing tag for the root node
	 * @param p output goes here
	 */
	protected void writeClosingTag(PrintWriter p, String rootElementName) {
		p.print("</"); //$NON-NLS-1$
		p.print(rootElementName);
		p.println(">"); //$NON-NLS-1$
	}

	/**
	 * Write the 'Place Content Here' comment
	 * (obtained from resource bundle)
	 * @param p output goes here
	 */
	public void writeComment(PrintWriter p) {
		indent(p);
		p.println(VoicetoolsResourceHandler.getString("NewFileWizards.UserMessage")); //$NON-NLS-1$
		p.println();
		p.println();
	}

	/**
	 * Checks configuration and if a meta tag was enabled 
	 * (both its name and content were specified), print it
	 * @param p output goes here
	 */
	public void writeMetaTag(PrintWriter p) {
		String name = "Generator";  //$NON-NLS-1$
		String content = "Eclipse";  //$NON-NLS-1$
		if (name != null && content != null && name.length() > 0 && content.length() > 0) {
			indent(p);
			p.print("<meta name=\""); //$NON-NLS-1$
			p.print(name);
			p.print("\" content=\""); //$NON-NLS-1$
			p.print(content);
			p.println("\"/>"); //$NON-NLS-1$
		}
	}

	/**
	 * Prints a JSP page directive with mime contentType, encoding and charset
	 * @param p output goes here
	 */
	public void writeJSPPageDirective(PrintWriter p, String encodingString) {
		p.print("<%@ page contentType=\" "); //$NON-NLS-1$
		p.print(getMimeType());
		p.print("; charset="); //$NON-NLS-1$
		p.print(encodingString);
		p.print("\" pageEncoding=\""); //$NON-NLS-1$
		p.print(encodingString);
		p.println("\" %>"); //$NON-NLS-1$
	}

	/**
	 * write the first line (XML 'processing instruction') with hardcoded
	 * version 1.0 and an encoding obtained from calling getXMLEncoding()
	 * @param p	output is written here
	 */
	public void writeXMLHeader(PrintWriter p) {
		p.print("<?xml version=\"1.0\" encoding =\""); //$NON-NLS-1$
		p.print(getXMLEncoding());
		p.println("\"?>"); //$NON-NLS-1$
	}

	/**
	 * @return XML encoding as set by preferences
	 */
	public String getXMLEncoding() {
			return "UTF-8";	 //$NON-NLS-1$
	}
	
	/**
	 * change the current vocabulary
	 * @param dtd new vocabulary
	 */
	public void setDTD(CatalogEntry dtd) {
		this.dtd = dtd;
	}
	
	/**
	 * @return String with entire new file content without JSP pagedirective
	 */
//	public String getNewFileContent() {
//		return getNewFileContent(false);
//	}
	
	/**
	 * @param writePageDirective use true to include a JSP page directive
	 * @return String with entire new file content with JSP pagedirective if passed true
	 */
	public String getNewFileContent(boolean writePageDirective, String rootElementName, String contentType) {
		StringWriter w = new StringWriter();
		PrintWriter p = new PrintWriter(w);
		writeXMLHeader(p);
		writeDocType(p, rootElementName, contentType);
		writeRootElement(p, writePageDirective, getXMLEncoding(), rootElementName);
		p.flush();
		return w.toString();
	}
}
