/*******************************************************************************
 * Copyright (c) 2005 Tellme Networks, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Tellme Networks, Inc. - Initial implementation
 *******************************************************************************/

package org.eclipse.vtp.launching.internal.tellme;

import java.net.URI;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Locale;
import java.util.StringTokenizer;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.debug.core.DebugEvent;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.debug.core.ILaunch;
import org.eclipse.debug.core.model.IProcess;
import org.eclipse.vtp.launching.IVoiceXMLBrowser;
import org.eclipse.vtp.launching.IVoiceXMLBrowserConstants;
import org.eclipse.vtp.launching.VoiceXMLBrowserInput;
import org.eclipse.vtp.launching.VoiceXMLBrowserProcess;
import org.eclipse.vtp.launching.VoiceXMLLogMessage;

/**
 * @author mgreenawalt
 * 
 */

public class GetLogItems implements Runnable {

	final private static SimpleDateFormat logDF = new SimpleDateFormat(
			"MM/dd/yy HH:mm:ss", Locale.getDefault()); //$NON-NLS-1$

	final private static SimpleDateFormat time = new SimpleDateFormat(
			"mm:ss.SSS", Locale.getDefault()); //$NON-NLS-1$

	private final static boolean DEBUGGING = TellmePlugin.getDefault()
			.isDebugging();

	public void run() {
		CallState.initLogFetch();
		IVoiceXMLBrowser activeBrowser = null;
		final ILaunch[] launches = DebugPlugin.getDefault().getLaunchManager()
				.getLaunches();
		if (launches == null) {
			CallState
					.reportLogFetchFail(Messages.GetLogItems_0);
			synchronized (CallState.getSynch()) {
				CallState.setLogFetchDone();
				if (DEBUGGING) {
					System.out.println(time.format(new Date())
							+ "calling launchDone from logFetch on failure"); //$NON-NLS-1$
					System.out.flush();
				}
				if (CallState.launchDone()) {
					CallState.endLaunch();
				}
			}
			return;
		} else {
			for (int i = 0; i < launches.length; i++) {
				IProcess[] t = launches[i].getProcesses();
				if (t != null && t.length > 0
						&& t[0] instanceof VoiceXMLBrowserProcess) {
					activeBrowser = ((VoiceXMLBrowserProcess) t[0])
							.getVoiceXMLBrowser();
				}
			}
		}

		/**
		 * if the call to Studio was successful, pull the debug log over
		 * 
		 * first, get the applicable log info
		 */
		CallLog theLog = null;

		if (DEBUGGING) {
			System.out.println(time.format(new Date())
					+ ": GetLogItems started"); //$NON-NLS-1$
			System.out.flush();
		}
		theLog = CallState.getLog();
		if (theLog == null) {
			CallState.reportLogFetchFail(Messages.GetLogItems_1);
			synchronized (CallState.getSynch()) {
				CallState.setLogFetchDone();
				if (DEBUGGING) {
					System.out
							.println(time.format(new Date())
									+ "calling launchDone from logFetch on failure (no log)"); //$NON-NLS-1$
					System.out.flush();
				}
				if (CallState.launchDone()) {
					CallState.endLaunch();
				}
			}
			return;
		} else {
			try {
				final LogItemReader theLogReader = StudioInteractions
						.getLogItemReader(theLog);
				LogItem theItem = theLogReader.nextLogItem();
				if (theItem == null) {
					DebugEvent event[] = new DebugEvent[1];
					event[0] = new DebugEvent(activeBrowser,
							DebugEvent.MODEL_SPECIFIC,
							IVoiceXMLBrowserConstants.EVENT_LOG_MESSAGE);
					event[0].setData(new VoiceXMLLogMessage(new Date(),
							Messages.GetLogItems_2));
					DebugPlugin.getDefault().fireDebugEventSet(event);
					CallState
							.reportLogFetchFail(Messages.GetLogItems_3);
					synchronized (CallState.getSynch()) {
						CallState.setLogFetchDone();
						if (DEBUGGING) {
							System.out
									.println(time.format(new Date())
											+ "calling launchDone from logFetch on failure (no log events)"); //$NON-NLS-1$
							System.out.flush();
						}
						if (CallState.launchDone()) {
							CallState.endLaunch();
						}
					}
					return;
				}
				Date logDate = null;
				while (theItem != null && !CallState.isStopping()) {
					if (DEBUGGING) {
						System.out.println(time.format(new Date())
								+ ": log item: \t" + theItem.getKind() + "\t" //$NON-NLS-1$//$NON-NLS-2$
								+ theItem.getTime() + "\t" + theItem.getBody()); //$NON-NLS-1$
						System.out.flush();
					}
					// convert string timestamp to Date object
					try {
						logDate = logDF.parse(theItem.getTime());
					} catch (ParseException e) {
						DebugPlugin.log(new Status(IStatus.ERROR,
								"org.eclipse.vtp.launching.tellme", //$NON-NLS-1$
								IStatus.ERROR,
								Messages.GetLogItems_4, e));
					}


					String body = theItem.getBody();
					if (body != null 
							&& body.trim().length() != 0 
							&& body.indexOf("https://studio.tellme.com/telephone/cgi-bin/runapp.pl") == -1 //$NON-NLS-1$
							&& body.indexOf("https://studio.tellme.com/telephone/cgi-bin/runscratch.pl") == -1 //$NON-NLS-1$
							&& body.indexOf("http://redirect.vui.en-us.tellme.com/index.vxml") == -1) { //$NON-NLS-1$
						DebugEvent event[] = new DebugEvent[1];
						event[0] = new DebugEvent(activeBrowser, DebugEvent.MODEL_SPECIFIC, IVoiceXMLBrowserConstants.EVENT_LOG_MESSAGE);
						event[0].setData(new VoiceXMLLogMessage(logDate, theItem.getBody()));
						DebugPlugin.getDefault().fireDebugEventSet(event);
						
						// FILE http://www.abc.com/abc.vxml LOADED [ Load time 150ms; HTTP code 200 ...
						if (body.startsWith("FILE") && body.indexOf(" LOADED [") != -1) { //$NON-NLS-1$ //$NON-NLS-2$
							StringTokenizer st = new StringTokenizer(body);
							if (st.hasMoreTokens()) {
								st.nextToken();
							}
							if (st.hasMoreTokens()) {
								try {
									String strURI = st.nextToken();
									DebugEvent resourceLoadEvent[] = new DebugEvent[1];
									resourceLoadEvent[0] = new DebugEvent(activeBrowser, DebugEvent.MODEL_SPECIFIC, IVoiceXMLBrowserConstants.EVENT_RESOURCE_LOADED);
									resourceLoadEvent[0].setData(new URI(strURI));
									DebugPlugin.getDefault().fireDebugEventSet(resourceLoadEvent);
								} catch (Exception e) {
									DebugPlugin.log(new Status(IStatus.ERROR,
											"org.eclipse.vtp.launching.tellme", //$NON-NLS-1$
											IStatus.ERROR,
											Messages.GetLogItems_11, e));
								}
							}
						}
						
						if (body.startsWith("RECOGNIZED")) { //$NON-NLS-1$
							try {
								Pattern p = Pattern.compile( ".*result = \\[([^\\]]*)\\].* confidence = ([0-9\\.]+)\\%"); //$NON-NLS-1$
								Matcher m = p.matcher(body);
								if (m.matches() && m.groupCount() == 2) {
									String result = m.group(1);
									double confidence = Double.parseDouble(m.group(2));
									VoiceXMLBrowserInput browserInput = new VoiceXMLBrowserInput();
									browserInput.setConfidence(confidence);
									if (confidence == 100) {
										browserInput.setInputType(VoiceXMLBrowserInput.TYPE_DTMF);
									} else {
										browserInput.setInputType(VoiceXMLBrowserInput.TYPE_VOICE);
									}
									browserInput.setInput(result);
									DebugEvent resourceLoadEvent[] = new DebugEvent[1];
									resourceLoadEvent[0] = new DebugEvent(activeBrowser, DebugEvent.MODEL_SPECIFIC, IVoiceXMLBrowserConstants.EVENT_INPUT_RECEIVED);
									resourceLoadEvent[0].setData(browserInput);
									DebugPlugin.getDefault().fireDebugEventSet(resourceLoadEvent);
								}
							} catch (Exception e) {
								
							}
						}
					}
					if ("SESSION_DONE".equals(theItem.getBody())) { //$NON-NLS-1$
						break;
					} else {
						theItem = theLogReader.nextLogItem();
					}
				}
			} catch (TellmeBrowserException e) {
				DebugPlugin.log(new Status(IStatus.ERROR,
						"org.eclipse.vtp.launching.tellme", IStatus.ERROR, //$NON-NLS-1$
						Messages.GetLogItems_13, e));
			}

			if (DEBUGGING) {
				System.out.println(time.format(new Date())
						+ ": no more log items"); //$NON-NLS-1$
				System.out.flush();
			}

			// DisplayActions.showDebugLogURL(theLog);

		}

		if (CallState.isStopping()){
			CallState.reportLogFetchFail(Messages.GetLogItems_14);
		} else if (theLog == null) {
			CallState.reportLogFetchFail(Messages.GetLogItems_15);
			// DisplayActions.showError("Failed to connect");
		} else  {
			CallState.setCallID(theLog.getCallID());
			CallState.reportLogFetchSucceed(""); //$NON-NLS-1$
			// DisplayActions.showEndOfCall(theLog.getCallID());
		}
		synchronized (CallState.getSynch()) {
			CallState.setLogFetchDone();
			if (DEBUGGING) {
				System.out.println(time.format(new Date())
						+ ": calling launchDone from logFetch at end"); //$NON-NLS-1$
				System.out.flush();
			}
			if (CallState.launchDone()) {
				CallState.endLaunch();
			}
		}
	}
}
