/*******************************************************************************
 * Copyright (c) 2005 Tellme Networks, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Tellme Networks, Inc. - Initial implementation
 *******************************************************************************/
package org.eclipse.vtp.launching.internal.tellme;

import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Preferences;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.debug.core.DebugEvent;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.vtp.internal.dialer.IVoipCall;
import org.eclipse.vtp.internal.dialer.VoipDialerException;
import org.eclipse.vtp.launching.IVoiceXMLBrowser;
import org.eclipse.vtp.launching.IVoiceXMLBrowserConstants;
import org.eclipse.vtp.launching.VoiceXMLBrowserInput;
import org.eclipse.vtp.launching.VoiceXMLBrowserProcess;
import org.eclipse.vtp.launching.VoiceXMLLogMessage;
import org.eclipse.vtp.launching.internal.tellme.config.ITellmeVoiceXMLBrowserConstants;

public class TellmeStudioBrowser extends Object implements IVoiceXMLBrowser {

	// the IVoiceXMLBrowser interface specifies this starts the browser so that
	// it is ready for input
	// in the start method we will
	// 1. set up Studio to handle running our application
	// 2. start the call until it is ACTIVE
	// 3. spawn a thread to oversee fetching the log from Studio
	// 4. spawn a thread to watch for the end of the phone call
	// 5. exit

	/**
	 * 
	 */
	private static final String PACKAGE_NAME = "org.eclipse.vtp.launching.tellme"; //$NON-NLS-1$

	IVoipCall callObject = null;

	final private static boolean DEBUGGING = TellmePlugin.getDefault()
			.isDebugging();

	final private static SimpleDateFormat time = new SimpleDateFormat(
			"mm:ss.SSS", Locale.getDefault()); //$NON-NLS-1$

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.vtp.launching.IVoiceXMLBrowser#start()
	 */
	public void start() {
		if (DEBUGGING) {
			System.out.println(time.format(new Date())
					+ ": TellmeStudioBrowser.start() called."); //$NON-NLS-1$
			System.out.flush();
		}

		// 0. set up Dialer plugin

		try {
			CallStudio.initializeCall();
		} catch (CoreException e) {
			DebugPlugin.log(new Status(IStatus.ERROR,
					"org.eclipse.vtp.launching.tellme", IStatus.ERROR, //$NON-NLS-1$
					Messages.TellmeStudioBrowser_0, e));
		} catch (VoipDialerException e) {
			CallState.reportCallWatchFail(e.getMessage());
			CallState.reportLogFetchFail(""); //$NON-NLS-1$
			CallState.reportLogListFail(""); //$NON-NLS-1$
			CallState.setCallWatchDone();
			CallState.setLogFetchDone();
			CallState.setLogListDone();
			CallState.endLaunch();
//			DisplayActions.showError(e.getMessage());
//			
//	        DebugEvent[] event = new DebugEvent[1];
//	        event[0] = new DebugEvent(this, DebugEvent.TERMINATE);
//	        DebugPlugin.getDefault().fireDebugEventSet(event);
	        return;
		}

		// 1. set up Studio to handle running our application

		/**
		 * Get the Studio phone number from the plugin preferences.ini file
		 * 
		 */

		final String studioPhone = TellmePlugin.getDefault()
				.getPluginPreferences().getString(
						ITellmeVoiceXMLBrowserConstants.STUDIO_PHONE);

		/**
		 * Studio configuration parameters These are derived from the
		 * launchConfiguration, and passed into this class as local properties
		 * 
		 */
		final String studioAppURL = (String) getProperty(IVoiceXMLBrowserConstants.LAUNCH_URL);
		final String studioEMAIL = (String) getProperty(ITellmeVoiceXMLBrowserConstants.DEVELOPER_EMAIL);
		final String studioPSWD = (String) getProperty(ITellmeVoiceXMLBrowserConstants.DEVELOPER_PSWD);
		final String studioID = (String) getProperty(ITellmeVoiceXMLBrowserConstants.DEVELOPER_ID);
		final String studioPIN = (String) getProperty(ITellmeVoiceXMLBrowserConstants.DEVELOPER_PIN);
		final boolean noAutoLogin = "true" //$NON-NLS-1$
				.equals(getProperty(ITellmeVoiceXMLBrowserConstants.NO_AUTO_LOGIN));

		// frequently used login string
		final String loginString = "usr_id=" + studioEMAIL + "&usr_pin=" //$NON-NLS-1$//$NON-NLS-2$
				+ studioPSWD;

		/**
		 * remember the Studio login inside the StudioInteractions class
		 */
		StudioInteractions.setLogin(loginString);

		/**
		 * Request Studio switch to scratchpad mode
		 */

		StudioInteractions.studioScratchpadMode();

		/**
		 * send the new Application URL to Studio
		 */
		StudioInteractions.sendApplicationURL(studioAppURL);

		// 2. start the call until it is ACTIVE

		synchronized (CallState.getSynch()) {
			CallState.setCallWatchDone(); // initialize state of call watch as
			// done
			CallState.setLogListDone(); // initialize state of log list as done
			// do not initialize logfetchdone when getting log at end
			CallState.setLogFetchDone(); // initialize state of log fetch as
			// done
		}

		/**
		 * Make the phone call to Studio
		 */
		try {
			callObject = CallStudio.startCall(studioPhone, noAutoLogin,
					studioID, studioPIN);
		} catch (VoipDialerException vde) {
			CallState
					.reportCallWatchFail(vde.getMessage()
							+ ": " + (vde.getCause() != null ? vde.getCause().getMessage() : "")); //$NON-NLS-1$//$NON-NLS-2$
			CallState.endLaunch();
			// DisplayActions.showError(vde.getMessage());
			return;
		}

		// 3. spawn a thread to oversee fetching the log from Studio

		new Thread(new GetLogListChange(), "watchloglist").start(); //$NON-NLS-1$

		// 4. spawn a thread to watch for the end of the phone call

		final IVoipCall localCall = callObject;

		final Runnable watchCall = new Runnable() {
			public void run() {
				CallState.initCallWatch();
				if (DEBUGGING) {
					System.out
							.println(time.format(new Date())
									+ ": TellmeStudioBrowser, thread watchcall started"); //$NON-NLS-1$
					System.out.flush();
				}
				String callResult = "NOT OK"; //$NON-NLS-1$
				try {
					callResult = CallStudio.finishCall(localCall);
				} catch (VoipDialerException vde) {
					callResult = vde.getMessage();
				}
				if (DEBUGGING) {
					System.out
							.println(time.format(new Date())
									+ ": TellmeStudioBrowser, call is finished, ending watchcall thread"); //$NON-NLS-1$
					System.out.flush();
				}
				if ("OK".equals(callResult)) { //$NON-NLS-1$
					CallState.reportCallWatchSucceed(callResult);
				} else {
					CallState.reportCallWatchFail(callResult);
					// DisplayActions.showError(callResult);
				}
				synchronized (CallState.getSynch()) {
					CallState.setCallWatchDone();
					// try {
					// localCall.hangUp();
					// localCall.getConnection().disconnect();
					// } catch (VoipDialerException exc) {
					// DebugPlugin.log(new Status(IStatus.ERROR,
					// "org.eclipse.vtp.launching.tellme",
					// IStatus.ERROR,
					// "Problem while hanging up or disconnecting",
					// exc));
					// }
					if (CallState.launchDone()) {
						CallState.endLaunch();
					}
				}
			}
		};

		new Thread(watchCall, "watchcall").start(); //$NON-NLS-1$

		// 5. exit

	}

	public void stop() {
		try {
			if (DEBUGGING) {
				System.out.println(time.format(new Date())
						+ ": TellmeStudioBrowser.stop() called. callObject is " //$NON-NLS-1$
						+ (callObject == null ? "" : "not") + " null"); //$NON-NLS-1$//$NON-NLS-2$//$NON-NLS-3$
	            StackTraceElement[] whereami = new Throwable().getStackTrace();
	            for (int i = 1; i < Math.min(whereami.length, 2); i++){
	            	System.out.println(whereami[i].toString());
	            }
				System.out.flush();
			}
			if (callObject != null) {
				if (DEBUGGING) {
					System.out
							.println(time.format(new Date())
									+ ": TellmeStudioBrowser.stop() call status is " //$NON-NLS-1$
									+ (callObject == null ? "no call object" : CallStudio.callStatus(callObject.status()))); //$NON-NLS-1$
					System.out.flush();
				}
				switch (callObject.status()) {
				case IVoipCall.CALL_ACTIVE:
					callObject.hangUp();
				case IVoipCall.CALL_COMPLETED:
					callObject.getConnection().disconnect();
				default:
					callObject = null;
				}
			}
			CallState.setStopping(true);
		} catch (VoipDialerException exc) {
			DebugPlugin.log(new Status(IStatus.ERROR,
					"org.eclipse.vtp.launching.tellme", IStatus.ERROR, //$NON-NLS-1$
					Messages.TellmeStudioBrowser_4, exc));
		}
	}

	public void sendInput(final VoiceXMLBrowserInput input) {
		if (DEBUGGING) {
			System.out.println(time.format(new Date())
					+ ": TellmeStudioBrowser.sendInput() called."); //$NON-NLS-1$
			System.out.flush();
		}
		if (input.getInputType() == VoiceXMLBrowserInput.TYPE_DTMF) {
			final String dtmf = (String) input.getInput();
			try {
				callObject.sendDTMF(dtmf);
			} catch (VoipDialerException e) {
				DebugPlugin.log(new Status(IStatus.ERROR,
						"org.eclipse.vtp.launching.tellme", IStatus.ERROR, //$NON-NLS-1$
						Messages.TellmeStudioBrowser_5, e));
				DisplayActions.showError(Messages.TellmeStudioBrowser_6, e
						.getMessage());
			}
		} else {
			DebugPlugin.log(new Status(IStatus.WARNING,
					"org.eclipse.vtp.launching.tellme", IStatus.WARNING, //$NON-NLS-1$
					Messages.TellmeStudioBrowser_7 + input.getInputType() + Messages.TellmeStudioBrowser_8,
					null));
		}

	}

	public void setProperty(final String propertyname, final Object value) {
		localPropStore.put(propertyname, value);
	}

	private Object getProperty(final String propertyname) {
		return localPropStore.get(propertyname);
	}

	private VoiceXMLBrowserProcess holdThatProcess = null;

	public void setProcess(final VoiceXMLBrowserProcess process) {
		holdThatProcess = process;
	}

	public VoiceXMLBrowserProcess getProcess() {
		return holdThatProcess;
	}

	public boolean hasCapability(final String capability) {
		if (DEBUGGING) {
			System.out.println(time.format(new Date())
					+ ": TellmeStudioBrowser.hasCapability() called."); //$NON-NLS-1$
			System.out.flush();
		}
		boolean hasCap = false;

		if (capability.equals(IVoiceXMLBrowser.CAPABILITY_INTERACTIVE)
				|| capability.equals(IVoiceXMLBrowser.CAPABILITY_DTMF)
				|| capability.equals(IVoiceXMLBrowser.CAPABILITY_LOG_EVENT)) {
			hasCap = true;
		}

		return hasCap;
	}

	final private Map localPropStore = new HashMap();

}
