/*******************************************************************************
 * Copyright (c) 2000, 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.jdt.internal.ui.fix;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.eclipse.core.runtime.CoreException;

import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;

import org.eclipse.jface.dialogs.IDialogSettings;

import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.dom.CompilationUnit;

import org.eclipse.jdt.internal.corext.fix.ControlStatementsFix;
import org.eclipse.jdt.internal.corext.fix.IFix;
import org.eclipse.jdt.internal.corext.util.JavaModelUtil;

import org.eclipse.jdt.ui.text.java.IProblemLocation;

public class ControlStatementsCleanUp extends AbstractCleanUp {
	
	/**
	 * Adds block to control statement body if the body is not a block.<p>
	 * i.e.:<pre><code>
	 * 	 if (b) foo(); -> if (b) {foo();}</code></pre>
	 */
	public static final int ADD_BLOCK_TO_CONTROL_STATEMENTS= 1;
	
	/**
	 * Convert for loops to enhanced for loops.<p>
	 * i.e.:<pre><code>
	 * for (int i = 0; i < array.length; i++) {} -> for (int element : array) {}</code></pre>
	 */
	public static final int CONVERT_FOR_LOOP_TO_ENHANCED_FOR_LOOP= 2;
	
	/**
	 * Remove unnecessary blocks in control statement bodies.<p>
	 * i.e.:<pre><code>
	 *   if (b) {foo();} -> if (b) foo();
	 */
	public static final int REMOVE_UNNECESSARY_BLOCKS= 4;

	private static final int DEFAULT_FLAG= 0;
	private static final String SECTION_NAME= "CleanUp_ControlStatements"; //$NON-NLS-1$

	public ControlStatementsCleanUp(int flag) {
		super(flag);
	}

	public ControlStatementsCleanUp(IDialogSettings settings) {
		super(getSection(settings, SECTION_NAME), DEFAULT_FLAG);
	}

	/**
	 * {@inheritDoc}
	 */
	public IFix createFix(CompilationUnit compilationUnit) throws CoreException {
		if (compilationUnit == null)
			return null;
		
		return ControlStatementsFix.createCleanUp(compilationUnit,
				isFlag(ADD_BLOCK_TO_CONTROL_STATEMENTS),
				isFlag(REMOVE_UNNECESSARY_BLOCKS),
				isFlag(CONVERT_FOR_LOOP_TO_ENHANCED_FOR_LOOP));
	}

	/**
	 * {@inheritDoc}
	 */
	public IFix createFix(CompilationUnit compilationUnit, IProblemLocation[] problems) throws CoreException {
		//No warnings generated by the compiler
		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	public Map getRequiredOptions() {
		return null;
	}
	
	public void saveSettings(IDialogSettings settings) {
		super.saveSettings(getSection(settings, SECTION_NAME));
	}

	/**
	 * {@inheritDoc}
	 */
	public Control createConfigurationControl(Composite parent, IJavaProject project) {
		
			Button button= new Button(parent, SWT.CHECK);
			button.setText(MultiFixMessages.ControlStatementsCleanUp_useBlocks_checkBoxLabel);
			button.setLayoutData(new GridData(SWT.LEFT, SWT.CENTER, true, false));
		indent(button);
		
			Composite sub= new Composite(parent, SWT.NONE);
			sub.setLayoutData(new GridData(SWT.FILL, SWT.TOP, true, false));
			GridLayout layout= new GridLayout(1, false);
			layout.marginHeight= 0;
			layout.marginWidth= 0;
			sub.setLayout(layout);
		indent(sub);
	
			final int[] flags= new int[] {ADD_BLOCK_TO_CONTROL_STATEMENTS, REMOVE_UNNECESSARY_BLOCKS};
			final int[] uiFlags= new int[] {1073741824, 536870912};
			final String[] labels= new String[] {MultiFixMessages.ControlStatementsCleanUp_always_checkBoxLabel, MultiFixMessages.ControlStatementsCleanUp_removeIfPossible_checkBoxLabel};
	
			createSubGroup(sub, button, SWT.RADIO, flags, labels, uiFlags, false);
		
		
		Button box1= addCheckBox(parent, CONVERT_FOR_LOOP_TO_ENHANCED_FOR_LOOP, MultiFixMessages.ControlStatementsCleanUp_convertLoops_checkBoxLabel);
		indent(box1);
		if (project != null && !JavaModelUtil.is50OrHigher(project)) {
			box1.setEnabled(false);
		}
		
		return parent;
	}

	/**
	 * {@inheritDoc}
	 */
	public String[] getDescriptions() {
		List result= new ArrayList();
		if (isFlag(ADD_BLOCK_TO_CONTROL_STATEMENTS))
			result.add(MultiFixMessages.CodeStyleMultiFix_ConvertSingleStatementInControlBodeyToBlock_description);
		if (isFlag(CONVERT_FOR_LOOP_TO_ENHANCED_FOR_LOOP))
			result.add(MultiFixMessages.Java50CleanUp_ConvertToEnhancedForLoop_description);
		
		return (String[])result.toArray(new String[result.size()]);
	}

	/**
	 * {@inheritDoc}
	 */
	public boolean canFix(CompilationUnit compilationUnit, IProblemLocation problem) throws CoreException {
		return false;
	}

	/**
	 * {@inheritDoc}
	 */
	public int maximalNumberOfFixes(CompilationUnit compilationUnit) {
		return -1;
	}

}
