/*******************************************************************************
 * Copyright (c) 2000, 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.jdt.internal.junit.ui;

import org.eclipse.core.runtime.IStatus;

import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.texteditor.ITextEditor;

import org.eclipse.jdt.core.IJavaElement;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.IMethod;
import org.eclipse.jdt.core.ISourceRange;
import org.eclipse.jdt.core.IType;
import org.eclipse.jdt.core.ITypeHierarchy;
import org.eclipse.jdt.core.JavaConventions;
import org.eclipse.jdt.core.JavaModelException;

import org.eclipse.jdt.internal.junit.Messages;

import org.eclipse.jface.dialogs.MessageDialog;

/**
 * Open a class on a Test method.
 */
public class OpenTestAction extends OpenEditorAction {
	
	private String fMethodName;
	private ISourceRange fRange;
	
	public OpenTestAction(TestRunnerViewPart testRunner, String className, String method) {
		this(testRunner, className, method, true);
	}
	
	public OpenTestAction(TestRunnerViewPart testRunner, String className) {
		this(testRunner, className, null);
	}

	public OpenTestAction(TestRunnerViewPart testRunner, String className, String method, boolean activate) {
		super(testRunner, className, activate);
		PlatformUI.getWorkbench().getHelpSystem().setHelp(this, IJUnitHelpContextIds.OPENTEST_ACTION);
		fMethodName= method;
	}

	protected IJavaElement findElement(IJavaProject project, String className) throws JavaModelException {
		IType type= findType(project, className);
		if (type == null) 
			return null;
		
		if (fMethodName == null)
			return type;
			
		IMethod method= findMethod(type);
		if (method == null) {
			ITypeHierarchy typeHierarchy= type.newSupertypeHierarchy(null);
			IType[] types= typeHierarchy.getAllSuperclasses(type);
			for (int i= 0; i < types.length; i++) {
				method= findMethod(types[i]);
				if (method != null)
					break;
			}
		}
		if (method == null) {
			String title= JUnitMessages.OpenTestAction_error_title; 
			String message= Messages.format(JUnitMessages.OpenTestAction_error_methodNoFound, fMethodName); 
			MessageDialog.openInformation(getShell(), title, message);
			return type;
		}
		fRange= method.getNameRange();
		return method;
	}
	
	IMethod findMethod(IType type) {
		IStatus status= JavaConventions.validateMethodName(fMethodName);
		if (! status.isOK())
			return null;
		IMethod method= type.getMethod(fMethodName, new String[0]);
		if (method != null && method.exists())
			return method;
		return null;
	}
	
	protected void reveal(ITextEditor textEditor) {
		if (fRange != null)
			textEditor.selectAndReveal(fRange.getOffset(), fRange.getLength());
	}

	public boolean isEnabled() {
		try {
			return findType(getLaunchedProject(), getClassName()) != null;
		} catch (JavaModelException e) {
		}
		return false;
	}
}
