/*
 *+------------------------------------------------------------------------+
 *| Licensed Materials - Property of IBM                                   |
 *| (C) Copyright IBM Corp. 2005.  All Rights Reserved.                    |
 *|                                                                        |
 *| US Government Users Restricted Rights - Use, duplication or disclosure |
 *| restricted by GSA ADP Schedule Contract with IBM Corp.                 |
 *+------------------------------------------------------------------------+
 */
package org.eclipse.wst.rdb.internal.core.rte.fe;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.wst.rdb.internal.core.containment.ContainmentServiceImpl;
import org.eclipse.wst.rdb.internal.core.rte.DDLGenerator;
import org.eclipse.wst.rdb.internal.core.rte.EngineeringOption;
import org.eclipse.wst.rdb.internal.core.rte.EngineeringOptionCategory;
import org.eclipse.wst.rdb.internal.models.sql.constraints.CheckConstraint;
import org.eclipse.wst.rdb.internal.models.sql.constraints.ForeignKey;
import org.eclipse.wst.rdb.internal.models.sql.constraints.Index;
import org.eclipse.wst.rdb.internal.models.sql.constraints.UniqueConstraint;
import org.eclipse.wst.rdb.internal.models.sql.schema.SQLObject;
import org.eclipse.wst.rdb.internal.models.sql.tables.PersistentTable;
import org.eclipse.wst.rdb.internal.models.sql.tables.Trigger;
import org.eclipse.wst.rdb.internal.models.sql.tables.ViewTable;

public final class GenericDdlGenerator implements DDLGenerator {
    public String[] generateDDL(SQLObject[] elements, IProgressMonitor progressMonitor) {
    	String[] statements = new String[0];

    	if (this.generateCreateStatement()) {
    			statements = this.createSQLObjects(elements, this.generateQuotedIdentifiers(),
                this.generateFullyQualifiedNames(), progressMonitor);
    	}
        if(this.generateDropStatement()) {
            String[] drop = this.dropSQLObjects(elements, this.generateQuotedIdentifiers(),
                    this.generateFullyQualifiedNames(), progressMonitor);
            String[] temp = statements;
            statements = new String[temp.length + drop.length];
            for(int i=0; i<drop.length; ++i) {
                statements[i] = drop[i];
            }
            for(int i=0; i<temp.length; ++i) {
                statements[i+drop.length] = temp[i];
            }
            
        }
        return statements;
    }

    public String[] createSQLObjects(SQLObject[] elements, boolean quoteIdentifiers, boolean qualifyNames, IProgressMonitor progressMonitor) {
        String[] statements = this.createStatements(elements, quoteIdentifiers,
        		qualifyNames, progressMonitor, 100);
        return statements;
    }

    public String[] dropSQLObjects(SQLObject[] elements, boolean quoteIdentifiers, boolean qualifyNames, IProgressMonitor progressMonitor) {
        String[] statements = this.dropStatements(elements, quoteIdentifiers,
        		qualifyNames, progressMonitor, 100);
        return statements;
    }
    
    private String[] createStatements(SQLObject[] elements, boolean quoteIdentifiers, boolean qualifyNames, IProgressMonitor progressMonitor, int task) {
        GenericDdlScript script = new GenericDdlScript();
        if(this.builder == null) {
            this.builder = new GenericDdlBuilder();
        }
        Iterator it = this.getAllContainedDisplayableElementSet(elements).iterator();
        while(it.hasNext()) {
            Object o = it.next();
            if(o instanceof PersistentTable) {
            	if (!this.generateTables()) continue;
                String statement = builder.createTable((PersistentTable) o, quoteIdentifiers, qualifyNames);
                if(statement != null) script.addCreateTableStatement(statement);
            }
            else if(o instanceof ViewTable) {
            	if (!this.generateViews()) continue;
                String statement = builder.createView((ViewTable) o, quoteIdentifiers, qualifyNames);
                if(statement != null) script.addCreateViewStatement(statement);
            }
            else if(o instanceof Trigger) {
            	if (!this.generateTriggers()) continue;
                String statement = builder.createTrigger((Trigger) o, quoteIdentifiers, qualifyNames);
                if(statement != null) script.addCreateTriggerStatement(statement);
            }
            else if(o instanceof CheckConstraint) {
                String statement = builder.addCheckConstraint((CheckConstraint) o, quoteIdentifiers, qualifyNames);
                if(statement != null) script.addAlterTableAddConstraintStatement(statement);
            }
            else if(o instanceof UniqueConstraint) {
                String statement = builder.addUniqueConstraint((UniqueConstraint) o, quoteIdentifiers, qualifyNames);
                if(statement != null) script.addAlterTableAddConstraintStatement(statement);
            }
            else if(o instanceof ForeignKey) {
                String statement = builder.addForeignKey((ForeignKey) o, quoteIdentifiers, qualifyNames);
                if(statement != null) script.addAlterTableAddForeignKeyStatement(statement);
            }
            else if(o instanceof Index) {
            	if (!this.generateIndexes()) continue;
                String statement = builder.createIndex((Index) o, quoteIdentifiers, qualifyNames);
                if(statement != null) script.addCreateIndexStatement(statement);
            }
        }
        return script.getStatements();
    }
    
    private String[] dropStatements(SQLObject[] elements, boolean quoteIdentifiers, boolean qualifyNames, IProgressMonitor progressMonitor, int task) {
        GenericDdlScript script = new GenericDdlScript();
        if(this.builder == null) {
            this.builder = new GenericDdlBuilder();
        }
        Iterator it = this.getAllContainedDisplayableElementSet(elements).iterator();
        while(it.hasNext()) {
            Object o = it.next();
            if(o instanceof PersistentTable) {
            	if (!this.generateTables()) continue;
                String statement = builder.dropTable((PersistentTable) o, quoteIdentifiers, qualifyNames);
                if(statement != null) script.addDropTableStatement(statement);
            }
            else if(o instanceof ViewTable) {
            	if (!this.generateViews()) continue;
                String statement = builder.dropView((ViewTable) o, quoteIdentifiers, qualifyNames);
                if(statement != null) script.addDropViewStatement(statement);
            }
            else if(o instanceof Trigger) {
            	if (!this.generateTriggers()) continue;
                String statement = builder.dropTrigger((Trigger) o, quoteIdentifiers, qualifyNames);
                if(statement != null) script.addDropTriggerStatement(statement);
            }
            else if(o instanceof CheckConstraint) {
                String statement = builder.dropTableConstraint((CheckConstraint) o, quoteIdentifiers, qualifyNames);
                if(statement != null) script.addAlterTableDropConstraintStatement(statement);
            }
            else if(o instanceof UniqueConstraint) {
                String statement = builder.dropTableConstraint((UniqueConstraint) o, quoteIdentifiers, qualifyNames);
                if(statement != null) script.addAlterTableDropConstraintStatement(statement);
            }
            else if(o instanceof ForeignKey) {
                String statement = builder.dropTableConstraint((ForeignKey) o, quoteIdentifiers, qualifyNames);
                if(statement != null) script.addAlterTableDropForeignKeyStatement(statement);
            }
            else if(o instanceof Index) {
            	if (!this.generateIndexes()) continue;
                String statement = builder.dropIndex((Index) o, quoteIdentifiers, qualifyNames);
                if(statement != null) script.addDropIndexStatement(statement);
            }
        }
        return script.getStatements();
    }
    
    public EngineeringOption[] getOptions() {
        if(this.options == null) {
            this.options = GenericDdlGenerationOptions.createDDLGenerationOptions(this.getOptionCategories());
        }
        
        return this.options;
    }
    
    public EngineeringOptionCategory[] getOptionCategories() {
        if(this.categories == null) {
            this.categories = GenericDdlGenerationOptions.createDDLGenerationOptionCategories();
        }
        return this.categories;
    }
    
    private boolean generateDropStatement() {
        return this.getOptions()[GenericDdlGenerationOptions.GENERATE_DROP_STATEMENTS].getBoolean();
    }

    private boolean generateCreateStatement() {
        return this.getOptions()[GenericDdlGenerationOptions.GENERATE_CREATE_STATEMENTS].getBoolean();
    }
    
    private final boolean generateQuotedIdentifiers() {
        return this.getOptions()[GenericDdlGenerationOptions.GENERATE_QUOTED_IDENTIFIER].getBoolean();
    }

    private final boolean generateFullyQualifiedNames() {
        return this.getOptions()[GenericDdlGenerationOptions.GENERATE_FULLY_QUALIFIED_NAME].getBoolean();
    }

	public final boolean generateTables(){
		return this.getOptions()[GenericDdlGenerationOptions.GENERATE_TABLES].getBoolean();
	}

	public boolean generateIndexes(){
		return this.getOptions()[GenericDdlGenerationOptions.GENERATE_INDICES].getBoolean();
	}
	
	public boolean generateViews(){
		return this.getOptions()[GenericDdlGenerationOptions.GENERATE_VIEWS].getBoolean();
	}

	public boolean generateTriggers(){
		return this.getOptions()[GenericDdlGenerationOptions.GENERATE_TRIGGERS].getBoolean();
	}
    
	private Set getAllContainedDisplayableElementSet(SQLObject[] elements) {
        Set s = new HashSet();
        for(int i=0; i<elements.length; ++i) {
            s.add(elements[i]);
            s.addAll(ContainmentServiceImpl.INSTANCE.getAllContainedDisplayableElements(elements[i]));
        }
        return s;
    }

    private EngineeringOption[] options = null;
    private EngineeringOptionCategory[] categories = null;
	private GenericDdlBuilder builder = null;
}
