/*******************************************************************************
 * Copyright (c) 2001, 2004 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.wst.common.ui.properties.internal.view;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.wst.common.ui.properties.internal.provisional.ISectionDescriptor;
import org.eclipse.wst.common.ui.properties.internal.provisional.ITypeMapper;


/**
 * Provides a section filtering mechanism where the selection is an
 * IStructuredSelection and filtering is based on class.
 * 
 * @author Anthony Hunter <a
 *         href="mailto:anthonyh@ca.ibm.com">anthonyh@ca.ibm.com </a>
 */
public class TabbedPropertyRegistryClassSectionFilter {

	private ITypeMapper typeMapper = null;

	/**
	 * constructor.
	 */
	protected TabbedPropertyRegistryClassSectionFilter(ITypeMapper typeMapper) {
		super();
		this.typeMapper = typeMapper;
	}

	/**
	 * Verifies if the property section extension represented by sectionElement
	 * applies to the given input.
	 */
	protected boolean appliesToSelection(ISectionDescriptor descriptor,
			ISelection selection) {

		boolean ret = false;

		if (selection instanceof IStructuredSelection
			&& false == selection.isEmpty()) {

			ret = true;

			Set effectiveTypes = new HashSet();

			for (Iterator i = ((IStructuredSelection) selection).iterator(); ret == true
				&& i.hasNext();) {

				Object object = i.next();
				Class effectiveType = object.getClass();

				// if there is no filter applied, remapType == effectiveType
				Class remapType = remapType(descriptor.getFilter(), object,
					effectiveType);

				if (effectiveTypes.add(remapType)) {

					// the effective types of the selection
					if (false == appliesToEffectiveType(descriptor, remapType)) {
						ret = false;
					}
				}
			}
		}

		return ret;
	}

	private boolean appliesToEffectiveType(ISectionDescriptor descriptor,
			Class inputClass) {

		ArrayList classTypes = getClassTypes(inputClass);

		List sectionInputTypes = descriptor.getInputTypes();
		for (Iterator j = sectionInputTypes.iterator(); j.hasNext();) {
			String type = (String) j.next();
			if (classTypes.contains(type)) {
				// found a match
				return true;
			}
		}

		return false;
	}

	/**
	 * see if this type needs to be remapped to a different type by the
	 * application of a the section filter or the global filter.
	 * 
	 * @param sectionFilter
	 * @param object
	 * @param effectiveType
	 * @return the new type (or the effectiveType if no filter has been applied)
	 */
	protected Class remapType(ITypeMapper sectionFilter, Object object,
			Class effectiveType) {
		if (sectionFilter != null) {
			// there is section filter for this section, use it to remap type
			return sectionFilter.remapType(object, effectiveType);
		} else if (typeMapper != null) {
			// there was no section filter - use page filter if there is one
			return typeMapper.remapType(object, effectiveType);
		}
		return effectiveType;
	}

	/**
	 * Returns the classes and interfaces the given target class
	 * extends/implements.
	 */
	protected ArrayList getClassTypes(Class target) {
		ArrayList result = new ArrayList();
		// add classes
		List classes = computeClassOrder(target);
		for (Iterator i = classes.iterator(); i.hasNext();) {
			result.add(((Class) i.next()).getName());
		}
		// add interfaces
		result.addAll(computeInterfaceOrder(classes));
		return result;
	}

	private List computeClassOrder(Class target) {
		List result = new ArrayList(4);
		Class clazz = target;
		while (clazz != null) {
			result.add(clazz);
			clazz = clazz.getSuperclass();
		}
		return result;
	}

	private List computeInterfaceOrder(List classes) {
		List result = new ArrayList(4);
		Map seen = new HashMap(4);
		for (Iterator iter = classes.iterator(); iter.hasNext();) {
			Class[] interfaces = ((Class) iter.next()).getInterfaces();
			internalComputeInterfaceOrder(interfaces, result, seen);
		}
		return result;
	}

	private void internalComputeInterfaceOrder(Class[] interfaces, List result,
			Map seen) {
		List newInterfaces = new ArrayList(seen.size());
		for (int i = 0; i < interfaces.length; i++) {
			Class interfac = interfaces[i];
			if (seen.get(interfac) == null) {
				result.add(interfac.getName());
				seen.put(interfac, interfac);
				newInterfaces.add(interfac);
			}
		}
		for (Iterator iter = newInterfaces.iterator(); iter.hasNext();) {
			internalComputeInterfaceOrder(
				((Class) iter.next()).getInterfaces(), result, seen);
		}
	}
}
