/*******************************************************************************
 * Copyright (c) 2003, 2004 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 * IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.wst.common.navigator.internal.views.extensions;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.ui.internal.ActionExpression;

/**
 * The navigator registry stores all navigator content extension descriptors the navigator action
 * builder and all navigator view descriptors.
 */
public class NavigatorRegistry {
	private Map rootContentDescriptors = new HashMap();
	private Map viewDescriptors = new HashMap();
	private List contentDescriptors = new ArrayList();

	/**
	 * Creates a new instance of the receiver.
	 */
	public NavigatorRegistry() {
	}

	/**
	 * Adds a navigator content descriptor to the receiver.
	 * 
	 * @param the
	 *            navigator content descriptor to add
	 */
	void add(NavigatorContentDescriptor descriptor) {
		contentDescriptors.add(descriptor);

		Collections.sort(contentDescriptors, new Comparator() {

			/*
			 * (non-Javadoc)
			 * 
			 * @see java.util.Comparator#compare(java.lang.Object, java.lang.Object)
			 */
			public int compare(Object o1, Object o2) {
				NavigatorContentDescriptor descriptor1 = null;
				NavigatorContentDescriptor descriptor2 = null;
				if (o1 instanceof NavigatorContentDescriptor)
					descriptor1 = (NavigatorContentDescriptor) o1;
				if (o2 instanceof NavigatorContentDescriptor)
					descriptor2 = (NavigatorContentDescriptor) o2;

				if (descriptor1 != null && descriptor2 != null)
					return descriptor1.getPriority() - descriptor2.getPriority();

				return -1;
			}


		});
	}

	/**
	 * Adds a navigator view extension descriptor to the receiver.
	 * 
	 * @param the
	 *            navigator view descriptor to add
	 */
	void add(NavigatorViewerDescriptor descriptor) {
		// more than one root descriptor can be defined
		String viewTargetId = descriptor.getViewerId();
		String id = descriptor.getRootExtensionId();
		NavigatorContentDescriptor extDesc = getContentDescriptor(id);

		// does the view extension contribute a navigator extension?
		if (extDesc != null) {
			List list = (List) rootContentDescriptors.get(viewTargetId);
			if (list == null) {
				list = new ArrayList();
				rootContentDescriptors.put(viewTargetId, list);
			}
			// a navigator extension that references a navigator view extension
			// is automatically a root extension
			extDesc.setRoot(true);
			list.add(extDesc);
		}
		List list = (List) viewDescriptors.get(viewTargetId);
		if (list == null) {
			list = new ArrayList();
			viewDescriptors.put(viewTargetId, list);
		}
		list.add(descriptor);
	}

	/**
	 * Returns the highest priority content descriptor which enables for the given element.
	 * 
	 * @param element
	 *            the element to return the best content descriptor for
	 * @return the best content descriptor for the given element.
	 */
	public NavigatorContentDescriptor getBestContentDescriptor(Object element) {
		NavigatorContentDescriptor extDescriptor = null;
		int priority = Integer.MAX_VALUE;
		for (int i = 0; i < contentDescriptors.size(); i++) {
			NavigatorContentDescriptor descriptor = (NavigatorContentDescriptor) contentDescriptors.get(i);

			if (descriptor.isEnabledFor(element))
				if (descriptor.getPriority() < priority) {
					priority = descriptor.getPriority();
					extDescriptor = descriptor;
				}
		}
		return extDescriptor;
	}

	/**
	 * Added method.
	 * 
	 * Returns all content descriptor(s) which enable for the given element.
	 * 
	 * @param element
	 *            the element to return the best content descriptor for
	 * @return the best content descriptor for the given element.
	 */
	public NavigatorContentDescriptor[] getAllContentDescriptors() {
		NavigatorContentDescriptor[] finalDescriptors = new NavigatorContentDescriptor[contentDescriptors.size()];
		contentDescriptors.toArray(finalDescriptors);
		return finalDescriptors;
	}

	/**
	 * Added method.
	 * 
	 * Returns all content descriptor(s) which enable for the given element.
	 * 
	 * @param element
	 *            the element to return the best content descriptor for
	 * @return the best content descriptor for the given element.
	 */
	public List getEnabledContentDescriptors(Object element) {
		List descriptors = new ArrayList();

		/* Find other ContentProviders which enable for this object */
		for (int i = 0; i < contentDescriptors.size(); i++) {
			NavigatorContentDescriptor descriptor = (NavigatorContentDescriptor) contentDescriptors.get(i);


			if (descriptor.isEnabledFor(element)) {
				descriptors.add(descriptor);
			}
		}

		return descriptors;
	}


	/**
	 * Returns the content descriptor that contributed the given element.
	 * 
	 * @param element
	 *            the element the content descriptor should be returned for
	 * @return the content descriptor that contributed the given element.
	 */
	public NavigatorContentDescriptor getBestContentContributionDescriptor(Object element) {
		NavigatorContentDescriptor extDescriptor = null;
		int priority = Integer.MAX_VALUE;
		for (int i = 0; i < contentDescriptors.size(); i++) {
			NavigatorContentDescriptor descriptor = (NavigatorContentDescriptor) contentDescriptors.get(i);
			ActionExpression enablement = descriptor.getContributionEnablement();
			if (enablement == null) {
				if (descriptor.getPriority() < priority)
					extDescriptor = descriptor;
			} else if (enablement.isEnabledFor(element) && descriptor.getPriority() < priority) {
				priority = descriptor.getPriority();
				extDescriptor = descriptor;
			}
		}
		return extDescriptor;
	}

	/**
	 * Returns the navigator content descriptor with the given id.
	 * 
	 * @param id
	 *            the id of the content descriptor that should be returned
	 */
	public NavigatorContentDescriptor getContentDescriptor(String id) {
		if (id == null)
			return null;
		for (int i = 0; i < contentDescriptors.size(); i++) {
			NavigatorContentDescriptor descriptor = (NavigatorContentDescriptor) contentDescriptors.get(i);
			if (descriptor.getId().equals(id))
				return descriptor;
		}
		return null;
	}

	public void removeContentDescriptor(String id) {
		for (int i = 0; i < contentDescriptors.size(); i++) {
			NavigatorContentDescriptor descriptor = (NavigatorContentDescriptor) contentDescriptors.get(i);
			if (descriptor.getId().equals(id))
				contentDescriptors.remove(i);
		}
	}

	/**
	 * Returns the navigator content descriptor with the given id.
	 * 
	 * @param id
	 *            the id of the content descriptor that should be returned
	 */
	public int getCategoryForContentDescriptor(String descriptorID) {
		return getCategoryForContentDescriptor(getContentDescriptor(descriptorID));
	}

	public int getBestCategory(Object element) {
		return getCategoryForContentDescriptor(getBestContentContributionDescriptor(element));

	}

	protected int getCategoryForContentDescriptor(NavigatorContentDescriptor descriptor) {
		if (descriptor != null)
			return descriptor.getPriority();
		return Integer.MAX_VALUE;
	}

	/**
	 * Returns the top level content descriptors for the navigator viewer with the given id.
	 * 
	 * @param viewerId
	 *            the id of the <code>NavigatorViewer</code> for which to return the root content
	 *            descriptors
	 */
	public List getRootContentDescriptors(String viewerId) {
		return (List) rootContentDescriptors.get(viewerId);
	}

	/**
	 * Returns the view descriptors for the navigator viewer with the given id.
	 * 
	 * @param viewerId
	 *            the id of the <code>NavigatorViewer</code> for which to return the view
	 *            descriptors
	 */
	public List getViewDescriptors(String viewerId) {
		return (List) viewDescriptors.get(viewerId);
	}


	/*
	 * public NavigatorContentDescriptor[] getAllDescriptorsEnabledByActivity() {
	 * IWorkbenchActivitySupport workbenchActivitySupport =
	 * PlatformUI.getWorkbench().getActivitySupport(); IActivityManager activityManager =
	 * workbenchActivitySupport.getActivityManager(); Set enabledActivityIDs =
	 * activityManager.getEnabledActivityIds(); List enabledDescriptors = new ArrayList();
	 * NavigatorContentDescriptor[] availableDescriptors = registry.getAllContentDescriptors(); for
	 * (int i = 0; i < availableDescriptors.length; i++) { IIdentifier identifier =
	 * activityManager.getIdentifier(WorkbenchActivityHelper.createUnifiedId(availableDescriptors[i].getContribution()));
	 * Set idActivities = identifier.getActivityIds(); if (idActivities.size() == 0) {
	 * enabledDescriptors.add(availableDescriptors[i]); } else { Iterator iterator =
	 * idActivities.iterator(); boolean added = false; while (!added && iterator.hasNext()) { if
	 * (enabledActivityIDs.contains(iterator.next())) { added = true;
	 * enabledDescriptors.add(availableDescriptors[i]); } } } } NavigatorContentDescriptor[] result =
	 * new NavigatorContentDescriptor[enabledDescriptors.size()];
	 * enabledDescriptors.toArray(result); return result; } @author mdelder
	 * 
	 * TODO To change the template for this generated type comment go to Window - Preferences - Java -
	 * Code Style - Code Templates
	 */

}