/*******************************************************************************
 * Copyright (c) 2003, 2004 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 * IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.jst.common.navigator.internal.actions;

import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.jdt.internal.ui.actions.ActionMessages;
import org.eclipse.jdt.ui.IContextMenuConstants;
import org.eclipse.jdt.ui.actions.JdtActionConstants;
import org.eclipse.jface.action.GroupMarker;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.IViewPart;
import org.eclipse.ui.IWorkbenchSite;
import org.eclipse.ui.actions.ActionGroup;
import org.eclipse.ui.actions.OpenWithMenu;
import org.eclipse.wst.common.frameworks.internal.AdaptabilityUtility;
import org.eclipse.wst.common.navigator.internal.provisional.views.ICommonMenuConstants;

/**
 * Action group that adds the actions opening a new editor to the context menu and the action bar's
 * navigate menu.
 * 
 * <p>
 * This class may be instantiated; it is not intended to be subclassed.
 * </p>
 * 
 * <B>Derived from the class of the same name in the JDT actions package due to limitations of the
 * existing class concerning how it manages adaptable objects. </B>
 * 
 * @plannedfor 2.0
 */
public class OpenEditorActionGroup extends ActionGroup {


	public static final String OPENWITH_MENU_ID = "org.eclipse.wst.common.frameworks.internal.navigator.openWith.menu"; //$NON-NLS-1$
	private IWorkbenchSite fSite;

	private OpenAction fOpen;

	/**
	 * Creates a new <code>OpenActionGroup</code>. The group requires that the selection provided
	 * by the part's selection provider is of type <code>
	 * org.eclipse.jface.viewers.IStructuredSelection</code>.
	 * 
	 * @param part
	 *            the view part that owns this action group
	 */
	public OpenEditorActionGroup(IViewPart part) {
		fSite = part.getSite();
		fOpen = new OpenAction(fSite);
		initialize(fSite.getSelectionProvider());
	}

	/**
	 * Returns the open action managed by this action group.
	 * 
	 * @return the open action. Returns <code>null</code> if the group doesn't provide any open
	 *         action
	 */
	public IAction getOpenAction() {
		return fOpen;
	}

	private void initialize(ISelectionProvider provider) {
		ISelection selection = provider.getSelection();
		fOpen.update(selection);
		/* if (!fIsEditorOwner) { */
		provider.addSelectionChangedListener(fOpen);
		/* } */
	}

	/*
	 * (non-Javadoc) Method declared in ActionGroup
	 */
	public void fillActionBars(IActionBars actionBar) {
		super.fillActionBars(actionBar);
		setGlobalActionHandlers(actionBar);
	}

	/*
	 * (non-Javadoc) Method declared in ActionGroup
	 */
	public void fillContextMenu(IMenuManager menu) {
		super.fillContextMenu(menu);
		fOpen.selectionChanged(getSelection());
		appendToGroup(menu, fOpen);
		/* if (!fIsEditorOwner) { */
		if (fOpen.isEnabled())
			addOpenWithMenu(menu);
		/* } */
	}

	/*
	 * @see ActionGroup#dispose()
	 */
	public void dispose() {
		ISelectionProvider provider = fSite.getSelectionProvider();
		provider.removeSelectionChangedListener(fOpen);
		super.dispose();
	}

	private void setGlobalActionHandlers(IActionBars actionBars) {
		actionBars.setGlobalActionHandler(JdtActionConstants.OPEN, fOpen);
	}

	private void appendToGroup(IMenuManager menu, IAction action) {
		if (action.isEnabled())
			menu.appendToGroup(IContextMenuConstants.GROUP_OPEN, action);
	}

	private void addOpenWithMenu(IMenuManager menu) {
		IStructuredSelection ss = getSelection();

		if (ss == null || ss.size() != 1)
			return;

		Object o = ss.getFirstElement();
		/*
		 * if (!(o instanceof IAdaptable)) return;
		 * 
		 * IAdaptable element = (IAdaptable) o;
		 */
		IResource resource = (IResource) AdaptabilityUtility.getAdapter(o, IResource.class);

		// Create a menu flyout.
		IMenuManager submenu = new MenuManager(ActionMessages.OpenWithMenu_label, OPENWITH_MENU_ID); //$NON-NLS-1$
		submenu.add(new GroupMarker(ICommonMenuConstants.COMMON_MENU_TOP));
		if (resource != null && !(resource instanceof IProject) && !(resource instanceof IFolder))
			submenu.add(new OpenWithMenu(fSite.getPage(), resource));

		addCustomOpenWith(submenu);
		submenu.add(new GroupMarker(ICommonMenuConstants.COMMON_MENU_ADDITIONS));

		// Add the submenu.
		if (submenu.getItems().length > 2 && submenu.isEnabled())
			menu.appendToGroup(IContextMenuConstants.GROUP_OPEN, submenu);
	}

	/**
	 * @param submenu
	 */
	protected void addCustomOpenWith(IMenuManager submenu) {
		// TODO MDE Finish implementation of custom open with menu
		//		ISelection selection = getContext().getSelection();
		//		if (selection == null || selection.isEmpty() || !(selection instanceof
		// IStructuredSelection))
		//			return;
		//		
		//		IStructuredSelection structuredSelection = (IStructuredSelection) selection;
		//
		//		INavigatorContentExtension[] extensions = getContributors(structuredSelection);
		//		for (int i = 0; i < extensions.length; i++) {
		//			if (extensions[i].getEditActionGroup() != null) {
		//				extensions[i].getEditActionGroup().setContext(new
		// ActionContext(getContext().getSelection()));
		//				try {
		//					extensions[i].getEditActionGroup().fillOpenWithMenu(submenu);
		//				} catch (RuntimeException e) {
		//					Logger.getLogger().logError(e);
		//				}
		//			}
		//		}
	}

	public IStructuredSelection getSelection() {

		ISelection selection = getContext().getSelection();
		if (selection.isEmpty() || !(selection instanceof IStructuredSelection))
			return StructuredSelection.EMPTY;
		IStructuredSelection ss = (IStructuredSelection) selection;
		return ss;
	}

}