/*
 *+------------------------------------------------------------------------+
 *| Licensed Materials - Property of IBM                                   |
 *| (C) Copyright IBM Corp. 2005.  All Rights Reserved.                    |
 *|                                                                        |
 *| US Government Users Restricted Rights - Use, duplication or disclosure |
 *| restricted by GSA ADP Schedule Contract with IBM Corp.                 |
 *+------------------------------------------------------------------------+
 */

package org.eclipse.wst.rdb.internal.derby.ddl;

import java.util.Iterator;

import org.eclipse.wst.rdb.internal.core.rte.fe.GenericDdlBuilder;
import org.eclipse.wst.rdb.internal.models.sql.expressions.ValueExpression;
import org.eclipse.wst.rdb.internal.models.sql.routines.Parameter;
import org.eclipse.wst.rdb.internal.models.sql.routines.ParameterMode;
import org.eclipse.wst.rdb.internal.models.sql.routines.Procedure;
import org.eclipse.wst.rdb.internal.models.sql.routines.Routine;
import org.eclipse.wst.rdb.internal.models.sql.routines.RoutineResultTable;
import org.eclipse.wst.rdb.internal.models.sql.routines.Source;
import org.eclipse.wst.rdb.internal.models.sql.routines.UserDefinedFunction;
import org.eclipse.wst.rdb.internal.models.sql.schema.Database;
import org.eclipse.wst.rdb.internal.models.sql.schema.IdentitySpecifier;
import org.eclipse.wst.rdb.internal.models.sql.tables.Column;

public class DerbyDdlBuilder extends GenericDdlBuilder {
 
    protected final static String IN                   = "IN"; //$NON-NLS-1$
    protected final static String OUT                  = "OUT"; //$NON-NLS-1$
    protected final static String INOUT                = "INOUT"; //$NON-NLS-1$
    protected final static String LOCATOR              = "LOCATOR"; //$NON-NLS-1$
    protected final static String EXTERNAL             = "EXTERNAL"; //$NON-NLS-1$
    protected final static String RETURNS              = "RETURNS"; //$NON-NLS-1$
    protected final static String CAST                 = "CAST"; //$NON-NLS-1$
    protected final static String FROM                 = "FROM"; //$NON-NLS-1$
    protected final static String COMMENT              = "COMMENT"; //$NON-NLS-1$
    protected final static String IS                   = "IS"; //$NON-NLS-1$
    protected final static String COLUMN               = "COLUMN"; //$NON-NLS-1$
	
    public String createProcedure(Procedure procedure, boolean quoteIdentifiers, boolean qualifyNames) {

        Database database = procedure.getSchema().getDatabase();

        String text = CREATE + SPACE + PROCEDURE + SPACE + getName(procedure, quoteIdentifiers, qualifyNames)
    	+ SPACE + LEFT_PARENTHESIS + getParameters(procedure) + RIGHT_PARENTHESIS;

        text += NEWLINE + TAB + "LANGUAGE " + procedure.getLanguage(); //$NON-NLS-1$
        text += getExternalNameOption(procedure,quoteIdentifiers,qualifyNames);
        text += getParameterStyleOption(procedure);
    	
        return text;      
    }
	
    public String createUserDefinedFunction(UserDefinedFunction function, boolean quoteIdentifiers, boolean qualifyNames) {
    	Database database = function.getSchema().getDatabase();

    	String text = CREATE + SPACE + FUNCTION + SPACE + getName(function, quoteIdentifiers, qualifyNames);
    	
    	Source source = function.getSource();
    	if (source != null) {
    		text += source.getBody();
    	}
    	text += getExternalNameOption(function,quoteIdentifiers,qualifyNames);
        
        return text;      
    }

    public String dropProcedure(Procedure procedure, boolean quoteIdentifiers, boolean qualifyNames) {
    	Database database = procedure.getSchema().getDatabase();
   		return DROP + SPACE + PROCEDURE + SPACE + getName(procedure, quoteIdentifiers, qualifyNames);
    }
    	
    public String dropFunction(UserDefinedFunction function, boolean quoteIdentifiers, boolean qualifyNames) {
    	Database database = function.getSchema().getDatabase();
   		return DROP + SPACE + FUNCTION + SPACE + getName(function, quoteIdentifiers, qualifyNames);
    }
    
    
    protected String getColumnString(Column column, boolean quoteIdentifiers) {
        String columnDefinition =  super.getColumnString(column, quoteIdentifiers);
        IdentitySpecifier spec = column.getIdentitySpecifier();
        ValueExpression v = column.getGenerateExpression();
        if(spec != null) {
            columnDefinition += " GENERATED ALWAYS AS IDENTITY "; //$NON-NLS-1$
            columnDefinition += SPACE + LEFT_PARENTHESIS + this.getIdentityString(spec) + RIGHT_PARENTHESIS;
        }
        else if(v != null){
            columnDefinition += " GENERATED ALWAYS AS " + v.getSQL();             //$NON-NLS-1$
        }

        return columnDefinition;
    }
	
    protected String getIdentityString(IdentitySpecifier identitySpecifier) {
        String clause = "START WITH " + identitySpecifier.getStartValue() //$NON-NLS-1$
        	+ " ,INCREMENT BY " + identitySpecifier.getIncrement(); //$NON-NLS-1$
        return clause;
    }
    
    protected String getParameters(Routine routine) {
        String parameters = EMPTY_STRING;
        Iterator it = routine.getParameters().iterator();
        while(it.hasNext()) {
            Parameter p = (Parameter) it.next();
            ParameterMode mode = p.getMode();
            if(mode == ParameterMode.INOUT_LITERAL) {
                parameters += INOUT + SPACE;
            }
            else if(mode == ParameterMode.OUT_LITERAL) {
                parameters += OUT + SPACE;
            }
            String name = p.getName();
            if(name != null && name.length() != 0) {
                parameters += p.getName() + SPACE;
            }
            parameters += this.getDataTypeString(p, routine.getSchema());
            if(p.isLocator()) {
                parameters += SPACE + AS + SPACE + LOCATOR;
            }
            if(it.hasNext()) {
                parameters += COMMA + NEWLINE + TAB + TAB;                
            }
        }
        return parameters;
    }
    
    protected String getExternalNameOption(Routine routine,boolean quoteIdentifiers, boolean qualifyNames) {
        String option = NEWLINE + TAB + EXTERNAL;
        String externalName = routine.getExternalName();
        if(externalName != null && externalName.length() != 0) {
            String schemaName = routine.getSchema().getName();
            
            if (routine.getLanguage().equalsIgnoreCase("JAVA")){ //$NON-NLS-1$
            	externalName = this.getSingleQuotedString(externalName);
            } else {
	            if(quoteIdentifiers) {
	            	externalName = this.getDoubleQuotedString(externalName);
	                schemaName  = this.getDoubleQuotedString(schemaName);
	            }
	            if(qualifyNames) {
	            	externalName = schemaName + DOT + externalName;
	            }
            }

            option += " NAME " + externalName; //$NON-NLS-1$
        }
        return option;
    }
 
    protected String getParameterStyleOption(Routine routine) {
        return NEWLINE + TAB + "PARAMETER STYLE " + routine.getParameterStyle(); //$NON-NLS-1$
    }

    protected String getFunctionReturnsClause(UserDefinedFunction function) {
        if(function.getReturnScaler() != null) {
            Parameter scaler = function.getReturnScaler();
            String text = RETURNS + SPACE + this.getDataTypeString(scaler,function.getSchema());
            if(function.getReturnCast() != null) {
                Parameter cast = function.getReturnCast();
                text += SPACE + CAST + SPACE + FROM + SPACE + this.getDataTypeString(cast,function.getSchema());
            }
            return text;
        }
        else if(function.getReturnTable() != null) {
            RoutineResultTable resultTable = function.getReturnTable();
            String text = RETURNS + SPACE + TABLE + SPACE + LEFT_PARENTHESIS;
            Iterator it = resultTable.getColumns().iterator();
            while(it.hasNext()) {
                Column c = (Column) it.next();
                text += c.getName() + SPACE + getDataTypeString(c,function.getSchema());
                if(it.hasNext()) {
                    text += COMMA + SPACE;
                }
            }
            text += RIGHT_PARENTHESIS;
            return text;
        }
        else {
        	Parameter parameter = function.getReturnCast();
        	if (parameter != null) {
        		return parameter.getName();
        	} else {
        		return null;
        	}
        }
    }
    
    
}

