/*
 *+------------------------------------------------------------------------+
 *| Licensed Materials - Property of IBM                                   |
 *| (C) Copyright IBM Corp. 2005.  All Rights Reserved.                    |
 *|                                                                        |
 *| US Government Users Restricted Rights - Use, duplication or disclosure |
 *| restricted by GSA ADP Schedule Contract with IBM Corp.                 |
 *+------------------------------------------------------------------------+
 */
package org.eclipse.wst.rdb.internal.derby.ddl;


import java.util.HashSet;
import java.util.Iterator;
import java.util.ResourceBundle;
import java.util.Set;
import java.util.Vector;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExecutableExtension;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.wst.rdb.internal.core.containment.ContainmentServiceImpl;
import org.eclipse.wst.rdb.internal.core.rte.DDLGenerator;
import org.eclipse.wst.rdb.internal.core.rte.EngineeringOption;
import org.eclipse.wst.rdb.internal.core.rte.EngineeringOptionCategory;
import org.eclipse.wst.rdb.internal.core.rte.EngineeringOptionCategoryID;
import org.eclipse.wst.rdb.internal.core.rte.EngineeringOptionID;
import org.eclipse.wst.rdb.internal.core.rte.fe.GenericDdlGenerationOptions;
import org.eclipse.wst.rdb.internal.models.sql.constraints.CheckConstraint;
import org.eclipse.wst.rdb.internal.models.sql.constraints.ForeignKey;
import org.eclipse.wst.rdb.internal.models.sql.constraints.Index;
import org.eclipse.wst.rdb.internal.models.sql.constraints.UniqueConstraint;
import org.eclipse.wst.rdb.internal.models.sql.routines.Procedure;
import org.eclipse.wst.rdb.internal.models.sql.routines.UserDefinedFunction;
import org.eclipse.wst.rdb.internal.models.sql.schema.SQLObject;
import org.eclipse.wst.rdb.internal.models.sql.tables.BaseTable;
import org.eclipse.wst.rdb.internal.models.sql.tables.Trigger;
import org.eclipse.wst.rdb.internal.models.sql.tables.ViewTable;

public final class DerbyDdlGenerator implements DDLGenerator ,IExecutableExtension {
	
	public void setInitializationData(IConfigurationElement config, String propertyName, Object data) throws CoreException {
		this.product = config.getAttribute("product"); //$NON-NLS-1$
		this.version = config.getAttribute("version"); //$NON-NLS-1$
	}
	
    public String[] generateDDL(SQLObject[] elements, IProgressMonitor progressMonitor) {
        String[] statements = this.createSQLObjects(elements, this.generateQuotedIdentifiers(),
                this.generateFullyQualifiedNames(), progressMonitor);
        if(this.generateDropStatement()) {
            String[] drop = this.dropSQLObjects(elements, this.generateQuotedIdentifiers(),
                    this.generateFullyQualifiedNames(), progressMonitor);
            String[] temp = statements;
            statements = new String[temp.length + drop.length];
            for(int i=0; i<drop.length; ++i) {
                statements[i] = drop[i];
            }
            for(int i=0; i<temp.length; ++i) {
                statements[i+drop.length] = temp[i];
            }
        }
        return statements;
    }

    public String[] createSQLObjects(SQLObject[] elements, boolean quoteIdentifiers, boolean qualifyNames, IProgressMonitor progressMonitor) {
    	return this.createStatements(elements,quoteIdentifiers,qualifyNames,progressMonitor,100);
    }
    	
    public String[] dropSQLObjects(SQLObject[] elements, boolean quoteIdentifiers, boolean qualifyNames, IProgressMonitor progressMonitor) {
    	return this.dropStatements(elements,quoteIdentifiers,qualifyNames,progressMonitor,100);
    }
    
    private String[] createStatements(SQLObject[] elements, boolean quoteIdentifiers, boolean qualifyNames, IProgressMonitor progressMonitor, int task) {
        DerbyDdlScript script = new DerbyDdlScript();
        if(this.builder == null) {
            this.builder = new DerbyDdlBuilder();
        }
        Iterator it = this.getAllContainedDisplayableElementSet(elements).iterator();
        while(it.hasNext()) {
            Object o = it.next();
            if(o instanceof BaseTable) {
            	if (!this.generateTables()) continue;
                String statement = builder.createTable((BaseTable) o, quoteIdentifiers, qualifyNames);
                if(statement != null) script.addCreateTableStatement(statement);
            }
            else if(o instanceof ViewTable) {
            	if (!this.generateViews()) continue;
                String statement = builder.createView((ViewTable) o, quoteIdentifiers, qualifyNames);
                if(statement != null) script.addCreateViewStatement(statement);
            }
            else if(o instanceof Procedure) {
            	if (!this.generateStoredProcedures()) continue;
                String statement = builder.createProcedure((Procedure) o, quoteIdentifiers, qualifyNames);
                if(statement != null) script.addCreateRoutineStatement(statement);
            }
            else if(o instanceof UserDefinedFunction) {
            	if (!this.generateFunctions()) continue;
                String statement = builder.createUserDefinedFunction((UserDefinedFunction) o, quoteIdentifiers, qualifyNames);
                if(statement != null) script.addCreateRoutineStatement(statement);
            }
            else if(o instanceof Trigger) {
            	if (!this.generateTriggers()) continue;
                String statement = builder.createTrigger((Trigger) o, quoteIdentifiers, qualifyNames);
                if(statement != null) script.addCreateTriggerStatement(statement);
            }
            else if(o instanceof CheckConstraint) {
            	if (!this.generateTables()) continue;
            	String statement = builder.addCheckConstraint((CheckConstraint) o, quoteIdentifiers, qualifyNames);
                if(statement != null) script.addAlterTableAddConstraintStatement(statement);
            }
            else if(o instanceof UniqueConstraint) {
            	if (!this.generateTables()) continue;
                String statement = builder.addUniqueConstraint((UniqueConstraint) o, quoteIdentifiers, qualifyNames);
                if(statement != null) script.addAlterTableAddConstraintStatement(statement);
            }
            else if(o instanceof ForeignKey) {
            	if (!this.generateTables()) continue;
                String statement = builder.addForeignKey((ForeignKey) o, quoteIdentifiers, qualifyNames);
                if(statement != null) script.addAlterTableAddForeignKeyStatement(statement);
            }
            else if(o instanceof Index) {
            	if (!this.generateIndexes()) continue;
                String statement = builder.createIndex((Index) o, quoteIdentifiers, qualifyNames);
                if(statement != null) script.addCreateIndexStatement(statement);
            }
        }
        return script.getStatements();
    }
    
    private String[] dropStatements(SQLObject[] elements, boolean quoteIdentifiers, boolean qualifyNames, IProgressMonitor progressMonitor, int task) {
        DerbyDdlScript script = new DerbyDdlScript();
        if(this.builder == null) {
            this.builder = new DerbyDdlBuilder();
        }
        Iterator it = this.getAllContainedDisplayableElementSet(elements).iterator();
        while(it.hasNext()) {
            Object o = it.next();
            if(o instanceof  BaseTable) {
            	if (!this.generateTables()) continue;
                String statement = builder.dropTable((BaseTable) o, quoteIdentifiers, qualifyNames);
                if(statement != null) script.addDropTableStatement(statement);
            }
            else if(o instanceof ViewTable) {
            	if (!this.generateViews()) continue;
                String statement = builder.dropView((ViewTable) o, quoteIdentifiers, qualifyNames);
                if(statement != null) script.addDropViewStatement(statement);
            }
            else if(o instanceof Procedure) {
            	if (!this.generateStoredProcedures()) continue;
                String statement = builder.dropProcedure((Procedure) o, quoteIdentifiers, qualifyNames);
                if(statement != null) script.addDropRoutineStatement(statement);
            }
            else if(o instanceof UserDefinedFunction) {
            	if (!this.generateFunctions()) continue;
                String statement = builder.dropFunction((UserDefinedFunction) o, quoteIdentifiers, qualifyNames);
                if(statement != null) script.addDropRoutineStatement(statement);
            }
            else if(o instanceof Trigger) {
            	if (!this.generateTriggers()) continue;
                String statement = builder.dropTrigger((Trigger) o, quoteIdentifiers, qualifyNames);
                if(statement != null) script.addDropTriggerStatement(statement);
            }
            else if(o instanceof CheckConstraint) {
            	if (!this.generateTables()) continue;
                String statement = builder.dropTableConstraint((CheckConstraint) o, quoteIdentifiers, qualifyNames);
                if(statement != null) script.addAlterTableDropConstraintStatement(statement);
            }
            else if(o instanceof UniqueConstraint) {
            	if (!this.generateTables()) continue;
                String statement = builder.dropTableConstraint((UniqueConstraint) o, quoteIdentifiers, qualifyNames);
                if(statement != null) script.addAlterTableDropConstraintStatement(statement);
            }
            else if(o instanceof ForeignKey) {
            	if (!this.generateTables()) continue;
                String statement = builder.dropTableConstraint((ForeignKey) o, quoteIdentifiers, qualifyNames);
                if(statement != null) script.addAlterTableDropForeignKeyStatement(statement);
            }
            else if(o instanceof Index) {
            	if (!this.generateIndexes()) continue;
                String statement = builder.dropIndex((Index) o, quoteIdentifiers, qualifyNames);
                if(statement != null) script.addDropIndexStatement(statement);
            }
        }
        return script.getStatements();
    }
    
    public EngineeringOption[] getOptions() {
        if(this.options == null) {
            ResourceBundle resource = ResourceBundle.getBundle("org.eclipse.wst.rdb.internal.core.rte.fe.GenericDdlGeneration"); //$NON-NLS-1$
            Vector optionVec = new Vector();
            EngineeringOption[] temp = GenericDdlGenerationOptions.createDDLGenerationOptions(this.getOptionCategories());
            for (int i = 0; i < temp.length; i++ ) {
            	optionVec.add(temp[i]);
            }
            
            EngineeringOptionCategory additional_element =null;
            for (int i = 0; i < categories.length; i++) {
            	if (categories[i].getId().equals(EngineeringOptionCategoryID.GENERATE_ELEMENTS)){
            		additional_element = categories[i];
            	}
            }

            optionVec.add(new EngineeringOption(EngineeringOptionID.GENERATE_STOREDPROCEDURES,resource.getString("GENERATE_STOREDPROCEDURE"), resource.getString("GENERATE_STOREDPROCEDURE_DES"), true,additional_element)); //$NON-NLS-1$ //$NON-NLS-2$
            optionVec.add(new EngineeringOption(EngineeringOptionID.GENERATE_FUNCTIONS,resource.getString("GENERATE_FUNCTION"), resource.getString("GENERATE_FUNCTION_DES"),true,additional_element)); //$NON-NLS-1$ //$NON-NLS-2$

            this.options = new EngineeringOption[optionVec.size()];
            optionVec.copyInto(this.options);
            
        }
        
        return this.options;
    }
    
    public EngineeringOptionCategory[] getOptionCategories() {
        if(this.categories == null) {
            this.categories = GenericDdlGenerationOptions.createDDLGenerationOptionCategories();
        }
        return this.categories;
    }
    

    private final boolean generateQuotedIdentifiers() {
        return this.getOptions()[GenericDdlGenerationOptions.GENERATE_QUOTED_IDENTIFIER].getBoolean();
    }

    private final boolean generateFullyQualifiedNames() {
        return this.getOptions()[GenericDdlGenerationOptions.GENERATE_FULLY_QUALIFIED_NAME].getBoolean();
    }

    private boolean generateDropStatement() {
        return this.getOptions()[GenericDdlGenerationOptions.GENERATE_DROP_STATEMENTS].getBoolean();
    }
    
    private boolean generateCreateStatement() {
        return this.getOptions()[GenericDdlGenerationOptions.GENERATE_CREATE_STATEMENTS].getBoolean();
    }

 	public final boolean generateTables(){
		return this.getOptions()[GenericDdlGenerationOptions.GENERATE_TABLES].getBoolean();
	}

	public boolean generateIndexes(){
		return this.getOptions()[GenericDdlGenerationOptions.GENERATE_INDICES].getBoolean();
	}
	
	public boolean generateViews(){
		return this.getOptions()[GenericDdlGenerationOptions.GENERATE_VIEWS].getBoolean();
	}

	public boolean generateTriggers(){
		return this.getOptions()[GenericDdlGenerationOptions.GENERATE_TRIGGERS].getBoolean();
	}

	public boolean generateStoredProcedures(){
		return this.getOptionValue(EngineeringOptionID.GENERATE_STOREDPROCEDURES);
	}

	public boolean generateFunctions(){
		return this.getOptionValue(EngineeringOptionID.GENERATE_FUNCTIONS);
	}
    
	private boolean getOptionValue(String optionID){
		boolean ret = false;
		for (int i = 0; i < this.getOptions().length; i++){
			EngineeringOption option = (EngineeringOption) this.options[i];
			if (option.getId().equals(optionID)){
				ret = option.getBoolean();
				break;
			}
		}
		return ret;
	}
    
    
	private Set getAllContainedDisplayableElementSet(SQLObject[] elements) {
        Set s = new HashSet();
        for(int i=0; i<elements.length; ++i) {
            s.add(elements[i]);
            s.addAll(ContainmentServiceImpl.INSTANCE.getAllContainedDisplayableElements(elements[i]));
        }
        return s;
    }

	

    private String product;
    private String version;
    private EngineeringOption[] options = null;
    private EngineeringOptionCategory[] categories = null;
	private DerbyDdlBuilder builder = null;
}

