/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - Initial API and implementation
 *******************************************************************************/
package org.eclipse.jst.server.geronimo.core.internal;

import java.io.File;
import java.io.FileOutputStream;
import java.io.InputStream;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.IPath;
import org.eclipse.emf.ecore.xmi.XMIResource;
import org.eclipse.jst.j2ee.internal.web.deployables.J2EEFlexProjWebDeployable;
import org.eclipse.jst.j2ee.web.componentcore.util.WebArtifactEdit;
import org.eclipse.jst.j2ee.webapplication.WebApp;
import org.eclipse.jst.server.core.IJ2EEModule;
import org.eclipse.jst.server.core.IWebModule;
import org.eclipse.jst.server.geronimo.core.internal.model.WebAppDeploymentPlan;
import org.eclipse.wst.common.componentcore.ComponentCore;
import org.eclipse.wst.common.componentcore.resources.ComponentHandle;
import org.eclipse.wst.common.componentcore.resources.IFlexibleProject;
import org.eclipse.wst.common.componentcore.resources.IVirtualComponent;
import org.eclipse.wst.server.core.IModule;

public class GeronimoUtils {

	public static String getConfigId(IModule module) {
		String configId=null;
		
		WebAppDeploymentPlan deploymentPlan=getDeploymentPlan(module);
		if(deploymentPlan!=null)
			configId=deploymentPlan.getConfigId();
		
		if(configId==null)
			configId=getWebAppId(module);
		
		return configId;
	}

	public static String getContextRoot(IModule module) {
		String contextRoot = null;
			
		WebAppDeploymentPlan deploymentPlan = getDeploymentPlan(module);
		if (deploymentPlan != null)
			contextRoot=deploymentPlan.getContextRoot();
						
		if (contextRoot == null)
			contextRoot=getWebAppId(module);
			
		return contextRoot;
	}

	public static String getWebAppId(IModule module) {
		// use the module ID
		String moduleId = module.getId();
					
		IJ2EEModule j2eeModule = (IJ2EEModule) module.loadAdapter(IJ2EEModule.class, null);
		if(j2eeModule != null && j2eeModule instanceof J2EEFlexProjWebDeployable)
		{
			J2EEFlexProjWebDeployable j2eeFlexWeb=(J2EEFlexProjWebDeployable)j2eeModule;
			//j2eeFlexWeb
			WebArtifactEdit webEdit=null;
			
			try {
				ComponentHandle handle=j2eeFlexWeb.getComponentHandle();
				webEdit = WebArtifactEdit.getWebArtifactEditForRead(handle);
				WebApp webApp=webEdit.getWebApp();
				XMIResource res = (XMIResource) webApp.eResource();
				moduleId=res.getID(webApp);
			} finally {
				if(webEdit!=null)
					webEdit.dispose();
			}
		}
		
		if (moduleId != null && moduleId.length() > 0)
			return moduleId;
	
		// ...but if there is no defined module ID, pick the best alternative
		
		IPath moduleLocation = j2eeModule.getLocation();
		if (moduleLocation != null) {
			moduleId = moduleLocation.removeFileExtension().lastSegment();
		}
	
		if (j2eeModule instanceof IWebModule) {
			// A better choice is to use the context root
			// For wars most appservers use the module name
			// as the context root
			String contextRoot = ((IWebModule) j2eeModule).getContextRoot();
			if (contextRoot.charAt(0) == '/')
				moduleId = contextRoot.substring(1);
		}
	
		return moduleId;
	}
	
	
	private static WebAppDeploymentPlan getDeploymentPlan(IModule module)
	{		
		IFile deployPlanFile=getDeployPlanFile(module);
				
		WebAppDeploymentPlan plan=null;
		try {
			if(deployPlanFile.exists())
				plan=new WebAppDeploymentPlan(deployPlanFile.getContents());
			else
				plan=createDeploymentPlan(deployPlanFile, module.getProject().getName(), module.getName());			
		} catch (Exception e) {
			Trace.trace(Trace.SEVERE, "Error reading deployment plan", e);
		}

		return plan;
	}

	private static IFile getDeployPlanFile(IModule module) {
		IProject project=module.getProject();
		
		IFlexibleProject flexProject = ComponentCore.createFlexibleProject(project);
		IVirtualComponent component=flexProject.getComponent(module.getName());
		IPath deployPlanPath=component
			.getRootFolder()
			.getUnderlyingFolder()
			.getProjectRelativePath()
			.append("WEB-INF")
			.append("geronimo-jetty.xml");

		IFile planFile =project.getFile(deployPlanPath);
		return planFile;
	}

	// Temporary workaround - ensure the .deployable copy 
	// of the file is also updated (the WTP should be doing this)
	public static void copyDeploymentPlanToDeployable(IModule module) {		
		IFile planFile=getDeployPlanFile(module);
		
		if (planFile.exists()) {
			try {
				IJ2EEModule j2eeModule = (IJ2EEModule) module.loadAdapter(IJ2EEModule.class, null);
				File deployableFile=j2eeModule
						.getLocation()		// returns .deployable location
						.addTrailingSeparator()
						.append("WEB-INF")
						.addTrailingSeparator()
						.append("geronimo-jetty.xml")
						.toFile();
				
				InputStream input = planFile.getContents();
				FileOutputStream output = new FileOutputStream(deployableFile);
				byte[] buffer=new byte[1000];
				int bytesRead=0;
				while (bytesRead > -1) {
					bytesRead=input.read(buffer);
					if(bytesRead>0)
						output.write(buffer, 0, bytesRead);
				}
				output.close();
				input.close();
			} catch (Exception e) {
				Trace.trace(Trace.SEVERE, "Error copying deployment plan", e);
			}
		}
	}

	private static WebAppDeploymentPlan createDeploymentPlan(IFile deployPlanFile, String projectName, String moduleName) {
		WebAppDeploymentPlan plan=null;		
		InputStream is=null;
		try {			
			plan=new WebAppDeploymentPlan();
			
			// Now initialize with some reasonable defaults
			plan.setConfigId(projectName+"/"+moduleName);
			plan.setContextRoot("/"+moduleName);
			plan.setContextPriorityClassloader(false);			
			
			is=plan.saveToStream();
			deployPlanFile.create(is, false, null);
		} catch (Exception e) {
			Trace.trace(Trace.SEVERE, "Error creating deployment plan", e);
		} finally {
			try {
				if (is!=null)
					is.close();
			} catch (Exception e) {
				// do nothing
			}			
		}
		
		return plan;
	}
}