/*******************************************************************************
 * Copyright (c) 2004 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.wst.rdb.derby.internal.ui.connection.dialogs;

import java.text.MessageFormat;

import org.eclipse.jface.wizard.Wizard;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Text;
import org.eclipse.swt.widgets.Widget;
import org.eclipse.wst.rdb.connection.internal.ui.factories.RSCCoreUIWidgetFactory;
import org.eclipse.wst.rdb.connection.internal.ui.providers.IServerConnectionInformation;
import org.eclipse.wst.rdb.connection.internal.ui.providers.IServerConnectionUIProvider;
import org.eclipse.wst.rdb.connection.internal.ui.wizards.NewConnectionWizard;
import org.eclipse.wst.rdb.derby.internal.ui.util.ResourceLoader;
import org.eclipse.wst.rdb.internal.core.definition.DatabaseDefinition;

/**
 * @author ljulien
 */
public class DerbyUniversalNewConnectionProvider implements
        IServerConnectionUIProvider, Listener {
    private static final RSCCoreUIWidgetFactory factory = RSCCoreUIWidgetFactory.INSTANCE;

    private static final ResourceLoader resource = ResourceLoader.INSTANCE;

    private static final String CUI_NEWCW_DATABASE_LBL_UI_ = resource
            .queryString("CUI_NEWCW_UNI_DATABASE_LBL_UI_"); //$NON-NLS-1$

    private static final String CUI_NEWCW_CREATEDB_BTN_UI_ = resource
            .queryString("CUI_NEWCW_CREATEDB_BTN_UI_"); //$NON-NLS-1$

    private static final String CUI_NEWCW_HOST_LBL_UI_ = resource
            .queryString("CUI_NEWCW_UNI_HOST_LBL_UI_"); //$NON-NLS-1$

    private static final String CUI_NEWCW_PORT_LBL_UI_ = resource
            .queryString("CUI_NEWCW_UNI_PORT_LBL_UI_"); //$NON-NLS-1$

    private static final String CUI_NEWCW_JDBCDRIVERCLS_LBL_UI_ = resource
            .queryString("CUI_NEWCW_UNI_JDBCDRIVERCLS_LBL_UI_"); //$NON-NLS-1$

    private static final String CUI_NEWCW_JDBCCLSLOCATION_LBL_UI_ = resource
            .queryString("CUI_NEWCW_UNI_JDBCCLSLOCATION_LBL_UI_"); //$NON-NLS-1$

    private static final String CUI_NEWCW_JARBROWSE_BTN_UI_ = resource
            .queryString("CUI_NEWCW_UNI_JARBROWSE_BTN_UI_"); //$NON-NLS-1$

    private static final String CUI_NEWCW_CONNECTIONURL_LBL_UI_ = resource
            .queryString("CUI_NEWCW_UNI_CONNECTIONURL_LBL_UI_"); //$NON-NLS-1$

    private static final String DRIVER_CLASS_NAME = "com.ibm.db2.jcc.DB2Driver"; //$NON-NLS-1$

    private static final String CREATE_EQUALS_TRUE_TEXT = "create=true"; //$NON-NLS-1$

    private static final String RETURN_MESSAGES = ":retrieveMessagesFromServerOnGetMessage=true;"; //$NON-NLS-1$
    
    private IServerConnectionInformation connectionInformation;

    private Label databaseLabel;

    private Text databaseCombo;

    private Button createCheck;

    private Label hostLabel;

    private Text hostCombo;

    private Label portLabel;

    private Label driverLabel;

    private Text driverDisplay;

    private Label locationLabel;

    private Text locationCombo;

    private Button browseLocation;

    private Label urlLabel;

    private Text urlDisplay;

    private Text portCombo;

    private WizardPage parentPage;

    private Composite parentComposite;

    private void createLayout(Composite parentComposite) {
        GridLayout layout = new GridLayout();
        layout.marginWidth = 0;
        layout.marginHeight = 0;
        layout.numColumns = 3;
        layout.horizontalSpacing = 5;
        layout.verticalSpacing = 5;
        parentComposite.setLayout(layout);
    }

    public boolean determinePageCompletion() {
        return validateControl();
    }

    private void setConnectionInformation() {
        connectionInformation.setDriverClassName(DRIVER_CLASS_NAME);
        connectionInformation.setDatabaseName(databaseCombo.getText());
        connectionInformation.setLoadingPath(locationCombo.getText());
        connectionInformation.setURL(urlDisplay.getText());
        if (determinePageCompletion()) {
            connectionInformation.setInformationComplete(true);
        }
    }

    private boolean validateControl() {
        if (databaseCombo.getText() == null
                || databaseCombo.getText().equals("")) { //$NON-NLS-1$
            parentPage
                    .setErrorMessage(MessageFormat
                            .format(
                                    NewConnectionWizard
                                            .getResourceString("CUI_NEWCW_VALIDATE_DATABASE_REQ_UI_"), //$NON-NLS-1$
                                    new String[] { NewConnectionWizard
                                            .getResourceString("CUI_NEWCW_DEFDBNAME_VAL_UI_") })); //$NON-NLS-1$
            return false;
        } else if (hostCombo != null && hostCombo.isEnabled()
                && hostCombo.getText().length() < 1) {
            parentPage.setErrorMessage(NewConnectionWizard
                    .getResourceString("CUI_NEWCW_VALIDATE_HOST_REQ_UI_")); //$NON-NLS-1$
            return false;
        } else if (portCombo != null && portCombo.isEnabled()
                && portCombo.getText().length() < 1) {
            parentPage.setErrorMessage(NewConnectionWizard
                    .getResourceString("CUI_NEWCW_VALIDATE_PORT_REQ_UI_")); //$NON-NLS-1$
            return false;
        } else if (driverDisplay != null && driverDisplay.getEnabled()
                && driverDisplay.getText().trim().length() < 1) {
            parentPage
                    .setErrorMessage(NewConnectionWizard
                            .getResourceString("CUI_NEWCW_VALIDATE_DRIVERCLASS_REQ_UI_")); //$NON-NLS-1$
            return false;
        } else if (locationCombo != null
                && locationCombo.getText().trim().length() < 1) {
            parentPage
                    .setErrorMessage(NewConnectionWizard
                            .getResourceString("CUI_NEWCW_VALIDATE_CLASSLOCATION_REQ_UI_")); //$NON-NLS-1$
            return false;
        } else if (urlDisplay != null
                && urlDisplay.getText().trim().length() == 0) {
            parentPage.setErrorMessage(NewConnectionWizard
                    .getResourceString("CUI_NEWCW_VALIDATE_URL_REQ_UI_")); //$NON-NLS-1$
            return false;
        }
        return true;
    }

    private void addListeners() {
        locationCombo.addListener(SWT.Modify, this);
        browseLocation.addListener(SWT.Selection, this);
        databaseCombo.addListener(SWT.Modify, this);
        createCheck.addListener(SWT.Selection, this);
        hostCombo.addListener(SWT.Modify, this);
        portCombo.addListener(SWT.Modify, this);
        urlDisplay.addListener(SWT.Modify, this);
    }

    private void initialize() {
        driverDisplay.setText(DRIVER_CLASS_NAME);
        this.connectionInformation.setDriverClassName(DRIVER_CLASS_NAME);

        // Set default values
        databaseCombo.setText("sample"); //$NON-NLS-1$
        hostCombo.setText("localhost"); //$NON-NLS-1$
        portCombo.setText("1527"); //$NON-NLS-1$

        updateURL();
        addListeners();

        determinePageCompletion();
    }

    private void updateURL() {
        String url = "jdbc:derby:net://" + hostCombo.getText() + ":" //$NON-NLS-1$ //$NON-NLS-2$
                + portCombo.getText() + "/" //$NON-NLS-1$
                + encodeDatabaseName(databaseCombo.getText())
                + RETURN_MESSAGES;
        urlDisplay.setText(url);
        connectionInformation.setURL(url);
    }

    private String encodeDatabaseName(String name) {
        String processedName = name;
        processedName += !createCheck.getSelection() ? "" //$NON-NLS-1$
                : (";" + CREATE_EQUALS_TRUE_TEXT); //$NON-NLS-1$
        if ((name.indexOf('/') > -1) || (name.indexOf('\\') > -1)) {
            processedName = "\"" + processedName + "\""; //$NON-NLS-1$ //$NON-NLS-2$
        }
        return processedName;
    }

    private String decodeDatabaseName(String name) {
        String processedName = name;
        if (name.indexOf('"') > -1) {
            String remainingName = name.substring(name.indexOf('"') + 1);
            processedName = remainingName.substring(0, remainingName
                    .indexOf('"'));
        } else {
            if (name.indexOf(':') > -1) {
                processedName = name.substring(0, name.indexOf(':'));
            }
        }
        return processedName;
    }

    private void removeListeners() {
        locationCombo.removeListener(SWT.Modify, this);
        browseLocation.removeListener(SWT.Selection, this);
        databaseCombo.removeListener(SWT.Modify, this);
        createCheck.removeListener(SWT.Selection, this);
        hostCombo.removeListener(SWT.Modify, this);
        portCombo.removeListener(SWT.Modify, this);
        urlDisplay.removeListener(SWT.Modify, this);
    }

    private Composite createServerConnectionUI(Composite parent) {
        parentComposite = factory.createComposite(parent, SWT.NULL);
        createLayout(parentComposite);

        GridData gd;

        databaseLabel = factory.createLabel(parentComposite, SWT.NONE);
        databaseLabel.setText(CUI_NEWCW_DATABASE_LBL_UI_);
        gd = new GridData();
        gd.verticalAlignment = GridData.BEGINNING;
        databaseLabel.setLayoutData(gd);

        databaseCombo = factory.createText(parentComposite, SWT.SINGLE
                | SWT.BORDER);
        gd = new GridData();
        gd.verticalAlignment = GridData.BEGINNING;
        gd.horizontalAlignment = GridData.FILL;
        gd.horizontalSpan = 2;
        databaseCombo.setLayoutData(gd);

        createCheck = factory.createButton(parentComposite, SWT.CHECK);
        createCheck.setText(CUI_NEWCW_CREATEDB_BTN_UI_);
        createCheck.setSelection(true);
        gd = new GridData();
        gd.verticalAlignment = GridData.BEGINNING;
        gd.horizontalSpan = 3;
        createCheck.setLayoutData(gd);

        hostLabel = factory.createLabel(parentComposite, SWT.NONE);
        hostLabel.setText(CUI_NEWCW_HOST_LBL_UI_);
        gd = new GridData();
        gd.verticalAlignment = GridData.BEGINNING;
        gd.horizontalSpan = 1;
        hostLabel.setLayoutData(gd);

        hostCombo = factory
                .createText(parentComposite, SWT.SINGLE | SWT.BORDER);
        gd = new GridData();
        gd.horizontalAlignment = GridData.FILL;
        gd.verticalAlignment = GridData.BEGINNING;
        gd.horizontalSpan = 2;
        gd.grabExcessHorizontalSpace = true;
        hostCombo.setLayoutData(gd);

        portLabel = factory.createLabel(parentComposite, SWT.NONE);
        portLabel.setText(CUI_NEWCW_PORT_LBL_UI_);
        gd = new GridData();
        gd.verticalAlignment = GridData.BEGINNING;
        gd.horizontalSpan = 1;
        portLabel.setLayoutData(gd);

        portCombo = factory
                .createText(parentComposite, SWT.SINGLE | SWT.BORDER);
        gd = new GridData();
        gd.horizontalAlignment = GridData.FILL;
        gd.verticalAlignment = GridData.BEGINNING;
        gd.grabExcessHorizontalSpace = true;
        gd.horizontalSpan = 2;
        portCombo.setLayoutData(gd);

        driverLabel = factory.createLabel(parentComposite, SWT.NONE);
        driverLabel.setText(CUI_NEWCW_JDBCDRIVERCLS_LBL_UI_);
        gd = new GridData();
        gd.verticalAlignment = GridData.BEGINNING;
        gd.horizontalSpan = 1;
        driverLabel.setLayoutData(gd);

        driverDisplay = factory.createText(parentComposite, SWT.SINGLE
                | SWT.BORDER | SWT.READ_ONLY);
        gd = new GridData();
        gd.horizontalAlignment = GridData.FILL;
        gd.verticalAlignment = GridData.BEGINNING;
        gd.grabExcessHorizontalSpace = true;
        gd.horizontalSpan = 2;
        driverDisplay.setLayoutData(gd);

        locationLabel = factory.createLabel(parentComposite, SWT.NONE);
        locationLabel.setText(CUI_NEWCW_JDBCCLSLOCATION_LBL_UI_);
        gd = new GridData();
        gd.verticalAlignment = GridData.BEGINNING;
        gd.horizontalSpan = 1;
        locationLabel.setLayoutData(gd);

        locationCombo = factory.createText(parentComposite, SWT.SINGLE
                | SWT.BORDER);
        gd = new GridData();
        gd.widthHint = 200;
        gd.horizontalAlignment = GridData.FILL;
        gd.grabExcessHorizontalSpace = true;
        gd.verticalAlignment = GridData.BEGINNING;
        gd.grabExcessHorizontalSpace = true;
        gd.horizontalSpan = 1;
        locationCombo.setLayoutData(gd);

        browseLocation = factory.createButton(parentComposite, SWT.PUSH);
        browseLocation.setText(CUI_NEWCW_JARBROWSE_BTN_UI_);
        gd = new GridData();
        gd.verticalAlignment = GridData.BEGINNING;
        gd.horizontalAlignment = GridData.FILL;
        gd.grabExcessHorizontalSpace = true;
        gd.horizontalSpan = 1;
        browseLocation.setLayoutData(gd);

        urlLabel = factory.createLabel(parentComposite, SWT.NONE);
        urlLabel.setText(CUI_NEWCW_CONNECTIONURL_LBL_UI_);
        gd = new GridData();
        gd.verticalAlignment = GridData.BEGINNING;
        gd.horizontalSpan = 1;
        urlLabel.setLayoutData(gd);

        urlDisplay = factory.createText(parentComposite, SWT.SINGLE
                | SWT.BORDER | SWT.READ_ONLY);
        gd = new GridData();
        gd.horizontalAlignment = GridData.FILL;
        gd.verticalAlignment = GridData.BEGINNING;
        gd.grabExcessHorizontalSpace = true;
        gd.horizontalSpan = 2;
        gd.widthHint = 200;
        urlDisplay.setLayoutData(gd);

        initialize();

        return parentComposite;
    }

    public Composite getServerConnectionUI(Composite parent) {
        if (parentComposite == null || parentComposite.isDisposed()) {
            parentComposite = createServerConnectionUI(parent);
        }
        setConnectionInformation();
        return parentComposite;
    }

    public void setServerConnectionInformation(
            IServerConnectionInformation connectionInformation) {
        this.connectionInformation = connectionInformation;
    }

    public void setDatabaseDefinition(DatabaseDefinition definition) {
    }

    public void setWizardPage(WizardPage parentPage) {
        this.parentPage = parentPage;
    }

    public void handleEvent(Event event) {
        Widget source = event.widget;
        if (source == databaseCombo) {
            updateURL();
        } else if (source == createCheck) {
            updateURL();
        } else if (source == locationCombo) {
        } else if (source == this.hostCombo) {
            updateURL();
        } else if (source == this.portCombo) {
            updateURL();
        } else if (source == browseLocation) {
            String file = NewConnectionWizard.getFile((Wizard) parentPage
                    .getWizard(), NewConnectionWizard.fExtensions);
            if (file != null && file.length() > 0) {
                locationCombo.removeListener(SWT.Modify, this);
                locationCombo.setText(file);
                locationCombo.addListener(SWT.Modify, this);

                this.connectionInformation.setLoadingPath(file);
            }
        }
        setConnectionInformation();
    }

    public void loadProperties() {
        removeListeners();
        String rawURL = connectionInformation.getURL();
		if (!(rawURL.indexOf(CREATE_EQUALS_TRUE_TEXT) > -1)) {
			createCheck.setSelection(false);
		} else {
		    rawURL = rawURL.replaceFirst(";" + CREATE_EQUALS_TRUE_TEXT, ""); //$NON-NLS-1$ //$NON-NLS-2$
		    createCheck.setSelection(true);
		}

        DerbyUniversalJDBCURL url = new DerbyUniversalJDBCURL(
                rawURL);
        locationCombo.setText(connectionInformation.getLoadingPath());
        hostCombo.setText(url.getNode());
        portCombo.setText(url.getPort());
        databaseCombo.setText(url.getDatabaseName());
        updateURL();
        addListeners();
        setConnectionInformation();
    }

    private class DerbyUniversalJDBCURL {
        private String subprotocol = ""; //$NON-NLS-1$

        private String node = ""; //$NON-NLS-1$

        private String port = ""; //$NON-NLS-1$

        private String databaseName = ""; //$NON-NLS-1$

        private String properties = ""; //$NON-NLS-1$

        /**
         * @param url
         */
        public DerbyUniversalJDBCURL(String url) {
            parseURL(url);
        }

        /**
         * @return Returns the databaseName.
         */
        public String getDatabaseName() {
            return databaseName;
        }

        /**
         * @return Returns the node.
         */
        public String getNode() {
            return node;
        }

        /**
         * @return Returns the subprotocol.
         */
        public String getSubprotocol() {
            return subprotocol;
        }

        private void parseURL(String url) {
            try {
                String remainingURL = url.substring(url.indexOf(':') + 1);
                this.subprotocol = remainingURL.substring(0, remainingURL
                        .indexOf(':'));
                remainingURL = remainingURL
                        .substring(remainingURL.indexOf(':') + 1);
                this.subprotocol = this.subprotocol
                        + remainingURL.substring(0, remainingURL.indexOf(':'));
                remainingURL = remainingURL
                        .substring(remainingURL.indexOf(':') + 3);
                this.node = remainingURL
                        .substring(0, remainingURL.indexOf('/'));
                if (this.node.indexOf(':') > -1) {
                    this.port = this.node.substring(this.node.indexOf(':') + 1);
                    this.node = this.node.substring(0, this.node.indexOf(':'));
                }
                remainingURL = remainingURL
                        .substring(remainingURL.indexOf('/') + 1);
                this.databaseName = decodeDatabaseName(remainingURL);
            } catch (Exception e) {
            }
        }

        /**
         * @return Returns the port.
         */
        public String getPort() {
            return port;
        }

        /**
         * @return Returns the properties.
         */
        public String getProperties() {
            return properties;
        }
    }
}