/*******************************************************************************
 * Copyright (c) 2001, 2004 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.wst.rdb.internal.derby.catalog;

import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;

import org.eclipse.core.runtime.Status;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.wst.rdb.internal.core.RDBCorePlugin;
import org.eclipse.wst.rdb.internal.core.definition.DataModelElementFactory;
import org.eclipse.wst.rdb.internal.core.definition.DatabaseDefinition;
import org.eclipse.wst.rdb.internal.core.rte.ICatalogObject;
import org.eclipse.wst.rdb.internal.core.rte.RefreshManager;
import org.eclipse.wst.rdb.internal.derby.DerbyPlugin;
import org.eclipse.wst.rdb.internal.models.dbdefinition.PredefinedDataTypeDefinition;
import org.eclipse.wst.rdb.internal.models.sql.datatypes.DataType;
import org.eclipse.wst.rdb.internal.models.sql.routines.DataAccess;
import org.eclipse.wst.rdb.internal.models.sql.routines.ParameterMode;
import org.eclipse.wst.rdb.internal.models.sql.routines.RoutineResultTable;
import org.eclipse.wst.rdb.internal.models.sql.routines.SQLRoutinesPackage;
import org.eclipse.wst.rdb.internal.models.sql.routines.Source;
import org.eclipse.wst.rdb.internal.models.sql.routines.impl.UserDefinedFunctionImpl;
import org.eclipse.wst.rdb.internal.models.sql.schema.Database;

public class DerbyCatalogUserDefinedFunction extends UserDefinedFunctionImpl implements ICatalogObject {
	private static final long serialVersionUID = 3905244528465163826L;


	public void refresh() {
		if (this.parameterLoaded){
			this.parameters.clear();
			this.parameterLoaded = false;
		}
		
		this.loaded = false;
		
		RefreshManager.getInstance().referesh(this);
	}

	public boolean isSystemObject() {
		return false;
	}

	public Connection getConnection() {
		Database database = this.getCatalogDatabase();
		return ((DerbyCatalogDatabase) database).getConnection();
	}
	
	public Database getCatalogDatabase() {
		return this.getSchema().getDatabase();		
	}
	
	public EList getParameters() {
		if(!this.parameterLoaded) this.loadParameters();
		return this.parameters;
	}
	
	public String getLanguage() {
		if(!this.loaded) this.load();
		return this.language;
	}

	public String getParameterStyle() {
		if(!this.loaded) this.load();
		return this.parameterStyle;
	}
	public DataAccess getSqlDataAccess() {
		if(!this.loaded) this.load();
		return this.sqlDataAccess;
	}
	
	public Source getSource() {
		if(!this.loaded) this.load();
		return super.getSource();
	}
	
	public boolean eIsSet(EStructuralFeature eFeature) {
		int id = eDerivedStructuralFeatureID(eFeature);
		if(id == SQLRoutinesPackage.PROCEDURE__PARAMETERS) {
			this.getParameters();
		}
		else if(id == SQLRoutinesPackage.PROCEDURE__LANGUAGE) {
			this.getLanguage();
		}
		else if(id == SQLRoutinesPackage.PROCEDURE__PARAMETER_STYLE) {
			this.getParameterStyle();
		}
		else if(id == SQLRoutinesPackage.PROCEDURE__SQL_DATA_ACCESS) {
			this.getSqlDataAccess();
		}
		
		return super.eIsSet(eFeature);
	}
	
	private synchronized void loadParameters() {
		if(this.parameterLoaded) return;
		boolean deliver = this.eDeliver();
		this.eSetDeliver(false);	
		
		try {
			DerbyCatalogUserDefinedFunction.loadParameters(this.getConnection(), super.getParameters(), this);
		}
		catch(Exception e) {
			System.out.println(e.toString());
		}
		
		this.parameterLoaded = true;
		this.eSetDeliver(deliver);		
	}
	
	public static void loadParameters(Connection connection, EList parameterList, DerbyCatalogUserDefinedFunction function) throws SQLException {
		final Database database = function.getSchema().getDatabase();
		final DatabaseDefinition databaseDefinition = RDBCorePlugin.getDefault().getDatabaseDefinitionRegistry().getDefinition(database);
		final DataModelElementFactory factory = databaseDefinition.getDataModelElementFactory();
	
		DatabaseMetaData metaData = connection.getMetaData();
		
		ResultSet r = metaData.getProcedureColumns(null, function.getSchema().getName(),function.getName(),null);
		try {
			while(r.next()) {

				boolean isTableFunction = false;
				final String schemaName = r.getString(2);
				final String functionName = r.getString(3);
				
				final String name = r.getString(4);

				final short columntype = r.getShort(5);

				if (columntype == DatabaseMetaData.procedureColumnResult)
					isTableFunction = true;
				
				DataType type=null;
				final String typeName = r.getString(7);
				PredefinedDataTypeDefinition typeDefinition = databaseDefinition.getPredefinedDataTypeDefinition(typeName);
				if(typeDefinition != null) {
					type = databaseDefinition.getPredefinedDataType(typeDefinition);
					if(typeDefinition.isLengthSupported()) {
						final int length = r.getInt(9);
						EStructuralFeature feature = type.eClass().getEStructuralFeature("length");  //$NON-NLS-1$
						type.eSet(feature, new Integer(length));
					}
					else if(typeDefinition.isPrecisionSupported()) {
						final int length = r.getInt(8);
						EStructuralFeature feature = type.eClass().getEStructuralFeature("precision"); //$NON-NLS-1$
						type.eSet(feature, new Integer(length));
					}
					
					if(typeDefinition.isScaleSupported()) {
						final int length = r.getInt(10);
						EStructuralFeature feature = type.eClass().getEStructuralFeature("scale"); //$NON-NLS-1$
						type.eSet(feature, new Integer(length));
					}
					
				}
				else {
				}
				
				
				if (isTableFunction) {
					RoutineResultTable resultTable = function.getReturnTable();
					if (resultTable == null) {
						resultTable  = (RoutineResultTable)factory.create(SQLRoutinesPackage.eINSTANCE.getRoutineResultTable());
						function.setReturnTable(resultTable);
					}
					DerbyCatalogColumn column = new DerbyCatalogColumn();
					column.setName(name);
					column.setDescription(r.getString(13));
					column.setDataType(type);
					resultTable.getColumns().add(column);
					
				} else {
				
					DerbyCatalogParameter parameter = new DerbyCatalogParameter();
					parameter.setName(name);
	
					if (columntype == DatabaseMetaData.procedureColumnInOut){
						parameter.setMode(ParameterMode.INOUT_LITERAL);
					} else if (columntype== DatabaseMetaData.procedureColumnOut) {
						parameter.setMode(ParameterMode.OUT_LITERAL);
					} else if (columntype== DatabaseMetaData.procedureColumnIn) {
						parameter.setMode(ParameterMode.IN_LITERAL);
					}

					parameter.setDescription(r.getString(13));
					parameter.setDataType(type);
					
					if (columntype == DatabaseMetaData.procedureColumnReturn)
						function.setReturnScaler(parameter);
					else
						parameterList.add(parameter);

				}
				
				
			}
			r.close();
			
		}
		catch(Exception e) {
		}
			
	}
	
	private synchronized void load() {
		if(this.loaded) return;
		this.loaded = true;

		boolean deliver = this.eDeliver();
		this.eSetDeliver(false);
		Connection connection = this.getConnection();

		try {
			String sql = "SELECT ALIAS,JAVACLASSNAME,ALIASINFO" + //$NON-NLS-1$
			" FROM SYS.SYSALIASES A,SYS.SYSSCHEMAS B"+ //$NON-NLS-1$
			" WHERE A.SCHEMAID=B.SCHEMAID" + //$NON-NLS-1$
			" AND B.SCHEMANAME = ?  AND A.ALIAS = ?"; //$NON-NLS-1$
			
		PreparedStatement stmt =connection.prepareStatement(sql);
		stmt.setString(1,this.getSchema().getName());
		stmt.setString(2, this.getName());
		ResultSet r = stmt.executeQuery();

			while(r.next()) {
		
				// Cloudscape only currently supports
				// Java SPs.
				this.language = "JAVA"; //$NON-NLS-1$
				this.parameterStyle = "JAVA"; //$NON-NLS-1$

				String javaClassName = r.getString(2);
				Object aliasInfo = r.getObject(3);
				setRestOfMetaData(javaClassName, aliasInfo);
			}
	
			r.close();
			stmt.close();
		}
		catch (Exception e) {
			Status status = new Status(
					Status.ERROR,
					DerbyPlugin.getDefault().getBundle().getSymbolicName(),
					Status.ERROR,
					"###Error..org.eclipse.wst.rdb.internal.derby.catalog.DerbyCatalogProcedure.setRestOfMetaData", //$NON-NLS-1$
					e);
			DerbyPlugin.getDefault().getLog().log(status);
		}
		this.eSetDeliver(deliver);
	}
	protected void setRestOfMetaData(String javaClassName, Object aliasInfo) {
		try {

			// Set SQL Data Access
			String aliasInfoString = aliasInfo.toString();
			this.loadSource(aliasInfoString);
			setSqlDataAccess(aliasInfoString);
			
			//	Set External name
			int index = aliasInfoString.indexOf("("); //$NON-NLS-1$
			String methodName = aliasInfoString.substring(0,index);
			this.setExternalName(javaClassName+"."+methodName); //$NON-NLS-1$
			
		} catch (Exception e) {
			Status status = new Status(
					Status.ERROR,
					DerbyPlugin.getDefault().getBundle().getSymbolicName(),
					Status.ERROR,
					"###Error..org.eclipse.wst.rdb.internal.derby.catalog.DerbyCatalogProcedure.load", //$NON-NLS-1$
					e);
			DerbyPlugin.getDefault().getLog().log(status);
		}
	}
	

	protected void setSqlDataAccess(String aliasInfo) {
		int index = aliasInfo.indexOf("NO SQL"); //$NON-NLS-1$
		if (index != -1) {
			this.setSqlDataAccess(DataAccess.NO_SQL_LITERAL);
		} else {
			index = aliasInfo.indexOf("MODIFIES SQL DATA"); //$NON-NLS-1$
			if (index != -1) {
				this.setSqlDataAccess(DataAccess.MODIFIES_SQL_DATA_LITERAL);

			} else {
				index = aliasInfo.indexOf("CONTAINS SQL"); //$NON-NLS-1$
				if (index != -1) {
					this.setSqlDataAccess(DataAccess.CONTAINS_SQL_LITERAL);

				} else {
					index = aliasInfo.indexOf("READS SQL DATA"); //$NON-NLS-1$
					if (index != -1) {
						this.setSqlDataAccess(DataAccess.READS_SQL_DATA_LITERAL);
					}
				}
			}
		}
	}
	
	private void loadSource(String aliasInfo) {
		int index = aliasInfo.indexOf("("); //$NON-NLS-1$
		String body = aliasInfo.substring(index);
		body = body.replaceAll("IN ", "");
        final DatabaseDefinition definition = RDBCorePlugin.getDefault().getDatabaseDefinitionRegistry().getDefinition(this.getCatalogDatabase());
		final DataModelElementFactory factory = definition.getDataModelElementFactory();

        Source s = (Source) factory.create(SQLRoutinesPackage.eINSTANCE.getSource());   
        s.setBody(body);
        this.setSource(s);
	}

	
	private boolean parameterLoaded = false;
	private boolean loaded = false;
}
