/*******************************************************************************
 * Copyright (c) 2000, 2004 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 * IBM Corporation - initial API and implementation
 ******************************************************************************/

package org.eclipse.wst.internet.internal.proxy;

import java.text.MessageFormat;
import java.util.Properties;

import org.eclipse.core.runtime.Platform;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.ui.IStartup;
import org.eclipse.ui.plugin.AbstractUIPlugin;
import org.osgi.framework.BundleContext;

/**
* This is the plugin class for the org.eclipse.wst.internet.internal.proxy plugin.
* This plugin contains the "Internet" preferences page.
*/
public class InternetPlugin extends AbstractUIPlugin implements IStartup
{
  /**
  * The identifier of the descriptor of this plugin in plugin.xml.
  */
  public static final String ID = "org.eclipse.wst.internet.proxy";

  /**
   * The instance of this plugin.
   */
  private static InternetPlugin instance;

  /**
   * Constructor for use by the Eclipse platform only.
   */
  public InternetPlugin()
  {
    super();
    instance = this;
  }

  /**
   * Returns the instance of this plugin.
   */
  static public InternetPlugin getInstance ()
  {
    return instance;
  }

  public void start(BundleContext context) throws Exception
  {
      super.start(context);
      updateProxyProperties();
  }
  
  public void earlyStartup()
  {
    updateProxyProperties();
  }

  /**
  * Returns the message string identified by the given key from
  * the plugin.properties file for the appropriate locale.
  * @param key The message key string prefixed by a "%" symbol.
  * That is, the string passed in must be of the form "%KEY"
  * where the plugin.properties file contains a line of the
  * form: "KEY = value".
  * @return The locale-specific message.
  */
  public static String getMessage ( String key )
  {
    return Platform.getResourceString(instance.getBundle(),key);
  }

  /**
  * Returns the message string identified by the given key from
  * the plugin.properties file for the appropriate locale.
  * Substitution sequences in the message string
  * are replaced by the given array of substitution objects (which
  * are most frequently strings). See java.text.MessageFormat for
  * further details on substitution.
  * @param key The message key string prefixed by a "%" symbol.
  * That is, the string passed in must be of the form "%KEY"
  * where the plugin.properties file contains a line of the
  * form: "KEY = value".
  * @param args The substitution values for the message
  * as required by the message in plugin.properties and
  * by the rules of class java.text.MessageFormat.
  * @return The locale-specific message.
  */
  public static String getMessage ( String key, Object[] args )
  {
    return MessageFormat.format(getMessage(key),args);
  }

  /**
    * Sets the proxy properties to the values in the preferences store.
  **/
  public void updateProxyProperties ()
  {
    // Set the JVM system proxy properties depending on the values in the system store.
    IPreferenceStore store = getPreferenceStore();
    boolean setProxy = store.getBoolean( PREFERENCE_PROXYCHECKED  );
    boolean useSOCKS = store.getBoolean( PREFERENCE_SOCKSCHECKED );
    boolean setName = store.getBoolean( PREFERENCE_NAMECHECKED  );
    Properties sysProps = System.getProperties();      
    Log.write( this, "updateProxyProperties", Log.INFO, "proxySet: " + setProxy );

    if (setProxy)
    {
      String hostname      = store.getString( PREFERENCE_HOSTNAME );
      String port          = store.getString( PREFERENCE_PORT );
      String nonProxyHosts = store.getString( PREFERENCE_NON_PROXY_HOSTS );
      
      sysProps.put( "proxySet", "true" );
      
      if (useSOCKS)
      {
        sysProps.put( "socksProxyHost", hostname );
        sysProps.put( "socksProxyPort", port );
        
        sysProps.remove( "proxyHost" );
        sysProps.remove( "proxyPort" );
        
        sysProps.remove( "http.proxySet" );
        sysProps.remove( "http.proxyHost" );
        sysProps.remove( "http.proxyPort" );
        sysProps.remove( "http.nonProxyHosts" );
      }
      else
      {
        sysProps.remove( "socksProxyHost" );
        sysProps.remove( "socksProxyPort" );
        
        sysProps.put( "proxyHost", hostname ); 
        sysProps.put( "proxyPort", port );
 
        sysProps.put( "http.proxySet", "true" );
        sysProps.put( "http.proxyHost", hostname );
        sysProps.put( "http.proxyPort", port );
        sysProps.put( "http.nonProxyHosts", nonProxyHosts );
      }
      
      // Authentication is currently supported only for HTTP proxies. The UI will enforce this restriction.
      if(setName)
      {
	      String userName = store.getString( PREFERENCE_USERNAME );
	      String password = store.getString( PREFERENCE_PASSWORD );
	      
	      sysProps.put( "http.proxyUserName", Encoder.isEncoded(userName)?Encoder.decode(userName):userName);
	      sysProps.put( "http.proxyPassword", Encoder.isEncoded(password)?Encoder.decode(password):password);
	      	      
	      Log.write( this, "updateProxyProperties", Log.INFO, "http.proxyUserName: " + userName);
	      Log.write( this, "updateProxyProperties", Log.INFO, "http.proxyPassword: " + password);	      
      }
      else
      {
	      Log.write( this, "updateProxyProperties", Log.INFO, "Removing proxy name properties" );
	
	      sysProps.remove( "http.proxyUserName" );
	      sysProps.remove( "http.proxyPassword" );	      	
      }
	
      Log.write( this, "updateProxyProperties", Log.INFO, "proxyHost: " + hostname );
      Log.write( this, "updateProxyProperties", Log.INFO, "proxyPort: " + port );
      Log.write( this, "updateProxyProperties", Log.INFO, "http.proxyHost: " + hostname);
      Log.write( this, "updateProxyProperties", Log.INFO, "http.proxyPort: " + port);     
      Log.write( this, "updateProxyProperties", Log.INFO, "http.nonProxyHosts: " + nonProxyHosts );     
    }
    else
    {
      Log.write( this, "updateProxyProperties", Log.INFO, "Removing proxy properties" );

      sysProps.remove( "proxySet" ); 
      sysProps.remove( "proxyHost" ); 
      sysProps.remove( "proxyPort" );
      
      sysProps.remove( "socksProxyHost" );
      sysProps.remove( "socksProxyPort" );
 
      sysProps.remove( "http.proxySet" );
      sysProps.remove( "http.proxyHost" );
      sysProps.remove( "http.proxyPort" );
      sysProps.remove( "http.proxyUserName" );
      sysProps.remove( "http.proxyPassword" );
      sysProps.remove( "http.nonProxyHosts" );
    }
  }

  /**
  * This constant string is used to lookup the checkBox internet preference from
  * the plugins local preferences store.
  */
  public static final String PREFERENCE_PROXYCHECKED = "proxyCheckBox";

  public static final String PREFERENCE_NAMECHECKED = "nameCheckBox";
  
  public static final String PREFERENCE_SOCKSCHECKED = "socksCheckBox";

  /**
  * This constant string is used to lookup the hostname internet preference from
  * the plugins local preferences store.
  */
  public static final String PREFERENCE_HOSTNAME = "hostnameText";

  /**
  * This constant string is used to lookup the port internet preference from
  * the plugins local preferences store.
  */
  public static final String PREFERENCE_PORT = "portText";

  /**
  * This constant string is used to lookup the userName internet preference from
  * the plugins local preferences store.
  */
  public static final String PREFERENCE_USERNAME = "userNameText";

  /**
  * This constant string is used to lookup the password internet preference from
  * the plugins local preferences store.
  */
  public static final String PREFERENCE_PASSWORD = "passwordText";
  
  /**
   * This constant string is used to lookup the non proxy hosts internet preference from
   * the plugins local preferences store.
   */
   public static final String PREFERENCE_NON_PROXY_HOSTS = "nonProxyHostsText";

}
