/*******************************************************************************
 * Copyright (c) 2001, 2004 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.wst.rdb.internal.core.connection;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.Driver;
import java.sql.DriverPropertyInfo;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.Map;
import java.util.Properties;
import java.util.StringTokenizer;
import java.util.Vector;

import org.eclipse.core.runtime.IPath;
import org.eclipse.emf.ecore.EAnnotation;
import org.eclipse.wst.rdb.internal.core.RDBCorePlugin;
import org.eclipse.wst.rdb.internal.core.definition.DatabaseDefinition;
import org.eclipse.wst.rdb.internal.models.sql.schema.SQLObject;


public class ConnectionManagerImpl implements ConnectionManager {
	private static final String CONNECTION_URI = "ConnectionURI"; //$NON-NLS-1$
	private static final String CONNECTION_URI_KEY = "ConnectionKey"; //$NON-NLS-1$
	
	public static final String CONNECTION = "connection/"; //$NON-NLS-1$
	public static final String PRODUCT = "Product"; //$NON-NLS-1$
	public static final String VERSION = "Version"; //$NON-NLS-1$
	public static final String LOADING_PATH = "LoadingPath"; //$NON-NLS-1$
	public static final String DRIVER_CLASS_NAME = "DriverClassName"; //$NON-NLS-1$
	public static final String URL = "URL"; //$NON-NLS-1$
	public static final String INFO_FILE_EXTENSION = "info"; //$NON-NLS-1$
	public static final String DATABASE_NAME = "DatabaseNameInternal"; //$NON-NLS-1$
	public static final String IDENTIFIER_QUOTE_STRING = "IdentiferQuoteString"; //$NON-NLS-1$
    public static final String DATABASE_PRODUCT_VERSION = "DatabaseProductVersion"; //$NON-NLS-1$
	public static final String FILTER = "filter"; //$NON-NLS-1$
	public static final String CUSTOM_PROPERTY_PREFIX = "@@_CUSTOMPROPERTY_"; //$NON-NLS-1$
	public static final ConnectionManagerImpl INSTANCE = new ConnectionManagerImpl();
	
	private ConnectionManagerImpl() {
	}
	
	public ConnectionInfo getConnectionInfo(DatabaseDefinition def) {
		if(def == null) throw new NullPointerException();
		if(this.infoMap.containsKey(def)) {
			return (ConnectionInfo) this.infoMap.get(def);
		}
		return this.loadConnectionInfo(def);
	}

	public ConnectionInfo getConnectionInfo(String name) {
		if(name == null) throw new NullPointerException();
		if(name.trim().equals("")) throw new IllegalArgumentException(); //$NON-NLS-1$
		
		String nameKey = name.toUpperCase();
		if(this.infoMap.containsKey(nameKey)) {
			return (ConnectionInfo) this.infoMap.get(nameKey);
		}
		return this.loadConnectionInfo(name);
	}

	public ConnectionInfo createConnectionInfo(DatabaseDefinition def, String name) {
		if(name == null) throw new NullPointerException();
		if(name.trim().equals("")) throw new IllegalArgumentException(); //$NON-NLS-1$
		if(def == null) throw new NullPointerException();

		String nameKey = name.toUpperCase();
		if(this.infoMap.containsKey(nameKey)) throw new IllegalArgumentException();
		if(this.loadConnectionInfo(name) != null)  throw new IllegalArgumentException();
		ConnectionInfo info = new ConnectionInfoImpl(def, name);
		this.infoMap.put(nameKey, info);
		
		Collection c = new LinkedList();
		c.addAll(this.listeners);
		Iterator it = c.iterator();
		while(it.hasNext()) {
			ConnectionManagerListener l = (ConnectionManagerListener) it.next();
			try {
				l.connectionInfoCreated(info);
			}
			catch(Throwable o) {
				this.removeListener(l);
			}
		}				
		
		return info;
	}

	public ConnectionInfo[] getAllNamedConnectionInfo() {
		IPath path = RDBCorePlugin.getDefault().getStateLocation();
		path = path.append(CONNECTION);
		File dir = path.toFile();

		if(dir.exists()) {
			File[] files = dir.listFiles();
			for(int i=0; i<files.length; ++i) {
				if(files[i].isDirectory()) {
					String n = files[i].getName();
					String nKey = n.toUpperCase();					
					if(!this.infoMap.containsKey(nKey)) {
						this.loadConnectionInfo(n);
					}
				}
			}
		}
		Vector v = new Vector();
		Iterator it = this.infoMap.keySet().iterator();
		while(it.hasNext()) {
			Object key = it.next();
			if(key instanceof String) {
				v.add(infoMap.get(key));
			}
		}
		
		ConnectionInfo[] namedInfo = new ConnectionInfo[v.size()];
		v.copyInto(namedInfo);
		return namedInfo;
	}

	public void removeConnectionInfo(String name) {
		if(name == null) throw new NullPointerException();
		if(name.trim().equals("")) throw new IllegalArgumentException(); //$NON-NLS-1$
		
		boolean removed = false;
		String nameKey = name.toUpperCase();
		if(this.infoMap.containsKey(nameKey)) {
			this.infoMap.remove(nameKey);
			removed = true;
		}
		
		IPath path = RDBCorePlugin.getDefault().getStateLocation();
		IPath infoPath = path.append(CONNECTION + name);
		File dir = infoPath.toFile();
		if(dir.exists()) {
			File[] files = dir.listFiles();
			for(int i=0; i<files.length; ++i) files[i].delete();
			dir.delete();
			removed = true;
		}
		
		if(removed) {
			Collection c = new LinkedList();
			c.addAll(this.listeners);
			Iterator it = c.iterator();
			while(it.hasNext()) {
				ConnectionManagerListener l = (ConnectionManagerListener) it.next();
				try {
					l.connectionInfoRemoved(name);
				}
				catch(Throwable o) {
					this.removeListener(l);
				}
			}
		}
		else {
			throw new IllegalArgumentException();
		}
	}
	
	public Connection connect(ConnectionInfo info) throws FileNotFoundException, ClassNotFoundException, IllegalAccessException, InstantiationException, SQLException {
		Driver driver = this.getDriver(info);
		Connection cnn = driver.connect(info.getURL(), info.getProperties());
		if(cnn == null) return null;
		Connection connection = new ConnectionAdapter(info, cnn);
		
		if(info.getIdentifierQuoteString() == null) {
		    try {
		        ((ConnectionInfoImpl) info).setIdentifierQuoteString(cnn.getMetaData().getIdentifierQuoteString());
		    }
		    catch(Throwable o) {
		    }
		}
        if (info.getDatabaseProductVersion() == null) {
           try {
            ((ConnectionInfoImpl)info).setDatabaseProductVersion(cnn.getMetaData()
                     .getDatabaseProductVersion());
           }
           catch (Throwable o) {
           }
        }
		
		Collection c = new LinkedList();
		c.addAll(this.listeners);
		Iterator it = c.iterator();
		while(it.hasNext()) {
			ConnectionManagerListener l = (ConnectionManagerListener) it.next();
			try {
				l.connected(info, connection);
			}
			catch(Throwable o) {
				this.removeListener(l);
			}
		}
		
		return connection;
	}
	
	public String getConnectionEAnnotationUri() {
        return CONNECTION_URI;
    }
	
	public String getConnectionEAnnotationKey () {
	    return CONNECTION_URI_KEY;
	}
	
	public String[] getSchemaNames(Connection connection) throws SQLException {
		Vector schemas = new Vector();
		String catalog = null;
		DatabaseMetaData meta = connection.getMetaData();
		if(meta.supportsCatalogsInTableDefinitions()) {
			catalog = connection.getCatalog();
		}
		ResultSet resultSet = meta.getSchemas();
		while(resultSet.next()) {
			String schemaName = resultSet.getString("TABLE_SCHEM"); //$NON-NLS-1$
			schemas.add(schemaName);
		}
		resultSet.close();
		
		String[] result = new String[schemas.size()];
		schemas.copyInto(result);
		
		return result;
	}
	
	
	
	
	public void setConnectionInfo(SQLObject obj, String infoName) {
		if(infoName == null) throw new NullPointerException();
		EAnnotation annotation = obj.getEAnnotation(getConnectionEAnnotationUri());
		if(annotation == null) {
			annotation = obj.addEAnnotation(getConnectionEAnnotationUri());
		}
		
        obj.addEAnnotationDetail(annotation, getConnectionEAnnotationKey(), infoName);
	}

	public void setConnectionInfo(SQLObject obj, ConnectionInfo info) {
		setConnectionInfo(obj, info.getName());
	}

	public String getConnectionInfoName(SQLObject obj) {
		EAnnotation annotation = obj.getEAnnotation(getConnectionEAnnotationUri());
		if(annotation == null) return null;
        return obj.getEAnnotationDetail(annotation, getConnectionEAnnotationKey());
		
	}

	public ConnectionInfo getConnectionInfo(SQLObject obj) {
		String name = getConnectionInfoName(obj);
		if(name == null) return null;
		return getConnectionInfo(name);
	}

	public DriverPropertyInfo[] getDriverPropertyInfo(ConnectionInfo info) throws FileNotFoundException, SQLException, ClassNotFoundException, IllegalAccessException, InstantiationException {
		Driver driver = this.getDriver(info);
		return driver.getPropertyInfo(info.getURL(), info.getProperties());
	}
	
	public boolean addListener(ConnectionManagerListener listener) {
		if(listener == null) throw new NullPointerException();
		Iterator it = this.listeners.iterator();
		while(it.hasNext()) {
			if(listener == it.next()) return false;
		}

		listeners.add(listener);
		return true;
	}

	public boolean removeListener(ConnectionManagerListener listener) {
		if(listener == null) throw new NullPointerException();
		Iterator it = this.listeners.iterator();
		while(it.hasNext()) {
			if(listener == it.next()) {
				it.remove();
				return true;
			}
		}
		return false;
	}

	void rename(String oldName, String newName) {
		String oldNameKey = oldName.toUpperCase();
		String newNameKey = newName.toUpperCase();
		Object info = this.infoMap.get(oldNameKey);
		if(info == null) throw new IllegalStateException();
		this.infoMap.remove(oldNameKey);
		this.infoMap.put(newNameKey, info);
		
		IPath path = RDBCorePlugin.getDefault().getStateLocation();
		IPath oldPath = path.append(CONNECTION + oldName);
		IPath newPath = path.append(CONNECTION + newName);
		File oldFile = oldPath.toFile();
		File newFile = newPath.toFile();

		if(!oldFile.exists()) oldFile.renameTo(newFile);
		
		Collection c = new LinkedList();
		c.addAll(this.listeners);
		Iterator it = c.iterator();
		while(it.hasNext()) {
			ConnectionManagerListener l = (ConnectionManagerListener) it.next();
			try {
				l.connectionInfoRenamed(oldName, (ConnectionInfo) info);
			}
			catch(Throwable o) {
				this.removeListener(l);
			}
		}					
	}
	
	void disconnect(ConnectionInfo info, Connection connection) {
		Collection c = new LinkedList();
		c.addAll(this.listeners);
		Iterator it = c.iterator();
		while(it.hasNext()) {
			ConnectionManagerListener l = (ConnectionManagerListener) it.next();
			try {
				l.disconnected(info, connection);
			}
			catch(Throwable o) {
				this.removeListener(l);
			}
		}				
	}

	private Driver getDriver(ConnectionInfo info) throws FileNotFoundException, ClassNotFoundException, IllegalAccessException, InstantiationException {
		return this.getDriver(info.getLoadingPath(), info.getDriverClassName());
	}

	private Driver getDriver(String loadingPath, String driverClassName) throws FileNotFoundException, ClassNotFoundException, IllegalAccessException, InstantiationException {
		URL[] urls = this.getURLs(loadingPath);
		String key = this.getLoaderKey(urls);
		if(!this.loaderMap.containsKey(key)) {
			ClassLoader l = new URLClassLoader(urls, ClassLoader.getSystemClassLoader());
			this.loaderMap.put(key, l);
		}
		ClassLoader loader = (ClassLoader) this.loaderMap.get(key);
		return (Driver) loader.loadClass(driverClassName).newInstance();
	}

	private URL[] getURLs(String loadingPath) throws FileNotFoundException {
		if(loadingPath == null) return new URL[0];
		Collection urls = new ArrayList(3);
		StringTokenizer st = new StringTokenizer(loadingPath, java.io.File.pathSeparator);
		while(st.hasMoreTokens()) {
			String fileName = st.nextToken();
			File file = new File(fileName);
			if(file.exists()) {
				try {
					URL url = file.toURL();
					urls.add(url);
				}
				catch(MalformedURLException e) {					
				}
			}
			else {
				throw new FileNotFoundException(fileName);
			}
		}
		
		URL[] urlArray = new URL[urls.size()];
		return (URL[]) urls.toArray(urlArray);
	}
	
	private String getLoaderKey(URL[] urls) {
		String key = ""; //$NON-NLS-1$
		for(int i=0; i<urls.length; ++i) {
			key += urls[i].getPath();
		}
		return key;
	}

	private ConnectionInfo loadConnectionInfo(String infoName) {
		IPath path = RDBCorePlugin.getDefault().getStateLocation();
		path = path.append(CONNECTION + infoName + "/connection.info"); //$NON-NLS-1$
		File file = path.toFile();

		if(!file.exists()) return null;

		ConnectionInfoImpl info = new ConnectionInfoImpl(null, infoName);
		try { 
			Properties properties = new Properties();
			InputStream input = new FileInputStream(file);
			properties.load(input);
			String product = null;
			String version = null;

			Enumeration e = properties.propertyNames();
			while(e.hasMoreElements()) {
				String name = (String) e.nextElement();
				if(name.equals(ConnectionManagerImpl.LOADING_PATH)) {
					info.setLoadingPath(properties.getProperty(name));					
				}
				else if(name.equals(ConnectionManagerImpl.DRIVER_CLASS_NAME)) {
					info.setDriverClassName(properties.getProperty(name));
				}
				else if(name.equals(ConnectionManagerImpl.URL)) {
					info.setURL(properties.getProperty(name));					
				}
				else if(name.equals(ConnectionManagerImpl.PRODUCT)) {
					product = properties.getProperty(name);					
				}
				else if(name.equals(ConnectionManagerImpl.VERSION)) {
					version = properties.getProperty(name);					
				}
				else if(name.equals(ConnectionManagerImpl.IDENTIFIER_QUOTE_STRING)) {
					info.setIdentifierQuoteString(properties.getProperty(name));					
				}
                else if(name.equals(ConnectionManagerImpl.DATABASE_PRODUCT_VERSION)) {
                   info.setDatabaseProductVersion(properties.getProperty(name));               
                }
				else if(name.equals(ConnectionManagerImpl.DATABASE_NAME)) {
					info.setDatabaseName(properties.getProperty(name));					
				}
				else {
				    if (name.startsWith(ConnectionManagerImpl.CUSTOM_PROPERTY_PREFIX)){
				        info.setCustomProperty(name.substring(ConnectionManagerImpl.CUSTOM_PROPERTY_PREFIX.length()), properties.getProperty(name));
				    } else {
				        info.getProperties().setProperty(name, properties.getProperty(name));
				    }
				}
			}
			input.close();
			info.setDatabaseDefinition(RDBCorePlugin.getDefault().getDatabaseDefinitionRegistry().getDefinition(product, version));
		}
		catch(FileNotFoundException e) {
		}
		catch(IOException e) {
		}
		
		String infoNameKey = infoName.toUpperCase();
		this.infoMap.put(infoNameKey, info);
		
		return info;
	}
	
	private ConnectionInfo loadConnectionInfo(DatabaseDefinition def) {
		ConnectionInfoImpl info = new ConnectionInfoImpl(def, null);
		IPath path = RDBCorePlugin.getDefault().getStateLocation();
		path = path.append(CONNECTION);
		String filename = def.getProduct() + " " + def.getVersion() + "." + INFO_FILE_EXTENSION; //$NON-NLS-1$ //$NON-NLS-2$
		path = path.append(filename);
		File file = path.toFile();
		if(file.exists()) {
			try { 
				Properties properties = new Properties();
				InputStream input = new FileInputStream(file);
				properties.load(input);
	
				Enumeration e = properties.propertyNames();
				while(e.hasMoreElements()) {
					String name = (String) e.nextElement();
					if(name.equals(ConnectionManagerImpl.LOADING_PATH)) {
						info.setLoadingPath(properties.getProperty(name));					
					}
					else if(name.equals(ConnectionManagerImpl.DRIVER_CLASS_NAME)) {
						info.setDriverClassName(properties.getProperty(name));
					}
					else if(name.equals(ConnectionManagerImpl.URL)) {
						info.setURL(properties.getProperty(name));					
					}
					else {
						info.getProperties().setProperty(name, properties.getProperty(name));					
					}
				}
				input.close();
			}
			catch(FileNotFoundException e) {
			}
			catch(IOException e) {
			}
		}
		
		this.infoMap.put(def, info);
		
		return info;
	}
	
	private Map infoMap = new HashMap();
	private Collection listeners = new LinkedList();
	private Map loaderMap = new HashMap();
}
