/*******************************************************************************
 * Copyright (c) 2001, 2004 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.wst.rdb.connection.internal.ui.wizards.shared;

import java.io.File;
import java.sql.Connection;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Properties;
import java.util.StringTokenizer;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.wizard.IWizard;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.custom.StackLayout;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeItem;
import org.eclipse.swt.widgets.Widget;
import org.eclipse.ui.help.WorkbenchHelp;
import org.eclipse.wst.rdb.connection.internal.ui.factories.RSCCoreUIWidgetFactory;
import org.eclipse.wst.rdb.connection.internal.ui.providers.IServerConnectionInformation;
import org.eclipse.wst.rdb.connection.internal.ui.providers.IServerConnectionUIProvider;
import org.eclipse.wst.rdb.connection.internal.ui.util.help.RSCCommonUIContextIds;
import org.eclipse.wst.rdb.connection.internal.ui.util.resource.ResourceLoader;
import org.eclipse.wst.rdb.connection.internal.ui.wizards.ConnectionWizardProperties;
import org.eclipse.wst.rdb.connection.internal.ui.wizards.NewCWUidPwd;
import org.eclipse.wst.rdb.connection.internal.ui.wizards.NewConnectionWizard;
import org.eclipse.wst.rdb.connection.internal.ui.wizards.RDBWizardPage;
import org.eclipse.wst.rdb.connection.internal.ui.wizards.helper.ConnectionUIHelper;
import org.eclipse.wst.rdb.internal.core.RDBCorePlugin;
import org.eclipse.wst.rdb.internal.core.connection.ConnectionInfo;
import org.eclipse.wst.rdb.internal.core.definition.DatabaseDefinition;

/**
 * JDBC page of the Connection Wizard. Here, users pick a DBMS, a driver, and
 * complete the connection URL. This page directly owns a tree to select a
 * vendor, a combo box to select a JDBC driver supported by the selected vendor,
 * and a panel to contain driver-specific fields used in the composition of the
 * connection URL.
 * 
 * @version 1.0
 */
public class NewCWJDBCPage extends RDBWizardPage implements Listener,
        SelectionListener, IServerConnectionInformation {
    private static final ResourceLoader resource = ResourceLoader.INSTANCE;

    private static final String JDBC_CONNECTION_NAME_GROUP = resource
            .queryString("JDBC_CONNECTION_NAME_GROUP"); //$NON-NLS-1$

    private static final String JDBC_CONNECTION_NAME_DEFAULT = resource
            .queryString("JDBC_CONNECTION_NAME_DEFAULT"); //$NON-NLS-1$

    private static final String JDBC_CONNECTION_NAME_LABEL = resource
            .queryString("JDBC_CONNECTION_NAME_LABEL"); //$NON-NLS-1$

    private static final String CUI_NEWCW_VALIDATE_CONNECTION_NAME_EMPTY = resource
            .queryString("CUI_NEWCW_VALIDATE_CONNECTION_NAME_EMPTY"); //$NON-NLS-1$

    private static final String CUI_NEWCW_VALIDATE_CONNECTION_NAME_EXIST = resource
            .queryString("CUI_NEWCW_VALIDATE_CONNECTION_NAME_EXIST"); //$NON-NLS-1$

    protected String MATCH = "match"; //$NON-NLS-1$

    protected String KEY1 = "child1"; //$NON-NLS-1$

    protected String KEY2 = "child2"; //$NON-NLS-1$

    protected int contentOptions;

    protected boolean hasBeenHere = false;

    private boolean informationProviderComplete = false;

    private boolean validateConnectionName = true;

    private boolean isFinalConnection;

    private String url = ""; //$NON-NLS-1$

    private String loadingPath = ""; //$NON-NLS-1$

    private String driverClassName = ""; //$NON-NLS-1$

    private String databaseName;

    private List existingConnectionNamesList;

    protected Label dbmsLabel;

    protected Tree dbmsTree;

    protected Label driverLabel;

    protected Combo driverCombo;

    protected Group detailsGroup;

    protected StackLayout stack;

    protected Button testConnection;

    private Label connectionNameLabel;

    private Text connectionNameText;

    private Button connectionNameCheck;

    private ConnectionUIHelper connectionUIhelper;
    
    private HashMap providerConnectionMap = new HashMap();

    private NewCWUidPwd userPanel;

    private IServerConnectionUIProvider currentDriverUIPanel;

    private DatabaseDefinition definition;

    private ConnectionWizardProperties connectionProperties = null;

    private DatabaseProductVersion[] allowedDatabaseProductVersions = null;

    private Properties customProperties = new Properties();

    private boolean isServerRecognized = true;

    private ScrolledComposite detailsScrolledComposite;

    /**
     * Constructs the page with default options.
     */
    public NewCWJDBCPage(String name) {
        this(name, getExistingConnectionNamesList());
    }

    /**
     * Constructs the page with default options.
     */
    public NewCWJDBCPage(String name, List existingConnectionNamesList) {
        this(name, NewConnectionWizard.DEFAULT_OPTIONS,
                existingConnectionNamesList);
    }

    /**
     * Constructs the page with given options.
     */
    public NewCWJDBCPage(String name, int contentOptions,
            List existingConnectionNamesList) {
        super(name);
        setTitle(getString("CUI_NEWCW_JDBC_TITLE_UI_")); //$NON-NLS-1$
        setDescription(getString("CUI_NEWCW_JDBC_DESC_UI_")); //$NON-NLS-1$
        this.contentOptions = contentOptions;
        this.existingConnectionNamesList = existingConnectionNamesList;
    }

    public void createControl(Composite parent) {
        Composite composite = RSCCoreUIWidgetFactory.INSTANCE.createComposite(
                parent, SWT.NULL);

        GridLayout layout = new GridLayout();
        GridData gridData = new GridData(GridData.FILL_BOTH);

        layout.numColumns = 2;
        layout.verticalSpacing = 3;

        composite.setLayout(layout);
        composite.setLayoutData(gridData);

        initializeDialogUnits(composite);
        createGUI(composite, RSCCoreUIWidgetFactory.INSTANCE);
        setControl(composite);

        setPageComplete(false);

        populateGUI();
        hasBeenHere = true;
    }

    /**
     * Populates the GUI.
     */
    protected void populateGUI() {
        addWorkbenchHelp(getControl().getParent());
        addListeners();
        fillTree(dbmsTree);
        setPageDefaults();
        setPageComplete(determinePageCompletion());
    }

    /**
     * Creates UI for picking an existing connection.
     */
    protected void createGUI(Composite parent, RSCCoreUIWidgetFactory factory) {
        GridData gd;

        Group connectionGroup = factory.createGroup(parent,
                JDBC_CONNECTION_NAME_GROUP, SWT.NONE);
        gd = new GridData(GridData.FILL_HORIZONTAL);
        gd.horizontalSpan = 2;
        GridLayout layout = new GridLayout();
        layout.numColumns = 2;
        connectionGroup.setLayoutData(gd);
        connectionGroup.setLayout(layout);

        connectionNameCheck = factory.createButton(connectionGroup, SWT.CHECK);
        gd = new GridData(GridData.FILL_HORIZONTAL);
        gd.horizontalSpan = 2;
        connectionNameCheck.setLayoutData(gd);
        connectionNameCheck.setText(JDBC_CONNECTION_NAME_DEFAULT);
        connectionNameCheck.setSelection(true);
        connectionNameCheck.addSelectionListener(new SelectionListener() {
            public void widgetSelected(SelectionEvent e) {
                connectionNameText.setEnabled(!connectionNameCheck
                        .getSelection());
                setDatabaseName(databaseName);
            }

            public void widgetDefaultSelected(SelectionEvent e) {
            }
        });

        connectionNameLabel = factory.createLabel(connectionGroup, SWT.NONE);
        connectionNameLabel.setText(JDBC_CONNECTION_NAME_LABEL);

        connectionNameText = factory.createText(connectionGroup, SWT.SINGLE
                | SWT.BORDER);
        gd = new GridData(GridData.FILL_HORIZONTAL);
        connectionNameText.setLayoutData(gd);
        connectionNameText.setEnabled(false);

        dbmsLabel = factory.createLabel(parent, SWT.NONE);
        dbmsLabel.setText(getString("CUI_NEWCW_DBMS_LBL_UI_")); //$NON-NLS-1$
        gd = new GridData();
        gd.horizontalSpan = 2;
        dbmsLabel.setLayoutData(gd);

        dbmsTree = factory.createTree(parent, SWT.SINGLE | SWT.BORDER
                | SWT.H_SCROLL | SWT.V_SCROLL);
        gd = new GridData(175, SWT.DEFAULT);
        gd.horizontalAlignment = GridData.BEGINNING;
        gd.verticalAlignment = GridData.FILL;
        gd.grabExcessHorizontalSpace = false;
        gd.grabExcessVerticalSpace = true;
        gd.horizontalSpan = 1;
        dbmsTree.setLayoutData(gd);

//        ScrolledComposite scrolledRightSide = new ScrolledComposite(parent, SWT.H_SCROLL|SWT.V_SCROLL);
//        gd = new GridData();
//        gd.horizontalAlignment = GridData.FILL;
//        gd.verticalAlignment = GridData.FILL;
//        gd.grabExcessHorizontalSpace = true;
//        gd.grabExcessVerticalSpace = true;
//        gd.horizontalSpan = 1;
//        scrolledRightSide.setLayoutData(gd);
        
        Composite rightside = factory.createComposite(parent, SWT.NONE);
        layout = new GridLayout();
        layout.marginWidth = 0;
        layout.marginHeight = 0;
        layout.numColumns = 2;
        layout.horizontalSpacing = 5;
        layout.verticalSpacing = 5;
        rightside.setLayout(layout);
        gd = new GridData();
        gd.horizontalAlignment = GridData.FILL;
        gd.verticalAlignment = GridData.FILL;
        gd.grabExcessHorizontalSpace = true;
        gd.grabExcessVerticalSpace = true;
        gd.horizontalSpan = 1;
        rightside.setLayoutData(gd);

//        scrolledRightSide.setContent(new Button(scrolledRightSide, SWT.PUSH));
        
        driverLabel = factory.createLabel(rightside, SWT.NONE);
        driverLabel.setText(getString("CUI_NEWCW_JDBC_DRV_LBL_UI_")); //$NON-NLS-1$
        gd = new GridData();
        gd.verticalAlignment = GridData.BEGINNING;
        driverLabel.setLayoutData(gd);

        driverCombo = factory.createCombo(rightside, SWT.DROP_DOWN
                | SWT.READ_ONLY | SWT.SINGLE | SWT.BORDER);
        gd = new GridData();
        gd.verticalAlignment = GridData.BEGINNING;
        gd.grabExcessHorizontalSpace = true;
        gd.horizontalAlignment = GridData.FILL;
        gd.horizontalSpan = 1;
        driverCombo.setLayoutData(gd);
        
        detailsScrolledComposite = new ScrolledComposite(rightside, SWT.H_SCROLL|SWT.V_SCROLL);
        detailsScrolledComposite.setExpandHorizontal(true);
        detailsScrolledComposite.setExpandVertical(true);
        gd = new GridData();
        gd.verticalAlignment = GridData.FILL;
        gd.grabExcessVerticalSpace = true;
        gd.horizontalAlignment = GridData.FILL;
        gd.grabExcessHorizontalSpace = true;
        gd.horizontalSpan = 2;
        detailsScrolledComposite.setLayoutData(gd);
        
        detailsGroup = factory.createGroup(detailsScrolledComposite,
                getString("CUI_NEWCW_DETAILS_GROUP_UI_"), SWT.NONE); //$NON-NLS-1$
        stack = new StackLayout();
        stack.marginWidth = 3;
        stack.marginHeight = 3;
        detailsGroup.setLayout(stack);
       
        detailsScrolledComposite.setContent(detailsGroup);

        userPanel = new NewCWUidPwd(this, contentOptions);
        Composite userDetails = userPanel.getComposite(parent, factory);
        gd = new GridData();
        gd.verticalAlignment = GridData.FILL;
        gd.horizontalAlignment = GridData.FILL;
        gd.grabExcessHorizontalSpace = true;
        gd.horizontalSpan = 2;
        userDetails.setLayoutData(gd);

        testConnection = factory.createButton(parent, SWT.PUSH);
        testConnection.setText(getString("CUI_NEWCW_JDBC_TESTCONNECTION_UI_")); //$NON-NLS-1$
        
        detailsGroup.setSize(detailsGroup.computeSize(SWT.DEFAULT, SWT.DEFAULT));
    }

    protected void fillTree(Tree tree) {
        boolean isDatabaseManagersFiltered = false;
        Hashtable productVersionsTable = new Hashtable();
        tree.removeAll();
        if (allowedDatabaseProductVersions != null) {
            isDatabaseManagersFiltered = true;
            for (int index = 0; index < allowedDatabaseProductVersions.length; index++) {
                productVersionsTable.put(allowedDatabaseProductVersions[index]
                        .getProduct()
                        + allowedDatabaseProductVersions[index].getVersion(),
                        allowedDatabaseProductVersions[index].getProduct());
            }
        }
        for (Iterator productIterator = RDBCorePlugin.getDefault()
                .getDatabaseDefinitionRegistry().getConnectibleProducts(); productIterator
                .hasNext();) {
            String databaseEngineName = (String) productIterator.next();
            if ((!isDatabaseManagersFiltered)
                    || productVersionsTable.contains(databaseEngineName)) {
                TreeItem item = new TreeItem(tree, SWT.NONE);
                item.setText(databaseEngineName);
                for (Iterator versionIterator = RDBCorePlugin.getDefault()
                        .getDatabaseDefinitionRegistry().getConnectibleVersions(
                                databaseEngineName); versionIterator.hasNext();) {
                    String version = (String) versionIterator.next();
                    if ((!isDatabaseManagersFiltered)
                            || productVersionsTable
                                    .containsKey(databaseEngineName + version)) {
                        TreeItem subItem = new TreeItem(item, SWT.NONE);
                        subItem.setText(version);
                    }
                }
            }
        }
    }

    /**
     * Adds workbench help to various components.
     */
    protected void addWorkbenchHelp(Composite parent) {
        WorkbenchHelp.setHelp(dbmsTree,
                RSCCommonUIContextIds.RSC_NEWCW_DBMSTREE);
        WorkbenchHelp.setHelp(driverCombo,
                RSCCommonUIContextIds.RSC_NEWCW_DRIVERCOMBO);
    }

    /**
     * Adds this as listener to various components.
     */
    protected void addListeners() {
        connectionNameText.removeListener(SWT.Modify, this);
        dbmsTree.removeSelectionListener(this);
        driverCombo.removeListener(SWT.Modify, this);
        testConnection.removeListener(SWT.Selection, this);
        userPanel.removeListeners();

        dbmsTree.addSelectionListener(this);
        driverCombo.addListener(SWT.Modify, this);
        testConnection.addListener(SWT.Selection, this);
        connectionNameText.addListener(SWT.Modify, this);
        userPanel.addListeners();
    }

    protected void setPageDefaults() {
        internalLoadWidgetValues();
        enableAll();
    }

    /**
     * Enables components according to the contentOptions.
     */
    protected void enableAll() {
        boolean vt = (contentOptions & NewConnectionWizard.DISABLE_VENDOR_TYPE) == 0;
        if (!vt) {
            dbmsLabel.setEnabled(false);
            dbmsTree.setEnabled(false);
        }
        boolean dd = (contentOptions & NewConnectionWizard.DISABLE_DRIVER) == 0;
        if (!dd) {
            driverLabel.setEnabled(false);
            driverCombo.setEnabled(false);
        }
    }

    public static String createUniqueConnectionName(Collection existed, String baseName) {
        int baseLength = baseName.length();
        int size = existed.size();
        boolean[] flags = new boolean[size];
        Iterator it = existed.iterator();
        while (it.hasNext()) {
            String name = (String) it.next();
            if (name.toLowerCase().startsWith(baseName.toLowerCase())) {
                String postfix = name.substring(baseLength);
                try {
                    int value = Integer.parseInt(postfix);
                    if (value > 0 && value <= size) {
                        flags[value - 1] = true;
                        continue;
                    }
                } catch (NumberFormatException e) {
                }
            }
            --size;
        }
        int seq = 0;
        for (int i = 0; i < size; ++i) {
            if (flags[i])
                ++seq;
            else
                break;
        }
        return seq == 0 ? baseName : baseName + ++seq;
    }

    private void setConnectionName(String connectionName) {
        String uniqueName = createUniqueConnectionName(
                this.existingConnectionNamesList, connectionName);
        if (this.existingConnectionNamesList.contains(uniqueName.toLowerCase())) {
            uniqueName += 1;
        }
        this.connectionNameText.setText(uniqueName);
    }

    /**
     * The <code>DialogPage</code> implementation of this
     * <code>IDialogPage</code> method sets the control to the given
     * visibility state. Subclasses may extend.
     */
    public void setVisible(boolean visible) {
        if (visible) {
            if (!hasBeenHere) {
                hasBeenHere = true;
                populateGUI();
            } else {
                setPageComplete(determinePageCompletion());
            }
        }
        super.setVisible(visible);
    }

    protected boolean determinePageCompletion() {
        boolean ok = validatePage()
                && this.currentDriverUIPanel.determinePageCompletion()
                && validateConnectionName()
                && userPanel.determinePageCompletion();
        if (ok) {
            setErrorMessage(null);
        }
        if (testConnection != null) {
            testConnection.setEnabled(ok);
        }
        return ok;
    }

    protected boolean validatePage() {
        return beenHere() && validateDbms() && validateDriver();
    }

    /**
     * Returns true only if this page has been visited or we are editing an
     * existing connection.
     */
    protected boolean beenHere() {
        return hasBeenHere;
    }

    /**
     * Validates the vendor selection.
     * 
     * @return True if a vendor is selected.
     */
    protected boolean validateDbms() {
        if (dbmsTree.getSelectionCount() < 1) {
            setErrorMessage(getString("CUI_NEWCW_VALIDATE_DBMS_REQ_UI_")); //$NON-NLS-1$
            return false;
        } else {
            return true;
        }
    }

    /**
     * Validates the driver selection.
     * 
     * @return True if a driver is selected.
     */
    protected boolean validateDriver() {
        if (driverCombo.getText().length() < 1) {
            setErrorMessage(getString("CUI_NEWCW_VALIDATE_DRIVER_REQ_UI_")); //$NON-NLS-1$
            return false;
        } else {
            return true;
        }
    }

    protected String[] getClassLocations(String classList) {
        StringTokenizer str = new StringTokenizer(classList,
                File.pathSeparator, false);
        String[] jFiles = new String[str.countTokens()];
        for (int i = 0; i < jFiles.length; i++) {
            jFiles[i] = str.nextToken();
        }
        return jFiles;
    }

    /** Implements the Listener interface. */
    public void handleEvent(Event e) {
        Widget source = e.widget;
        if (source == testConnection) {
            this.performTestConnection(true, true);
            return;
        } else if (source == driverCombo) {
            updateDetails();
        }
        setPageComplete(determinePageCompletion());
    }

    private boolean validateConnectionName() {
        if (connectionNameText.getText().equals("")) { //$NON-NLS-1$
            this.setErrorMessage(CUI_NEWCW_VALIDATE_CONNECTION_NAME_EMPTY);
            return false;
        } else if (existingConnectionNamesList.contains(connectionNameText
                .getText().toLowerCase())) {
            String message = MessageFormat.format(
                    CUI_NEWCW_VALIDATE_CONNECTION_NAME_EXIST,
                    new String[] { connectionNameText.getText() });
            this.setErrorMessage(message);
            return false;
        } else {
            return true;
        }
    }

    /**
     * Updates the details, given the current selection in the driverCombo.
     * 
     * @param wiz
     *            The main wizard class.
     */
    protected void updateDetails() {
        //	userPanel.enableCurrentUidPwd(true);
        String driverName = driverCombo.getText();
        currentDriverUIPanel = connectionUIhelper.getUIProvider(driverName);
        if (currentDriverUIPanel != null) {
            currentDriverUIPanel.setServerConnectionInformation(this);
            currentDriverUIPanel.setDatabaseDefinition(this.definition);
            currentDriverUIPanel.setWizardPage(this);
            stack.topControl = currentDriverUIPanel
                    .getServerConnectionUI(detailsGroup);
            currentDriverUIPanel.determinePageCompletion();
            detailsGroup.layout();
            detailsScrolledComposite.setMinSize(detailsGroup.computeSize(SWT.DEFAULT, SWT.DEFAULT));
        }
    }

    public void widgetSelected(SelectionEvent e) {
        TreeItem item = (TreeItem) e.item;
        updateDrivers(item);
    }

    private DatabaseDefinition getDefinition(TreeItem item) {
        if (item.getParentItem() != null) {
            String product = item.getParentItem().getText();
            String version = item.getText();
            return RDBCorePlugin.getDefault().getDatabaseDefinitionRegistry()
                    .getDefinition(product, version);
        }
        return null;
    }

    protected void updateDrivers(TreeItem item) {
        if (item.getParentItem() != null && item.getItems().length == 0) {
            definition = getDefinition(item);
            if (definition != null) {
                driverCombo.removeListener(SWT.Modify, this);
                driverCombo.removeAll();
                connectionUIhelper = new ConnectionUIHelper(definition
                        .getProduct(), definition.getVersion(), providerConnectionMap);
                String[] names = connectionUIhelper.getDriverNames();
                for (int i = 0, n = names.length; i < n; i++) {
                    driverCombo.add(names[i]);
                }
                driverCombo.addListener(SWT.Modify, this);
                String defaultDriver = connectionUIhelper.getDefaultDriver();
                if (!connectionUIhelper.getDefaultDriver().equals("")) { //$NON-NLS-1$
                    driverCombo.select(driverCombo.indexOf(defaultDriver));
                } else {
                    driverCombo.select(0);
                }

                updateDetails();
            }
        }
        setPageComplete(determinePageCompletion());
    }

    public void widgetDefaultSelected(SelectionEvent e) {
    }

    /**
     * Sets the connection details.
     * 
     * @param connection
     *            An RDBConnection object to reflect the choices in this panel.
     */
    public void setConnectionDetails(ConnectionInfo connection) {
        connection.setDatabaseName(this.databaseName);
        connection.setURL(this.url);
        connection.setDriverClassName(this.driverClassName);
        connection.setLoadingPath(this.loadingPath);

        Enumeration customProperties = this.customProperties.propertyNames();
        while (customProperties.hasMoreElements()) {
            String name = (String) customProperties.nextElement();
            connection.setCustomProperty(name, this.customProperties
                    .getProperty(name));
        }

        userPanel.setConnectionDetails(connection);
    }

    /**
     * Determines our own completeness before calling super.
     * 
     * @see org.eclipse.jface.wizard.IWizardPage#isPageComplete()
     */
    public boolean isPageComplete() {
        return determinePageCompletion() && super.isPageComplete();
    }

    /**
     * Loads widget values from values saved by the previous instance of this
     * class.
     */
    protected void internalLoadWidgetValues() {
        IWizard wizard = getWizard();
        IDialogSettings dialogSettings = null;
//        if (wizard instanceof NewConnectionWizard) {
//            NewConnectionWizard wiz = (NewConnectionWizard) getWizard();
//            dialogSettings = wiz.getDialogSettings();
//        }
        if ((dialogSettings != null)
                && (dialogSettings.get("Database Product") != null)) { //$NON-NLS-1$
            String product = dialogSettings.get("Database Product"); //$NON-NLS-1$
            String version = dialogSettings.get("Database Version"); //$NON-NLS-1$
            TreeItem item = getTreeItem(dbmsTree.getItems(), RDBCorePlugin
                    .getDefault().getDatabaseDefinitionRegistry()
                    .getDefinition(product, version));
            if (item != null) {
                dbmsTree.setSelection(new TreeItem[] { item });
                updateDrivers(item);
            }
            String driver = dialogSettings.get("Driver"); //$NON-NLS-1$
            if (driver != null) {
                driverCombo.setText(driver);
            }
        } else {
            TreeItem item = getFirstDatabaseDefinition(dbmsTree.getItems());
            if (item != null) {
                dbmsTree.setSelection(new TreeItem[] { item });
                updateDrivers(item);
            }
        }
        userPanel.internalLoadWidgetValues();
    }

    private TreeItem getTreeItem(TreeItem[] items, DatabaseDefinition definition) {
        TreeItem result = null;
        if (items != null) {
            for (int i = 0, n = items.length; i < n && result == null; i++) {
                DatabaseDefinition currentDefinition = this
                        .getDefinition(items[i]);
                if (currentDefinition != null
                        && currentDefinition.equals(definition)) {
                    result = items[i];
                    break;
                } else {
                    result = getTreeItem(items[i].getItems(), definition);
                }
            }
        }
        return result;
    }

    private TreeItem getFirstDatabaseDefinition(TreeItem[] items) {
        TreeItem result = null;
        if (items != null) {
            for (int i = 0, n = items.length; i < n && result == null; i++) {
                DatabaseDefinition currentDefinition = this
                        .getDefinition(items[i]);
                if (currentDefinition != null) {
                    result = items[i];
                    break;
                } else {
                    result = getFirstDatabaseDefinition(items[i].getItems());
                }
            }
        }
        return result;
    }

    /**
     * Hook method for saving widget values for restoration by the next instance
     * of this class.
     */
    public void internalSaveWidgetValues() {
        IDialogSettings dialogSettings = getWizard().getDialogSettings();
        if (dialogSettings != null) {
            dialogSettings
                    .put("Database Product", this.definition.getProduct()); //$NON-NLS-1$
            dialogSettings
                    .put("Database Version", this.definition.getVersion()); //$NON-NLS-1$
            dialogSettings.put("Driver", driverCombo.getText()); //$NON-NLS-1$
            dialogSettings.put("DatabaseName", this.databaseName); //$NON-NLS-1$
            dialogSettings.put("URL", this.url); //$NON-NLS-1$
            dialogSettings.put("LoadingPath", this.loadingPath); //$NON-NLS-1$
            dialogSettings.put("DriverClassName", this.driverClassName); //$NON-NLS-1$

            userPanel.internalSaveWidgetValues();
        }
    }

    public void setInformationComplete(boolean value) {
        this.informationProviderComplete = value;
        setPageComplete(determinePageCompletion());
    }

    public void setDatabaseName(String databaseName) {
        if (databaseName != null) {
            String connectionName;

            this.databaseName = databaseName;
            if (connectionNameCheck.getSelection()) {
                connectionName = this.databaseName;
            } else {
                connectionName = this.connectionNameText.getText();
            }
            setConnectionName(connectionName);
        }
    }

    public void setURL(String url) {
        this.url = url;
    }

    public String getURL() {
        return this.url;
    }

    public String getConnectionName() {
        if (this.connectionNameText != null) {
            return this.connectionNameText.getText();
        }
        return ""; //$NON-NLS-1$
    }

    public String getDatabaseName() {
        return this.databaseName;
    }

    public void setDriverClassName(String driverClassName) {
        this.driverClassName = driverClassName;
    }

    public String getDriverClassName() {
        return this.driverClassName;
    }

    public void setLoadingPath(String loadingPath) {
        this.loadingPath = loadingPath;
    }

    public String getLoadingPath() {
        return this.loadingPath;
    }

    public void setDefaultDatabaseDefinition(
            DatabaseDefinition databaseDefinition) {
        this.definition = databaseDefinition;
    }

    public boolean performTestConnection(boolean showSuccessfulMessage) {
        return performTestConnection(showSuccessfulMessage, false);
    }

    public boolean performTestConnection(boolean showSuccessfulMessage,
            boolean showServerMismatchWarning) {
        isServerRecognized = true;
        ConnectionInfo connection = RDBCorePlugin.getDefault()
                .getConnectionManager().createConnectionInfo(
                        definition,
                        createUniqueConnectionName(existingConnectionNamesList,
                                getConnectionName()));
        setConnectionDetails(connection);
        try {
            boolean showStatus = false;
            try {
                getContainer().run(
                        true,
                        true,
                        getTestRunnable(connection,
                                NewConnectionWizard.JDBC_CONNECTION_TYPE,
                                showSuccessfulMessage,
                                showServerMismatchWarning));

            } catch (InterruptedException e) {
                showStatus = false;
                if (connection != null) {
                    RDBCorePlugin.getDefault().getConnectionManager()
                            .removeConnectionInfo(connection.getName());
                }
            } finally {
            }
        } catch (Exception e) {
            MessageDialog.openError((Shell) null, RDBWizardPage.getBundle()
                    .getString("CUI_WIZARD_ERRTITLE_UI_"), NewConnectionWizard //$NON-NLS-1$
                    .getMessages(e));
            return false;
        }
        return true;
    }

    /**
     * Gets a Runnable to make the connection.
     */
    public IRunnableWithProgress getTestRunnable(
            final ConnectionInfo connection, final int conType,
            final boolean showStatus, final boolean showMismatchWarning) {
        IRunnableWithProgress progMonitor = new IRunnableWithProgress() {
            public void run(IProgressMonitor monitor) {
                if (monitor == null) {
                    monitor = new NullProgressMonitor();
                }
                monitor.beginTask("", 3); //$NON-NLS-1$
                if (showStatus) {
                    monitor.setTaskName(resource.queryString("CUI_NEWCW_TEST_CONNECTION_PROGRESS_BAR_MESSAGE_UI_")); //$NON-NLS-1$
                }
                monitor.worked(1);
                try {
                    String remoteServerProduct = ""; //$NON-NLS-1$
                    String remoteServerVersion = ""; //$NON-NLS-1$
                    if (connection.getSharedConnection() == null) {
                        Connection cnx = connection.connect();

                        DatabaseDefinition recognizedDatabaseDefinition = RDBCorePlugin
                                .getDefault().getDatabaseDefinitionRegistry()
                                .recognize(cnx);
                        if (recognizedDatabaseDefinition == null) {
                            // Ignore - Recognizer not implemented
                        } else {
                            if ((!recognizedDatabaseDefinition.getProduct()
                                    .equals(
                                            connection.getDatabaseDefinition()
                                                    .getProduct()))
                                    || (!recognizedDatabaseDefinition
                                            .getVersion()
                                            .equals(
                                                    connection
                                                            .getDatabaseDefinition()
                                                            .getVersion()))) {
                                isServerRecognized = false;
                                remoteServerProduct = recognizedDatabaseDefinition
                                        .getProduct();
                                remoteServerVersion = recognizedDatabaseDefinition
                                        .getVersion();
                            }
                        }

                        if (cnx == null) {
                            throw new Exception(RDBWizardPage.getBundle()
                                    .getString("CUI_TESTCONNECTION_ERROR_UI_")); //$NON-NLS-1$
                        }

                    }
                    if (!isServerRecognized && showMismatchWarning) {
                        String message = ""; //$NON-NLS-1$
                        message += resource.queryString("_UI_MESSAGE_MISMATCH_WARNING"); //$NON-NLS-1$
                        message += resource.queryString("_UI_MESSAGE_SPECIFIED_DB_MANAGER") //$NON-NLS-1$
                                + connection.getDatabaseDefinition()
                                        .getProduct()
                                + " " //$NON-NLS-1$
                                + connection.getDatabaseDefinition()
                                        .getVersion() + "\n"; //$NON-NLS-1$
                        message += resource.queryString("_UI_MESSAGE_REMOTE_SERVER") + remoteServerProduct //$NON-NLS-1$
                                + " " + remoteServerVersion + "\n\n"; //$NON-NLS-1$ //$NON-NLS-2$
                        message += resource.queryString("_UI_MESSAGE_CONTINUE_QUESTION"); //$NON-NLS-1$

                        DisplayQuestion question = new DisplayQuestion(
                        		resource.queryString("_UI_TITLE_MISMATCH_DIALOG"), message); //$NON-NLS-1$
                        getShell().getDisplay().syncExec(question);
                        if (!question.getResult()) {
                            throw new Exception(RDBWizardPage.getBundle()
                                    .getString("CUI_TESTCONNECTION_ERROR_UI_")); //$NON-NLS-1$
                        }
                    } else if (showStatus) {
                        String title = RDBWizardPage.getBundle().getString(
                                "CUI_TESTCONNECTION_TITLE_UI_"); //$NON-NLS-1$
                        String message = RDBWizardPage.getBundle().getString(
                                "CUI_TESTCONNECTION_SUCCESS_UI_"); //$NON-NLS-1$
                        message = MessageFormat
                                .format(message, new String[] { connection
                                        .getDatabaseDefinition().getProduct() });
                        monitor.worked(2);
                        getShell().getDisplay().asyncExec(
                                new DisplayMessage(title, message));
                    }
                    isFinalConnection = true;
                    RDBCorePlugin.getDefault().getConnectionManager()
                            .removeConnectionInfo(connection.getName());
                } catch (Exception e) {
                    monitor.worked(2);
                    RDBCorePlugin.getDefault().getConnectionManager()
                            .removeConnectionInfo(connection.getName());
                    isFinalConnection = false;
                    if (isServerRecognized) {
                        String title = RDBWizardPage.getBundle().getString(
                                "CUI_TESTCONNECTION_TITLE_UI_"); //$NON-NLS-1$
                        String message = RDBWizardPage.getBundle().getString(
                                "CUI_TESTCONNECTION_FAILURE_UI_"); //$NON-NLS-1$
                        message = MessageFormat
                                .format(message, new String[] {
                                        connection.getDatabaseDefinition()
                                                .getProduct(), e.getMessage() });
                        getShell().getDisplay().asyncExec(
                                new DisplayMessage(title, message));
                    }
                }
                monitor.setTaskName(RDBWizardPage.getBundle().getString(
                        "CUI_CREATECONDONE_UI_")); //$NON-NLS-1$
                monitor.worked(3);
                monitor.done();
            };
        };
        return progMonitor;
    }

    /** Runnable to display success or failure of Test Connection. */
    public class DisplayMessage implements Runnable {
        protected String title, message;

        public DisplayMessage(String title, String message) {
            this.title = title;
            this.message = message;
        }

        public void run() {
            MessageDialog.openInformation(getShell(), title, message);
        }
    }

    /** Runnable to display warning */
    public class DisplayQuestion implements Runnable {
        protected String title, message;

        protected boolean result = true;

        public DisplayQuestion(String title, String message) {
            this.title = title;
            this.message = message;
        }

        public boolean getResult() {
            return result;
        }

        public void run() {
            result = MessageDialog.openQuestion(getShell(), title, message);
        }
    }

    public boolean isFinalConnection() {
        return isFinalConnection;
    }

    public ConnectionInfo getConnection() {
        ConnectionInfo connection = RDBCorePlugin.getDefault()
                .getConnectionManager().createConnectionInfo(definition,
                        getConnectionName());
        setConnectionDetails(connection);
        return connection;
    }

    public static List getExistingConnectionNamesList() {
        ConnectionInfo[] infos = RDBCorePlugin.getDefault()
                .getConnectionManager().getAllNamedConnectionInfo();
        List nameList = new ArrayList(infos.length);
        for (int i = 0, n = infos.length; i < n; i++) {
            if (!nameList.contains(infos[i].getName())) {
                nameList.add(infos[i].getName().toLowerCase());
            }
        }
        String[] names = (String[]) nameList
                .toArray(new String[nameList.size()]);
        return Arrays.asList(names);
    }

    public void updateExistingConnectionNamesList() {
        existingConnectionNamesList = getExistingConnectionNamesList();
    }

    /**
     * Fills all the entry fields with information from the passed connection
     * properties. Listeners are disabled.
     */
    public void setConnectionProperties(
            ConnectionWizardProperties connectionProperties) {
        if (!hasBeenHere) {
            populateGUI();
            hasBeenHere = true;
        }

        if (connectionProperties.getConnectionName() != null) {
            setConnectionName(connectionProperties.getConnectionName());
            connectionNameCheck.setSelection(false);
            connectionNameText.setEnabled(true);
        }
        
        String product = connectionProperties.getProduct();
        String version = connectionProperties.getVersion();
        if ((product != null) && (version != null)) {
            TreeItem select = findDBMSItem(dbmsTree, product, version);
            if (select != null) {
                dbmsTree.setSelection(new TreeItem[] { select });
                dbmsTree.showItem(select.getParentItem());
                updateDrivers(select);
            }
        }
        //  TODO: This most likely needs to handle translated driver names
        if (driverCombo != null) {
            String driver = connectionProperties.getDriverName();
            if (driver != null) {
                driverCombo.select(driverCombo.indexOf(driver));
            }
        }

        if (connectionProperties.getDatabaseName() != null) {
            setDatabaseName(connectionProperties.getDatabaseName());
        }
        
        if (connectionProperties.getClassLocation() != null) {
            this.setLoadingPath(connectionProperties.getClassLocation());
        }

        if (connectionProperties.getDriverClassName() != null) {
            this.setDriverClassName(connectionProperties.getDriverClassName());
        }

        if (connectionProperties.getURL() != null) {
            this.setURL(connectionProperties.getURL());
        }

        if (connectionProperties.getUserID() != null) {
            if (userPanel != null) {
                userPanel.setUserId(connectionProperties.getUserID());
                //				userPanel.selectCurrentUidPwdCheckbox(false);
            }
        }

        currentDriverUIPanel.loadProperties();
    }

    /**
     * Finds the TreeItem for a particular vendor.
     * 
     * @param tree
     *            The tree
     * @param vendor
     *            The SQLVendor to find in the tree.
     * @return The TreeItem for the vendor.
     */
    protected TreeItem findDBMSItem(Tree tree, String product, String version) {
        TreeItem foundNode = null;
        if (tree != null) {
            TreeItem[] productNodes = tree.getItems();
            TreeItem[] versionNodes;
            for (int productIndex = 0; productIndex < productNodes.length; productIndex++) {
                if (productNodes[productIndex].getText().equals(product)) {
                    versionNodes = productNodes[productIndex].getItems();
                    for (int versionIndex = 0; versionIndex < versionNodes.length; versionIndex++) {
                        if (versionNodes[versionIndex].getText()
                                .equals(version)) {
                            foundNode = versionNodes[versionIndex];
                        }
                    }
                }
            }
        }
        return foundNode;
    }

    public void setAllowedProductVersions(
            DatabaseProductVersion[] allowedDatabaseProductVersions) {
        this.allowedDatabaseProductVersions = allowedDatabaseProductVersions;
        if (dbmsTree != null) {
            populateGUI();
        }
    }

    public void setCustomProperty(String name, String value) {
        if (customProperties.containsKey(name)) {
            customProperties.remove(name);
            customProperties.put(name, value);
        } else {
            customProperties.put(name, value);
        }
    }
}