/*******************************************************************************
 * Copyright (c) 2004 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.wst.html.ui.internal.edit.ui;

import org.eclipse.core.runtime.Preferences;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.help.WorkbenchHelp;
import org.eclipse.wst.html.core.internal.HTMLCorePlugin;
import org.eclipse.wst.html.core.internal.preferences.HTMLCorePreferenceNames;
import org.eclipse.wst.html.core.internal.provisional.contenttype.ContentTypeIdForHTML;
import org.eclipse.wst.html.ui.internal.HTMLUIMessages;
import org.eclipse.wst.html.ui.internal.editor.IHelpContextIds;
import org.eclipse.wst.sse.core.internal.encoding.CommonEncodingPreferenceNames;
import org.eclipse.wst.sse.core.internal.ltk.modelhandler.IModelHandler;
import org.eclipse.wst.sse.core.internal.provisional.IStructuredModel;
import org.eclipse.wst.xml.core.internal.cleanup.XMLCleanupPreferencesImpl;
import org.eclipse.wst.xml.core.internal.provisional.contenttype.ContentTypeIdForXML;

public class CleanupDialogHTML extends Dialog implements SelectionListener {

	protected Button fRadioButtonTagNameCaseAsis;
	protected Button fRadioButtonTagNameCaseLower;
	protected Button fRadioButtonTagNameCaseUpper;
	protected Button fRadioButtonAttrNameCaseAsis;
	protected Button fRadioButtonAttrNameCaseLower;
	protected Button fRadioButtonAttrNameCaseUpper;
	protected Button fCheckBoxInsertRequiredAttrs;
	protected Button fCheckBoxInsertMissingTags;
	protected Button fCheckBoxQuoteAttrValues;
	protected Button fCheckBoxFormatSource;
	protected Button fCheckBoxConvertEOLCodes;
	protected Button fRadioButtonConvertEOLWindows;
	protected Button fRadioButtonConvertEOLUnix;
	protected Button fRadioButtonConvertEOLMac;
	protected IStructuredModel fModel = null;
	protected Preferences fPreferences = null;

	public CleanupDialogHTML(Shell shell) {
		super(shell);
	}

	public Control createDialogArea(Composite parent) {
		getShell().setText(HTMLUIMessages.Cleanup_UI_);
		Composite composite = new Composite(parent, SWT.NULL);

		createDialogAreaInComposite(composite);
		initializeOptions();

		return composite;
	}

	protected void createDialogAreaInCompositeForHTML(Composite composite) {
		// Convert tag name case
		// d257064 need to associate group w/ radio buttons so radio buttons header can be read
		Group tagNameCase = new Group(composite, SWT.NONE);
		tagNameCase.setText(HTMLUIMessages.Tag_name_case_for_HTML_UI_);
		GridLayout hLayout = new GridLayout();
		hLayout.numColumns = 3;
		tagNameCase.setLayout(hLayout);
		fRadioButtonTagNameCaseAsis = new Button(tagNameCase, SWT.RADIO);
		fRadioButtonTagNameCaseAsis.setText(HTMLUIMessages.Tag_name_case_As_is_UI_);
		fRadioButtonTagNameCaseAsis.addSelectionListener(this);
		fRadioButtonTagNameCaseLower = new Button(tagNameCase, SWT.RADIO);
		fRadioButtonTagNameCaseLower.setText(HTMLUIMessages.Tag_name_case_Lower_UI_);
		fRadioButtonTagNameCaseLower.addSelectionListener(this);
		fRadioButtonTagNameCaseUpper = new Button(tagNameCase, SWT.RADIO);
		fRadioButtonTagNameCaseUpper.setText(HTMLUIMessages.Tag_name_case_Upper_UI_);
		fRadioButtonTagNameCaseUpper.addSelectionListener(this);

		// Convert attr name case
		// d257064 need to associate group w/ radio buttons so radio buttons header can be read
		Group attrNameCase = new Group(composite, SWT.NONE);
		attrNameCase.setText(HTMLUIMessages.Attribute_name_case_for_HTML_UI_);
		attrNameCase.setLayout(hLayout);
		fRadioButtonAttrNameCaseAsis = new Button(attrNameCase, SWT.RADIO);
		fRadioButtonAttrNameCaseAsis.setText(HTMLUIMessages.Attribute_name_case_As_is_UI_);
		fRadioButtonAttrNameCaseAsis.addSelectionListener(this);
		fRadioButtonAttrNameCaseLower = new Button(attrNameCase, SWT.RADIO);
		fRadioButtonAttrNameCaseLower.setText(HTMLUIMessages.Attribute_name_case_Lower_UI_);
		fRadioButtonAttrNameCaseLower.addSelectionListener(this);
		fRadioButtonAttrNameCaseUpper = new Button(attrNameCase, SWT.RADIO);
		fRadioButtonAttrNameCaseUpper.setText(HTMLUIMessages.Attribute_name_case_Upper_UI_);
		fRadioButtonAttrNameCaseUpper.addSelectionListener(this);
	}

	protected void createDialogAreaInComposite(Composite composite) {
		if (isHTMLType()) {
			createDialogAreaInCompositeForHTML(composite);
			WorkbenchHelp.setHelp(composite, IHelpContextIds.CLEANUP_HTML_HELPID); // use HTML specific help
		}
		else
			WorkbenchHelp.setHelp(composite, org.eclipse.wst.xml.ui.internal.editor.IHelpContextIds.CLEANUP_XML_HELPID); // use XML specific help

		GridLayout layout = new GridLayout();
		layout.numColumns = 1;
		layout.makeColumnsEqualWidth = true;
		composite.setLayout(layout);

		// Insert missing required attrs
		fCheckBoxInsertRequiredAttrs = new Button(composite, SWT.CHECK);
		fCheckBoxInsertRequiredAttrs.setText(HTMLUIMessages.Insert_required_attributes_UI_);
		fCheckBoxInsertRequiredAttrs.addSelectionListener(this);

		// Insert missing begin/end tags
		fCheckBoxInsertMissingTags = new Button(composite, SWT.CHECK);
		fCheckBoxInsertMissingTags.setText(HTMLUIMessages.Insert_missing_tags_UI_);
		fCheckBoxInsertMissingTags.addSelectionListener(this);

		// Quote attribute values
		fCheckBoxQuoteAttrValues = new Button(composite, SWT.CHECK);
		fCheckBoxQuoteAttrValues.setText(HTMLUIMessages.Quote_attribute_values_UI_);
		fCheckBoxQuoteAttrValues.addSelectionListener(this);

		// Format source
		fCheckBoxFormatSource = new Button(composite, SWT.CHECK);
		fCheckBoxFormatSource.setText(HTMLUIMessages.Format_source_UI_);
		fCheckBoxFormatSource.addSelectionListener(this);

		// Convert EOL code
		fCheckBoxConvertEOLCodes = new Button(composite, SWT.CHECK);
		fCheckBoxConvertEOLCodes.setText(HTMLUIMessages.Convert_EOL_codes_UI_);
		fCheckBoxConvertEOLCodes.addSelectionListener(this);
		Composite EOLCodes = new Composite(composite, SWT.NULL);
		GridLayout hLayout = new GridLayout();
		hLayout.numColumns = 3;
		EOLCodes.setLayout(hLayout);
		fRadioButtonConvertEOLWindows = new Button(EOLCodes, SWT.RADIO);
		fRadioButtonConvertEOLWindows.setText(HTMLUIMessages.EOL_Windows_UI);
		fRadioButtonConvertEOLWindows.addSelectionListener(this);
		fRadioButtonConvertEOLUnix = new Button(EOLCodes, SWT.RADIO);
		fRadioButtonConvertEOLUnix.setText(HTMLUIMessages.EOL_Unix_UI);
		fRadioButtonConvertEOLUnix.addSelectionListener(this);
		fRadioButtonConvertEOLMac = new Button(EOLCodes, SWT.RADIO);
		fRadioButtonConvertEOLMac.setText(HTMLUIMessages.EOL_Mac_UI);
		fRadioButtonConvertEOLMac.addSelectionListener(this);
	}

	protected void okPressed() {
		storeOptions();

		super.okPressed();
	}

	protected void initializeOptionsForHTML() {
		int tagNameCase = getModelPreferences().getInt(HTMLCorePreferenceNames.CLEANUP_TAG_NAME_CASE);
		if (tagNameCase == HTMLCorePreferenceNames.UPPER)
			fRadioButtonTagNameCaseUpper.setSelection(true);
		else if (tagNameCase == HTMLCorePreferenceNames.LOWER)
			fRadioButtonTagNameCaseLower.setSelection(true);
		else
			fRadioButtonTagNameCaseAsis.setSelection(true);

		int attrNameCase = getModelPreferences().getInt(HTMLCorePreferenceNames.CLEANUP_ATTR_NAME_CASE);
		if (attrNameCase == HTMLCorePreferenceNames.UPPER)
			fRadioButtonAttrNameCaseUpper.setSelection(true);
		else if (attrNameCase == HTMLCorePreferenceNames.LOWER)
			fRadioButtonAttrNameCaseLower.setSelection(true);
		else
			fRadioButtonAttrNameCaseAsis.setSelection(true);
	}

	protected void initializeOptions() {
		if (isHTMLType())
			initializeOptionsForHTML();

		fCheckBoxInsertRequiredAttrs.setSelection(getModelPreferences().getBoolean(HTMLCorePreferenceNames.INSERT_REQUIRED_ATTRS));
		fCheckBoxInsertMissingTags.setSelection(getModelPreferences().getBoolean(HTMLCorePreferenceNames.INSERT_MISSING_TAGS));
		fCheckBoxQuoteAttrValues.setSelection(getModelPreferences().getBoolean(HTMLCorePreferenceNames.QUOTE_ATTR_VALUES));
		fCheckBoxFormatSource.setSelection(getModelPreferences().getBoolean(HTMLCorePreferenceNames.FORMAT_SOURCE));
		fCheckBoxConvertEOLCodes.setSelection(getModelPreferences().getBoolean(HTMLCorePreferenceNames.CONVERT_EOL_CODES));
		if (fCheckBoxConvertEOLCodes.getSelection()) {
			String EOLCode = getModelPreferences().getString(HTMLCorePreferenceNames.CLEANUP_EOL_CODE);
			if (EOLCode == CommonEncodingPreferenceNames.LF)
				fRadioButtonConvertEOLUnix.setSelection(true);
			else if (EOLCode == CommonEncodingPreferenceNames.CR)
				fRadioButtonConvertEOLMac.setSelection(true);
			else
				fRadioButtonConvertEOLWindows.setSelection(true);
		}
		enableEOLCodeRadios(fCheckBoxConvertEOLCodes.getSelection());
	}

	protected void storeOptionsForHTML() {
		if (fRadioButtonTagNameCaseUpper.getSelection())
			getModelPreferences().setValue(HTMLCorePreferenceNames.CLEANUP_TAG_NAME_CASE, HTMLCorePreferenceNames.UPPER);
		else if (fRadioButtonTagNameCaseLower.getSelection())
			getModelPreferences().setValue(HTMLCorePreferenceNames.CLEANUP_TAG_NAME_CASE, HTMLCorePreferenceNames.LOWER);
		else
			getModelPreferences().setValue(HTMLCorePreferenceNames.CLEANUP_TAG_NAME_CASE, HTMLCorePreferenceNames.ASIS);

		if (fRadioButtonAttrNameCaseUpper.getSelection())
			getModelPreferences().setValue(HTMLCorePreferenceNames.CLEANUP_ATTR_NAME_CASE, HTMLCorePreferenceNames.UPPER);
		else if (fRadioButtonAttrNameCaseLower.getSelection())
			getModelPreferences().setValue(HTMLCorePreferenceNames.CLEANUP_ATTR_NAME_CASE, HTMLCorePreferenceNames.LOWER);
		else
			getModelPreferences().setValue(HTMLCorePreferenceNames.CLEANUP_ATTR_NAME_CASE, HTMLCorePreferenceNames.ASIS);

		// explicitly save plugin preferences so values are stored
		HTMLCorePlugin.getDefault().savePluginPreferences();
	}

	protected void storeOptions() {
		if (isHTMLType()) {
			storeOptionsForHTML();
			XMLCleanupPreferencesImpl.getInstance().setTagNameCase(getModelPreferences().getInt(HTMLCorePreferenceNames.CLEANUP_TAG_NAME_CASE));
			XMLCleanupPreferencesImpl.getInstance().setAttrNameCase(getModelPreferences().getInt(HTMLCorePreferenceNames.CLEANUP_ATTR_NAME_CASE));
		}
		else {
			XMLCleanupPreferencesImpl.getInstance().setTagNameCase(HTMLCorePreferenceNames.ASIS);
			XMLCleanupPreferencesImpl.getInstance().setAttrNameCase(HTMLCorePreferenceNames.ASIS);
		}

		getModelPreferences().setValue(HTMLCorePreferenceNames.INSERT_REQUIRED_ATTRS, fCheckBoxInsertRequiredAttrs.getSelection());
		getModelPreferences().setValue(HTMLCorePreferenceNames.INSERT_MISSING_TAGS, fCheckBoxInsertMissingTags.getSelection());
		getModelPreferences().setValue(HTMLCorePreferenceNames.QUOTE_ATTR_VALUES, fCheckBoxQuoteAttrValues.getSelection());
		getModelPreferences().setValue(HTMLCorePreferenceNames.FORMAT_SOURCE, fCheckBoxFormatSource.getSelection());
		getModelPreferences().setValue(HTMLCorePreferenceNames.CONVERT_EOL_CODES, fCheckBoxConvertEOLCodes.getSelection());
		XMLCleanupPreferencesImpl.getInstance().setInsertMissingTags(fCheckBoxInsertMissingTags.getSelection());
		XMLCleanupPreferencesImpl.getInstance().setQuoteAttrValues(fCheckBoxQuoteAttrValues.getSelection());
		XMLCleanupPreferencesImpl.getInstance().setFormatSource(fCheckBoxFormatSource.getSelection());
		XMLCleanupPreferencesImpl.getInstance().setConvertEOLCodes(fCheckBoxConvertEOLCodes.getSelection());
		if (fCheckBoxConvertEOLCodes.getSelection()) {
			if (fRadioButtonConvertEOLUnix.getSelection()) {
				getModelPreferences().setValue(HTMLCorePreferenceNames.CLEANUP_EOL_CODE, CommonEncodingPreferenceNames.LF);
				XMLCleanupPreferencesImpl.getInstance().setEOLCode(CommonEncodingPreferenceNames.LF);
			}
			else if (fRadioButtonConvertEOLMac.getSelection()) {
				getModelPreferences().setValue(HTMLCorePreferenceNames.CLEANUP_EOL_CODE, CommonEncodingPreferenceNames.CR);
				XMLCleanupPreferencesImpl.getInstance().setEOLCode(CommonEncodingPreferenceNames.CR);
			}
			else {
				getModelPreferences().setValue(HTMLCorePreferenceNames.CLEANUP_EOL_CODE, CommonEncodingPreferenceNames.CRLF);
				XMLCleanupPreferencesImpl.getInstance().setEOLCode(CommonEncodingPreferenceNames.CRLF);
			}
		}
		
		// explicitly save plugin preferences so values are stored
		HTMLCorePlugin.getDefault().savePluginPreferences();
	}

	public void widgetDefaultSelected(SelectionEvent e) {
		widgetSelected(e);
	}

	public void widgetSelected(SelectionEvent e) {
		getButton(OK).setEnabled((fRadioButtonTagNameCaseLower != null && (fRadioButtonTagNameCaseLower.getSelection() || fRadioButtonTagNameCaseUpper.getSelection())) || (fRadioButtonAttrNameCaseLower != null && (fRadioButtonAttrNameCaseLower.getSelection() || fRadioButtonAttrNameCaseUpper.getSelection())) || fCheckBoxInsertMissingTags.getSelection() || fCheckBoxQuoteAttrValues.getSelection() || fCheckBoxFormatSource.getSelection() || fCheckBoxConvertEOLCodes.getSelection() || (fRadioButtonConvertEOLUnix != null && (fRadioButtonConvertEOLUnix.getSelection() || fRadioButtonConvertEOLMac.getSelection() || fRadioButtonConvertEOLWindows.getSelection())));
		if (e.widget == fCheckBoxConvertEOLCodes)
			enableEOLCodeRadios(fCheckBoxConvertEOLCodes.getSelection());
	}

	public void setModel(IStructuredModel model) {
		fModel = model;
	}

	protected Preferences getModelPreferences() {
		return HTMLCorePlugin.getDefault().getPluginPreferences();
	}

	protected boolean isXMLType() {
		boolean result = false;

		if (fModel != null) {
			IModelHandler modelHandler = fModel.getModelHandler();
			if (modelHandler.getAssociatedContentTypeId().equals(ContentTypeIdForXML.ContentTypeID_XML))
				result = true;
		}

		return result;
	}

	protected boolean isHTMLType() {
		boolean result = true;

		if (fModel != null) {
			IModelHandler modelHandler = fModel.getModelHandler();
			if (modelHandler.getAssociatedContentTypeId().equals(ContentTypeIdForHTML.ContentTypeID_HTML))
				result = true;
		}

		return result;
	}

	protected void enableEOLCodeRadios(boolean enable) {
		if ((fRadioButtonConvertEOLWindows != null) && (fRadioButtonConvertEOLUnix != null) && (fRadioButtonConvertEOLMac != null)) {
			fRadioButtonConvertEOLWindows.setEnabled(enable);
			fRadioButtonConvertEOLUnix.setEnabled(enable);
			fRadioButtonConvertEOLMac.setEnabled(enable);

			if (!fRadioButtonConvertEOLWindows.getSelection() && !fRadioButtonConvertEOLUnix.getSelection() && !fRadioButtonConvertEOLMac.getSelection())
				fRadioButtonConvertEOLWindows.setSelection(true);
		}
	}
}