/*******************************************************************************
 * Copyright (c) 2003, 2004 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 * IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.jst.common.navigator.internal.actions;

import java.util.Iterator;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IStorage;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.jdt.core.IJavaElement;
import org.eclipse.jdt.core.ISourceReference;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jdt.internal.ui.IJavaHelpContextIds;
import org.eclipse.jdt.internal.ui.IJavaStatusConstants;
import org.eclipse.jdt.internal.ui.JavaPlugin;
import org.eclipse.jdt.internal.ui.actions.OpenActionUtil;
import org.eclipse.jdt.internal.ui.actions.SelectionConverter;
import org.eclipse.jdt.internal.ui.javaeditor.JavaEditor;
import org.eclipse.jdt.internal.ui.util.ExceptionHandler;
import org.eclipse.jdt.ui.actions.SelectionDispatchAction;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.text.ITextSelection;
import org.eclipse.jface.util.OpenStrategy;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.ui.IWorkbenchSite;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.texteditor.IEditorStatusLine;
import org.eclipse.wst.common.navigator.internal.provisional.views.AdaptabilityUtility;


/**
 * This action opens a Java editor on a Java element or file.
 * <p>
 * The action is applicable to selections containing elements of type <code>ICompilationUnit</code>,
 * <code>IMember</code> or <code>IFile</code>.
 * 
 * <p>
 * This class may be instantiated; it is not intended to be subclassed.
 * </p>
 * 
 * <B>Derived from the class of the same name in the JDT actions package due to limitations of the
 * existing class concerning how it manages adaptable objects. </B>
 * 
 * @plannedfor 2.0
 */
public class OpenAction extends SelectionDispatchAction {

	protected static final Class ISOURCE_REFERENCE_CLASS = ISourceReference.class;
	protected static final Class IFILE_CLASS = IFile.class;
	protected static final Class ISTORAGE_CLASS = IStorage.class;

	private JavaEditor fEditor;

	/**
	 * Creates a new <code>OpenAction</code>. The action requires that the selection provided by
	 * the site's selection provider is of type <code>
	 * org.eclipse.jface.viewers.IStructuredSelection</code>.
	 * 
	 * @param site
	 *            the site providing context information for this action
	 */
	public OpenAction(IWorkbenchSite site) {
		super(site);
		setText(ActionMessages.OpenAction_label); //$NON-NLS-1$
		setToolTipText(ActionMessages.OpenAction_tooltip); //$NON-NLS-1$
		setDescription(ActionMessages.OpenAction_description); //$NON-NLS-1$	
		PlatformUI.getWorkbench().getHelpSystem().setHelp(this, IJavaHelpContextIds.OPEN_ACTION);
	}

	/**
	 * Note: This constructor is for internal use only. Clients should not call this constructor.
	 */
	public OpenAction(JavaEditor editor) {
		this(editor.getEditorSite());
		fEditor = editor;
		setText(ActionMessages.OpenAction_declaration_label); //$NON-NLS-1$
		setEnabled(SelectionConverter.canOperateOn(fEditor));
	}

	/*
	 * (non-Javadoc) Method declared on SelectionDispatchAction.
	 */
	public void selectionChanged(ITextSelection selection) {
		//Do nothing
	}

	/*
	 * (non-Javadoc) Method declared on SelectionDispatchAction.
	 */
	public void selectionChanged(IStructuredSelection selection) {
		setEnabled(checkEnabled(selection));
	}

	private boolean checkEnabled(IStructuredSelection selection) {
		if (selection.isEmpty())
			return false;
		for (Iterator iter = selection.iterator(); iter.hasNext();)
			if (getElementToOpen(iter.next()) == null)
				return false;
		return true;
	}

	/*
	 * (non-Javadoc) Method declared on SelectionDispatchAction.
	 */
	public void run(ITextSelection selection) {
		try {
			IJavaElement element = SelectionConverter.codeResolve(fEditor, getShell(), getDialogTitle(), ActionMessages.OpenAction_select_element); //$NON-NLS-1$
			if (element == null) {
				IEditorStatusLine statusLine = (IEditorStatusLine) fEditor.getAdapter(IEditorStatusLine.class);
				if (statusLine != null)
					statusLine.setMessage(true, ActionMessages.OpenAction_error_messageBadSelection, null); //$NON-NLS-1$
				getShell().getDisplay().beep();
				return;
			}
			IJavaElement input = SelectionConverter.getInput(fEditor);
			int type = element.getElementType();
			if (type == IJavaElement.JAVA_PROJECT || type == IJavaElement.PACKAGE_FRAGMENT_ROOT || type == IJavaElement.PACKAGE_FRAGMENT)
				element = input;
			run(new Object[]{element});
		} catch (JavaModelException e) {
			showError(e);
		}
	}

	/*
	 * (non-Javadoc) Method declared on SelectionDispatchAction.
	 */
	public void run(IStructuredSelection selection) {
		if (!checkEnabled(selection))
			return;
		run(selection.toArray());
	}

	public void run(Object[] elements) {
		if (elements == null)
			return;
		for (int i = 0; i < elements.length; i++) {
			Object element = elements[i];
			try {
				element = getElementToOpen(element);
				boolean activateOnOpen = fEditor != null ? true : OpenStrategy.activateOnOpen();
				OpenActionUtil.open(element, activateOnOpen);
			} catch (JavaModelException e) {
				JavaPlugin.log(new Status(IStatus.ERROR, JavaPlugin.getPluginId(), IJavaStatusConstants.INTERNAL_ERROR, ActionMessages.OpenAction_error_message, e)); //$NON-NLS-1$

				ErrorDialog.openError(getShell(), getDialogTitle(), ActionMessages.OpenAction_error_messageProblems, //$NON-NLS-1$
							e.getStatus());

			} catch (PartInitException x) {

				String name = null;

				if (element instanceof IJavaElement) {
					name = ((IJavaElement) element).getElementName();
				} else if (element instanceof IStorage) {
					name = ((IStorage) element).getName();
				} else if (element instanceof IResource) {
					name = ((IResource) element).getName();
				}

				if (name != null) {
					MessageDialog.openError(getShell(), ActionMessages.OpenAction_error_messageProblems, //$NON-NLS-1$
								ActionMessages.bind(ActionMessages.OpenAction_error_messageArgs, //$NON-NLS-1$
											new String[]{name, x.getMessage()}));
				}
			}
		}
	}

	/**
	 * Note: this method is for internal use only. Clients should not call this method.
	 */
	public Object getElementToOpen(Object object) {
		Object openableObject = null;
		if ((openableObject = AdaptabilityUtility.getAdapter(object, ISOURCE_REFERENCE_CLASS)) != null)
			return openableObject;
		if ((openableObject = AdaptabilityUtility.getAdapter(object, IFILE_CLASS)) != null)
			return openableObject;
		if ((openableObject = AdaptabilityUtility.getAdapter(object, ISTORAGE_CLASS)) != null)
			return openableObject;
		// will be null if this line is reached
		return openableObject;
	}

	private String getDialogTitle() {
		return ActionMessages.OpenAction_error_title; //$NON-NLS-1$
	}

	private void showError(CoreException e) {
		ExceptionHandler.handle(e, getShell(), getDialogTitle(), ActionMessages.OpenAction_error_message); //$NON-NLS-1$
	}
}