/***************************************************************************************************
 * Copyright (c) 2003, 2004 IBM Corporation and others. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: IBM Corporation - initial API and implementation
 **************************************************************************************************/
package org.eclipse.wst.common.navigator.internal.provisional.views;

import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.IStatusLineManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.ui.IMemento;
import org.eclipse.wst.common.navigator.internal.views.extensions.CommonOpenService;

/**
 * <p>
 * Manages the non-viewer responsibilities of the Common Navigator View Part, including the display
 * and population of the context menu and the registration of extensions for opening content. 
 * </p>
 * <p>
 * This class is not intended to be instantiated or subclassed by clients
 * </p> 
 * <p>
 * The following class is experimental until fully documented.
 * </p>
 */
public class CommonNavigatorManager implements ISelectionChangedListener {

	private final CommonNavigator commonNavigator;
	private final NavigatorContentService contentService;
	private NavigatorActionService actionService;
	private final CommonOpenService commonOpenService;
	private IDescriptionProvider commonDescriptionProvider;
	private IStatusLineManager statusLineManager;
	private ILabelProvider labelProvider;


	/**
	 * <p>
	 * Adds listeners to aNavigator to listen for selection changes and respond to mouse events.
	 * </p>
	 * 
	 * @param aNavigator
	 *            The CommonNavigator managed by this class. Requires a non-null value.
	 */
	public CommonNavigatorManager(CommonNavigator aNavigator) {
		super();
		commonNavigator = aNavigator;
		contentService = commonNavigator.getNavigatorContentService();
		commonOpenService = new CommonOpenService(contentService, commonNavigator);
		statusLineManager = commonNavigator.getViewSite().getActionBars().getStatusLineManager();
		commonDescriptionProvider = new NavigatorContentServiceDescriptionProvider(contentService);
		labelProvider = (ILabelProvider) commonNavigator.getCommonViewer().getLabelProvider();
		init();
	}

	private void init() {
		commonNavigator.getCommonViewer().addSelectionChangedListener(this);
		updateStatusBar(commonNavigator.getCommonViewer().getSelection());
		actionService = new NavigatorActionService(commonNavigator, commonNavigator.getCommonViewer(), contentService);
		commonNavigator.getCommonViewer().addOpenListener(commonOpenService);
		initContextMenu();
	}

	/**
	 * <p>Called by {@see CommonNavigator} when the View Part is disposed.
	 *
	 */
	protected void dispose() {
		commonNavigator.getCommonViewer().removeSelectionChangedListener(this);
		commonNavigator.getCommonViewer().removeOpenListener(commonOpenService);
		commonOpenService.dispose();
		actionService.dispose();
	}

	/** 
	 * 
	 * @param anEvent An event indicating the current selection of the {@see CommonViewer}
	 * 
	 * @see org.eclipse.jface.viewers.ISelectionChangedListener#selectionChanged(org.eclipse.jface.viewers.SelectionChangedEvent)
	 */
	public void selectionChanged(SelectionChangedEvent anEvent) {
		updateStatusBar(anEvent.getSelection());
		if (anEvent.getSelection() instanceof IStructuredSelection) {
			IStructuredSelection structuredSelection = (IStructuredSelection) anEvent.getSelection();
			actionService.fillActionBars(commonNavigator.getViewSite().getActionBars(), structuredSelection);
		}
	}


	/**
	 * @param aMemento Used to restore state of action extensions via the {@see NavigatorActionService}. 
	 */
	protected void restoreState(IMemento aMemento) {
		actionService.restoreState(aMemento);
	}

	/**
	 * @param aMemento Used to save state of action extensions via the {@see NavigatorActionService}.
	 */
	protected void saveState(IMemento aMemento) {
		actionService.saveState(aMemento);
	}

	/**
	 * <p>
	 * Fills aMenuManager with menu contributions from the {@see NavigatorActionService}.
	 * </p>
	 * @param aMenuManager A popup menu
	 * @see NavigatorActionService#fillContextMenu(IMenuManager, IStructuredSelection)
	 *  
	 */
	protected void fillContextMenu(IMenuManager aMenuManager) {
		ISelection selection = commonNavigator.getCommonViewer().getSelection();
		if (selection instanceof IStructuredSelection)
			actionService.fillContextMenu(aMenuManager, (IStructuredSelection) selection);
		else
			actionService.fillContextMenu(aMenuManager, StructuredSelection.EMPTY);
	}


	/**
	 * <p>
	 * Initializes and registers the context menu.
	 * </p>
	 */
	protected void initContextMenu() {
		MenuManager menuMgr = new MenuManager(contentService.getViewerDescriptor().getPopupMenuId());
		menuMgr.setRemoveAllWhenShown(true);
		menuMgr.addMenuListener(new IMenuListener() {

			public void menuAboutToShow(IMenuManager manager) {
				fillContextMenu(manager);
			}
		});
		TreeViewer commonViewer = commonNavigator.getCommonViewer();
		Menu menu = menuMgr.createContextMenu(commonViewer.getTree());

		commonViewer.getTree().setMenu(menu);

		/* Hooks into the Eclipse framework for Object contributions, and View contributions. */
		commonNavigator.getSite().registerContextMenu(contentService.getViewerDescriptor().getPopupMenuId(), menuMgr, commonViewer);
	}

	/**
	 * @param aSelection The current selection from the {@see CommonViewer}
	 */
	protected void updateStatusBar(ISelection aSelection) {

		Image img = null;
		if(aSelection != null && !aSelection.isEmpty() && aSelection instanceof IStructuredSelection)
			img =  labelProvider.getImage( ((IStructuredSelection) aSelection).getFirstElement());
		
		statusLineManager.setMessage(img, commonDescriptionProvider.getDescription(aSelection));
	}



}