/*******************************************************************************
 * Copyright (c) 2001, 2004 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.wst.rdb.data.internal.core.editor;

import java.io.*;
import java.sql.*;
import java.util.*;

import org.eclipse.emf.common.util.*;
import org.eclipse.wst.rdb.data.internal.core.*;
import org.eclipse.wst.rdb.data.internal.core.common.*;
import org.eclipse.wst.rdb.internal.core.*;
import org.eclipse.wst.rdb.internal.core.definition.*;
import org.eclipse.wst.rdb.internal.core.rte.*;
import org.eclipse.wst.rdb.internal.core.util.*;
import org.eclipse.wst.rdb.internal.models.sql.constraints.*;
import org.eclipse.wst.rdb.internal.models.sql.datatypes.*;
import org.eclipse.wst.rdb.internal.models.sql.schema.*;
import org.eclipse.wst.rdb.internal.models.sql.tables.*;
import org.eclipse.wst.rdb.internal.outputview.*;


/**
 * Implementation of the ISqlTableData based on JDBC 1.0 API (should work with any driver).
 * The table is queried through a 'select * from ...' statement and the data is stored in a bidimensional
 * array of objects.
 * Reading data is done by directly accessing the im-memory array.
 * Modifying the data is done by issuing insert/update/delete statements, using one of the possible unique
 * constraints of the table to uniquely identify the row.
 * If no unique constraints is available, all the columns will be used, which doesn't garentee that a single
 * column will be modified.
 * 
 * @author groux
 */
public class TableDataImpl implements ITableData {

    protected Table sqlTable;
    protected Connection con;  
    
    /** Vector<SqlRowImpl> Table data. */
    protected Vector rows = new Vector();  
    
    /** Column types as definies in java.sql.Types. */
    protected int[] colTtypes;
    
    /** Column names. */
    protected String[] colNames;
    
    /** Indices of the columns that will be used as part of the key for the insert/update/delete statements. 
     * Only if readonly == false; */
    protected int[] key = null;
    
    /** The table is readonly because it is not a base table (view). */
    protected boolean readonly;
    
    

    public TableDataImpl(Table sqlTable) throws SQLException, IOException
    {
        super();
        this.sqlTable = sqlTable;        
        con = ((ICatalogObject)sqlTable).getConnection();
        
        if (sqlTable instanceof BaseTable) {
            findKey((BaseTable) sqlTable);
            readonly = false;
        } else {
            readonly = true;
        }
        
        Statement stmt = con.createStatement();
        boolean setLimit = RDBCorePlugin.getDefault().getPluginPreferences().getBoolean(RDBCorePluginConstants.LIMIT_ROWS_RETRIEVED);
        if (setLimit) {
            int integer = RDBCorePlugin.getDefault().getPluginPreferences().getInt(RDBCorePluginConstants.MAX_ROW_RETRIEVED);
            stmt.setMaxRows(integer);
        } else {
            stmt.setMaxRows(0);
        }
        ResultSet rs = stmt.executeQuery("SELECT * FROM " + getQualifiedTableName()); //$NON-NLS-1$
         
        ResultSetMetaData rsmd = rs.getMetaData();
        int cc = rsmd.getColumnCount();
        colTtypes = new int[cc];
        colNames = new String[cc];
        for (int i=0; i<cc; ++i) {
            colTtypes[i] = rsmd.getColumnType(i+1);
            colNames[i] = rsmd.getColumnName(i+1);       
        }
        
        int lobLimit = RDBCorePlugin.getDefault().getPluginPreferences().getInt(RDBCorePluginConstants.MAX_LOB_LENGTH);
        if (lobLimit<=0)
            lobLimit = -1;
        
        while (rs.next()) {
            Object[] a = new Object[cc];
            for (int col=0; col<cc; ++col)
                a[col] = ResultSetReader.read(rs, col, lobLimit);
            RowDataImpl row = new RowDataImpl(this, RowDataImpl.STATE_ORIGINAL, a);
            rows.add(row);
        }
        rs.close();
        stmt.close();
    }
    
    protected void findKey(BaseTable baseTable)
    {
        EList constraints = baseTable.getConstraints();
        
        ReferenceConstraint chosenConstr = null;
        Iterator it = constraints.iterator();
        while (it.hasNext()) {
            TableConstraint constr = (TableConstraint)it.next();
            if (constr instanceof UniqueConstraint) {
                UniqueConstraint uniqueConstr = (UniqueConstraint)constr;
	            if (chosenConstr==null || uniqueConstr.getMembers().size()<chosenConstr.getMembers().size())
	                chosenConstr = uniqueConstr;
            }
        }
            
        if (chosenConstr==null) {
            EList cols = sqlTable.getColumns();
            key = new int[cols.size()];
            for (int i=0; i<cols.size(); ++i) {
                key[i] = i;
            } 
        } else {
	        EList cols = chosenConstr.getMembers();
	        key = new int[cols.size()];
	        for (int i=0; i<cols.size(); ++i) {
	            Column col = (Column)cols.get(i);
	            key[i] = col.getTable().getColumns().indexOf(col);
	        }          
        }
    }
    
    public void dispose() {
    }
    
    public int getColumnCount() {
        return sqlTable.getColumns().size();
    }
    
    public String getColumnHeader(int col) {
        Column sqlCol = (Column) sqlTable.getColumns().get(col);
        return sqlCol.getName() + " [" + getFormattedTypeName(sqlCol) + "]";  //$NON-NLS-1$//$NON-NLS-2$
    }
    
    protected static String getFormattedTypeName(Column sqlCol)
    {
        Table table = sqlCol.getTable();
        Schema schema = table.getSchema();
        Database db = schema.getDatabase();
        DatabaseDefinition dbDef = RDBCorePlugin.getDefault().getDatabaseDefinitionRegistry().getDefinition(db);
        DataType dt = sqlCol.getDataType();
        if (dt != null) {
            if (dt instanceof PredefinedDataType)
                return dbDef .getPredefinedDataTypeFormattedName((PredefinedDataType) dt);
            else
                return dt.getName();
        } else {
            return ""; //$NON-NLS-1$
        }
    }
    
    public String getColumnName(int col)
    {
        Column sqlCol = (Column) sqlTable.getColumns().get(col);
        return sqlCol.getName();
    }
    
    public String getQuotedColumnName(int col)
    {
        return DataCorePlugin.quoteIdentifier(sqlTable, getColumnName(col));
    }
    
    public int getColumnType(int col)
    {
        return colTtypes[col];
    }
    
    public Vector getRows()
    {
        Vector v = new Vector();
        Iterator it = rows.iterator();
        while (it.hasNext()) {
            RowDataImpl row = (RowDataImpl)it.next();
            if (row.getState()!=RowDataImpl.STATE_DELETED)
                v.add(row);
        }
        return v;
    }
    
    public int[] getKeyColumns() {
        return key;
    }
    
    public Connection getConnection() {
        return con;
    }
    
    public boolean save() throws SQLException
    {        
        boolean autocomit = con.getAutoCommit();
        con.setAutoCommit(false);
        con.commit();
        
        boolean res;
        Exception resEx = null;
        TableDataSaveStatus status = new TableDataSaveStatus();
        try {         
	        Iterator it = rows.iterator();
	        while (it.hasNext()) {
	            RowDataImpl row = (RowDataImpl)it.next();
	            row.save(status);
	        }
	        con.commit();
	        con.setAutoCommit(autocomit);
	        res = true;
        } catch (Exception ex) {
            con.rollback();
            con.setAutoCommit(autocomit);
            res = false;
            resEx = ex;
            status.reset();
        }
        
        if (res)
            resetRowsToOriginal();
        
        writeOutputView(res, resEx, status);

        return res;
    }
    
    public void revert()
    {
        int i = 0;
        while (i<rows.size()) {
            RowDataImpl row = (RowDataImpl)rows.elementAt(i);
            if (row.getState()==RowDataImpl.STATE_UPDATED || row.getState()==RowDataImpl.STATE_DELETED) {
                row.revertToOriginal();
                ++i;
            } else if (row.getState()==RowDataImpl.STATE_INSERTED)
                rows.remove(i);
            else if (row.getState()==RowDataImpl.STATE_ORIGINAL)
                ++i;
        }
    }
    
    protected void resetRowsToOriginal()
    {
        int i = 0;
        while (i<rows.size()) {
            RowDataImpl row = (RowDataImpl)rows.elementAt(i);
            if (row.getState()==RowDataImpl.STATE_UPDATED || row.getState()==RowDataImpl.STATE_INSERTED) {
                row.resetToOriginal();
                ++i;
            } else if (row.getState()==RowDataImpl.STATE_DELETED)
                rows.remove(i);
            else if (row.getState()==RowDataImpl.STATE_ORIGINAL)
                ++i;
        }  
    }
    
    protected void writeOutputView(boolean res, Exception resEx, TableDataSaveStatus status)
    {
        String endl = System.getProperty("line.separator"); //$NON-NLS-1$
        
        int statusCode;
        if (res)
            statusCode = (status.duplicateRow) ? OutputItem.STATUS_WARNING : OutputItem.STATUS_SUCCESS;
        else 
            statusCode = OutputItem.STATUS_FAILURE; 
        
        OutputItem item = OutputViewAPI.getInstance().findOutputItem(getQualifiedTableName(), OutputItem.ACTION_EDIT, true);
        if (item==null) {
            item = new OutputItem(statusCode, OutputItem.ACTION_EDIT, sqlTable.getName(), getQualifiedTableName());
            OutputViewAPI.getInstance().addOutputItem(item, true);            
        } else {
            OutputViewAPI.getInstance().resetOutputItem(item, true);
            OutputViewAPI.getInstance().updateStatus(item, statusCode, true);
        }
        
        if (res)
            OutputViewAPI.getInstance().showMessage(item, Messages.getString("TableDataImpl.DataSuccessfullySaved"), true); //$NON-NLS-1$
        else
            OutputViewAPI.getInstance().showMessage(item, Messages.getString("TableDataImpl.ErrorSavingData"), true); //$NON-NLS-1$
        
        if (resEx!=null)
            OutputViewAPI.getInstance().showMessage(item, resEx.toString(), true);
        
        if (status.duplicateRow)
            OutputViewAPI.getInstance().showMessage(item, Messages.getString("TableDataImpl.DuplicateRows"), true); //$NON-NLS-1$
        
        String msg = ""; //$NON-NLS-1$
        msg += Messages.getString("TableDataImpl.Inserted") + String.valueOf(status.inserted) + Messages.getString("TableDataImpl.rows") + endl; //$NON-NLS-1$ //$NON-NLS-2$
        msg += Messages.getString("TableDataImpl.Updated") + String.valueOf(status.updated) + Messages.getString("TableDataImpl.rows") + endl; //$NON-NLS-1$ //$NON-NLS-2$
        msg += Messages.getString("TableDataImpl.Deleted") + String.valueOf(status.deleted) + Messages.getString("TableDataImpl.rows"); //$NON-NLS-1$ //$NON-NLS-2$
        OutputViewAPI.getInstance().showMessage(item, msg, true);
    }

    public void deleteRow(IRowData row)
    {
        if (((RowDataImpl)row).getState()==RowDataImpl.STATE_INSERTED)
            rows.remove(row);
        else
            ((RowDataImpl)row).setState(RowDataImpl.STATE_DELETED);
            
        
    }
    
    public IRowData insertRow() {
        Object data[] = new Object[getColumnCount()];
        IRowData row = new RowDataImpl(this, RowDataImpl.STATE_INSERTED, data);
        rows.add(row);
        return row;
    }
    
    public boolean isReadonly() {
        return readonly;
    }
    
    public String getQualifiedTableName()
    {
        return DataCorePlugin.getQualifiedTableName(sqlTable);
    }
}
