/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.wst.rdb.sqleditor.internal.actions;

import java.sql.Connection;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.ResourceBundle;

import org.eclipse.jface.text.BadLocationException;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.ITextSelection;
import org.eclipse.jface.text.ITypedRegion;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.texteditor.IDocumentProvider;
import org.eclipse.ui.texteditor.ResourceAction;
import org.eclipse.wst.rdb.internal.core.connection.ConnectionInfo;
import org.eclipse.wst.rdb.server.internal.ui.query.execute.QueryOutputHelper;
import org.eclipse.wst.rdb.sqleditor.internal.SQLEditor;
import org.eclipse.wst.rdb.sqleditor.internal.sql.SQLPartitionScanner;
import org.eclipse.wst.rdb.sqleditor.internal.utils.SQLDBUtils;
import org.eclipse.wst.rdb.sqleditor.internal.utils.SQLStatementTerminatorSupport;

/**
 * This class implements the "Run SQL" action for the SQL Editor.
 * 
 * @see org.eclipse.wst.rdb.sqleditor.internal.SQLEditorActionContributor
 * @see org.eclipse.ui.texteditor.ResourceAction
 */
public class SQLRunAction extends ResourceAction {
    private SQLEditor fSQLEditor;
    
    /**
     * Creates an instance of this class using the given resource bundle
     * and prefix of a set of resources within the bundle.
     * 
     * @param bundle the resource bundle to use
     * @param prefix the resource prefix to use
     */
    public SQLRunAction( ResourceBundle bundle, String prefix ) {
        super( bundle, prefix );
    }
    
    /**
     * Gets the SQLEditor instance associated with this action.
     * 
     * @return the SQLEditor instance for this action
     */
    public SQLEditor getSQLEditor() {
        return fSQLEditor;
    }

    /**
     * Gets the SQL statements contained in the given document, as a list of
     * strings.  
     * 
     * @return the list of SQL statements
     */
    protected List getSQLStatementList( ) {
        List stmtList = new ArrayList();
 
        // Get the document from the SQL Editor.
        SQLEditor sqlEditor = getSQLEditor();
        IEditorInput editorInput = sqlEditor.getEditorInput();
        IDocumentProvider docProvider = sqlEditor.getDocumentProvider();
        IDocument doc = docProvider.getDocument( editorInput );
        
        // Get an array of document regions.
        ITypedRegion[] regions = SQLPartitionScanner.getDocumentRegions( doc );

        // Get the current statement terminator.  The statement terminator to use
        // might be imbedded in the document itself as an XML-like tag.
        SQLStatementTerminatorSupport statementTerminatorSupport = sqlEditor.getSQLStatementTerminatorSupport();
        String statementTerminator = statementTerminatorSupport.getStatementTerminator();

        // Process the regions.  Build up a statement string until we find a
        // statement terminator, whereupon we add that statement string to the
        // list of statements and start another statement string.
        if (regions != null) {
            StringBuffer stmtBuf = new StringBuffer();
            ITypedRegion region = null;
            
            // Process each region in turn.
            for (int i=0; i<regions.length; i++) {
                // Get the next region.
                region = regions[ i ];
                
                // Ignore it if it's a comment regions, otherwise process it.
                if (!(region.getType().equals( SQLPartitionScanner.SQL_COMMENT ))) {
                    // Get the text of the region from the document.
                    int regionOffset = region.getOffset();
                    int regionLength = region.getLength();
                    String stmtPart = "";
                    try {
                       stmtPart = doc.get( regionOffset, regionLength );
                    }
                    catch (BadLocationException e ) {
                        // ignore, shouldn't happen
                    }
                    
                    // For quoted literals and delimited identifiers, simply add 
                    // them to the current statement string.
                    if ( (region.getType().equals( SQLPartitionScanner.SQL_QUOTED_LITERAL ))
                      || (region.getType().equals( SQLPartitionScanner.SQL_DELIMITED_IDENTIFIER ))
                       ) {
                        stmtBuf.append( stmtPart );
                    }
                    // Otherwise it must be a SQL code region.  Scan it for statement
                    // terminators and break it up accordingly.
                    else {
                        // Look for a terminator.
                        int scanStart = 0;
                        int terminatorPos = stmtPart.indexOf( statementTerminator, scanStart );
                        while (terminatorPos != -1 && scanStart < stmtPart.length()) {
                            // Add this part of the statement to the buffer, add the statement
                            // to the statement list, clear the buffer, and set up to scan again.
                            stmtBuf.append( stmtPart.substring( scanStart, terminatorPos ));
                            String stmt = stmtBuf.toString();
                            stmt = stmt.trim();
                            if (stmt.length() > 0) {
                                stmtList.add( stmt );
                            }
                            stmtBuf.setLength( 0 );
                            
                            // Look for another terminator.
                            scanStart = terminatorPos + statementTerminator.length();
                            if (scanStart < stmtPart.length()) {
                                terminatorPos = stmtPart.indexOf( statementTerminator, scanStart );
                            }
                        }
                        // Append any leftover text to the statement buffer.
                        if (scanStart < stmtPart.length()) {
                            stmtBuf.append( stmtPart.substring( scanStart, stmtPart.length() ));
                        }
                    }
                }
            }
            
            // Make sure the last statement buffer is added to the list.
            if (stmtBuf.length() > 0) {
                String stmt = stmtBuf.toString();
                stmt = stmt.trim();
                if (stmt.length() > 0) {
                    stmtList.add( stmt );
                }
            }
        }
        
        return stmtList;
    }
    
    /**
     * Runs this action.  Runs the SQL statements contained in the SQL editor
     * document with the connected database.
     * 
     * @see org.eclipse.jface.action.IAction#run()
     */
    public void run() {
        SQLEditor sqlEditor = getSQLEditor();
        if (sqlEditor != null) {
            // Create an empty list of SQL statements to start.
            List stmtList = new ArrayList();
            
            // Get the selected text in the editor, if any, as the SQL 
            // statement to run, and add it to the statement list.
            ISelectionProvider selectionProvider = fSQLEditor.getSelectionProvider();
            ISelection selection = selectionProvider.getSelection();
            if (selection.isEmpty() == false) {
                if (selection instanceof ITextSelection) {
                    ITextSelection textSelection = (ITextSelection) selection;
                    String text = textSelection.getText();
                    if (text.length() > 0) {
                        stmtList.add( text );
                    }
                }
            }
            
            // If there's no selection, use the entire document.  Search the
            // document for one or more SQL statements that it might contain
            // and add them to the statement list.
            if (stmtList.size() == 0) {
                
                stmtList = getSQLStatementList();
            }

            // If we have at least one statement to run, get the connection
            // and try to run them.
            if (stmtList.size() > 0) {
                Connection conn = null;
                ConnectionInfo connInfo = fSQLEditor.getConnectionInfo();
                
                // If we don't have a connection, prompt the user for one.
                if (connInfo == null) {
                    connInfo = sqlEditor.requestConnectionFromUser();
                    // Remember the connection for next time.
                    if (connInfo != null) {
                        sqlEditor.setConnectionInfo( connInfo );
                    }
                }
                
                // If we have the connection info, make sure that the connection
                // is established.
                if (connInfo != null) {
                    boolean connected = SQLDBUtils.reestablishConnection( connInfo );
                    if (connected == true) {
                        conn = connInfo.getSharedConnection();
                    }
                }
                    
                // If we were able to get a connection, run each SQL statement and
                // display the results in the output view.
                if (conn != null) {
                    Iterator stmtListIter = stmtList.iterator();
                    while (stmtListIter.hasNext()) {
                        // Get and run the next statement.
                        String sql = (String) stmtListIter.next();
                        QueryOutputHelper outputHelper = new QueryOutputHelper( sql, conn );
                        outputHelper.runSQLScrapbookQuery();
                        
                        // Make sure the output view is open.
                        try {
                            PlatformUI.getWorkbench().getActiveWorkbenchWindow().getActivePage().showView("org.eclipse.wst.rdb.internal.outputview.OutputView");
                        } catch (PartInitException e) {
                        }
                    }
                }
            }
        }
    }

    /**
     * Sets the SQLEditor instance associated with this action to the given object.
     * 
     * @param sqlEditor the SQLEditor instance for this action
     */
    public void setSQLEditor( SQLEditor sqlEditor ) {
        fSQLEditor = sqlEditor;
    }
}
