/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.wst.rdb.sqleditor.internal.actions;

import java.util.ResourceBundle;

import org.eclipse.jface.window.Window;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.texteditor.ResourceAction;
import org.eclipse.wst.rdb.sqleditor.internal.SQLEditor;
import org.eclipse.wst.rdb.sqleditor.internal.utils.SQLStatementTerminatorSupport;

/**
 * This class implements the "Set statement terminator" action for the SQL Editor.
 * The statement terminator is a string (typically a single character) that separates 
 * SQL statements in the SQL source text.
 * 
 * @see org.eclipse.wst.rdb.sqleditor.internal.SQLEditorActionContributor
 * @see org.eclipse.ui.texteditor.ResourceAction
 */
public class SQLSetStatementTerminatorAction extends ResourceAction {
    
    /** The SQL Editor with with this action is associated */
    private SQLEditor fSQLEditor;
    
    /**
     * Creates an instance of this class using the given resource bundle
     * and prefix of a set of resources within the bundle.
     * 
     * @param bundle the resource bundle to use
     * @param prefix the resource prefix to use
     */
    public SQLSetStatementTerminatorAction( ResourceBundle bundle, String prefix ) {
        super( bundle, prefix );
    }

    /**
     * Gets the <code>SQLEditor</code> instance associated with this action.
     * 
     * @return the <code>SQLEditor</code> instance for this action
     */
    public SQLEditor getSQLEditor() {
        return fSQLEditor;
    }

    /**
     * Runs this action.  Prompts the user to specify the statement terminator
     * string to use with this editor.
     * 
     * @see org.eclipse.jface.action.IAction#run()
     */
    public void run() {
        SQLEditor sqlEditor = getSQLEditor();
        SQLStatementTerminatorSupport statementTerminatorSupport = sqlEditor.getSQLStatementTerminatorSupport();
        String statementTerminator = statementTerminatorSupport.getStatementTerminator();
        
        // Open a dialog in which the user can specify the statement terminator.
        Shell shell = Display.getCurrent().getActiveShell();
        SQLSetStatementTerminatorDialog dialog = new SQLSetStatementTerminatorDialog( shell, statementTerminator );
        int result = dialog.open();
        if (result == Window.OK) {
            // Get the new statement terminator string and inform listeners about
            // the change.  Update the SQL Editor with the new statement terminator.
            String newStatementTerminator = dialog.getValue();
            firePropertyChange( SQLStatementTerminatorSupport.STATEMENT_TERMINATOR, statementTerminator, newStatementTerminator );
        }
    }

    /**
     * Sets the <code>SQLEditor</code> instance associated with this action to the given object.
     * 
     * @param sqlEditor the <code>SQLEditor</code> instance for this action
     */
    public void setSQLEditor( SQLEditor sqlEditor ) {
        fSQLEditor = sqlEditor;
    }

} // end class
