/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - Initial API and implementation
 *******************************************************************************/
package org.eclipse.jst.server.geronimo.core.internal;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.jar.JarEntry;
import java.util.jar.JarOutputStream;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Status;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.debug.core.ILaunch;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationType;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.debug.core.ILaunchManager;
import org.eclipse.debug.core.model.IProcess;
import org.eclipse.jdt.launching.IJavaLaunchConfigurationConstants;
import org.eclipse.jdt.launching.IVMInstall;
import org.eclipse.jst.server.core.IJ2EEModule;
import org.eclipse.jst.server.generic.core.internal.GenericServerBehaviour;
import org.eclipse.jst.server.generic.core.internal.GenericServerRuntime;
import org.eclipse.jst.server.generic.core.internal.Trace;
import org.eclipse.wst.server.core.IModule;
import org.eclipse.wst.server.core.IServer;

public class GeronimoServerBehaviour extends GenericServerBehaviour {

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.wst.server.core.model.ServerBehaviourDelegate#stop(boolean)
	 *      There's no Java API to stop Geronimo (short of a JMX invocation), so
	 *      always just kill the process - the server will handle the signal and
	 *      exit cleanly
	 */
	public void stop(boolean force) {
		super.stop(true);
	}
	
	/*
	 * (non-Javadoc)
	 * 
	 * This method is overridden due to the lack of clean communication with
	 * the server.  Because we (currently) use external commands to start the 
	 * server, the only way to know that it's up is when the web port becomes 
	 * available.  The problem is that this port actually becomes available 
	 * before the server is completely up, and any immediate attempt to 
	 * deploy an app will fail.  
	 * 
	 * The oh-I-hate-this workaround is to add a delay in this method.  The 
	 * method is invoked from org.eclipse.jst.server.generic.core.internal.PingThread, 
	 * once the port is up.  By adding an (arbitary) delay of 10 seconds in here, 
	 * the goal is to allow the server to complete its startup before trying 
	 * any publishing.  Any contributions to this code, ideally to better determine
	 * when the server is really ready for publishing to, will be most greatfully 
	 * received.
	 */
 	protected void setServerStarted() {
  		try {
			Thread.sleep(10000);		// TODO yuck, see above comment block
			String hello="hello";
			hello.trim();
		} catch (InterruptedException e) {
			Trace.trace(Trace.WARNING, "Server start wait failed");
		}		
 		setServerState(IServer.STATE_STARTED);
 	}

	public void publishModule(int kind, int deltaKind, IModule[] module,
			IProgressMonitor monitor) throws CoreException {

		// Can only publish when the server is running
		int state = getServer().getServerState();
		if (state == IServer.STATE_STOPPED || state == IServer.STATE_STOPPING) {
			throw new CoreException(Status.CANCEL_STATUS);
		}
		
		if(state == IServer.STATE_STARTING)
		{
			int timeout=25;
			while(getServer().getServerState()== IServer.STATE_STARTING)
			{
				if(--timeout==0)
					throw new CoreException(Status.CANCEL_STATUS);
				try {
					Thread.sleep(1000);
				} catch (InterruptedException e) {
					e.printStackTrace();
				}
			}
		}
		
		if(deltaKind == NO_CHANGE) 	// Temporary workaround for WTP server tools bug 
			deltaKind = CHANGED;
		
		if (!(deltaKind == ADDED || deltaKind == REMOVED || deltaKind == CHANGED))
			return;

		
		String[] commands = getCommands(deltaKind, module[module.length - 1]);
		for(int i=0; i<commands.length; i++)
		{
			// TODO: This is not ideal; we should really execute these commands 
			// within this JVM - but invoke an external command for now...
			executeExternalCommand(commands[i], true);
		}
	}
	
	private String[] getCommands(int deltaKind, IModule module) {
		StringBuffer command = new StringBuffer();
		IJ2EEModule j2eeModule = (IJ2EEModule) module.loadAdapter(IJ2EEModule.class, null);
		String moduleId=GeronimoUtils.getConfigId(module);
		GeronimoUtils.copyDeploymentPlanToDeployable(module);	// Tsemporary

		switch (deltaKind) {
		case ADDED: {
			String[] result=new String[1];
			File jarFile = createJarFile(j2eeModule.getLocation());
			command.append(" deploy ");
			command.append(jarFile.getAbsolutePath());
			result[0]=command.toString();
			return result;
		}

		case CHANGED: {
			String[] result=new String[2];
			File jarFile = createJarFile(j2eeModule.getLocation());
			
			command.append(" undeploy ");
			command.append(moduleId);
			result[0]=command.toString();
			
			command.setLength(0);
			command.append(" deploy ");
			command.append(jarFile.getAbsolutePath());
			result[1]=command.toString();
			
			return result;
		}

		case REMOVED: {
			String[] result=new String[1];
			command.append(" undeploy ");
			command.append(moduleId);
			result[0]=command.toString();
			return result;
		}

		default:
			throw new IllegalArgumentException();

		}
	}
	
		
	private int executeExternalCommand(String command, boolean blockForCompletion)
	{
		int rc=0;
	
		try {
			Trace.trace(Trace.FINEST, "Publishing module");
			ILaunchManager mgr = DebugPlugin.getDefault().getLaunchManager();

			ILaunchConfigurationType type = mgr
					.getLaunchConfigurationType(IJavaLaunchConfigurationConstants.ID_JAVA_APPLICATION);

			String launchName = "GeronimoServerPublisher";
			String uniqueLaunchName = mgr
					.generateUniqueLaunchConfigurationNameFrom(launchName);
			ILaunchConfiguration conf = null;

			ILaunchConfiguration[] lch = mgr.getLaunchConfigurations(type);
			for (int i = 0; i < lch.length; i++) {
				if (launchName.equals(lch[i].getName())) {
					conf = lch[i];
					break;
				}
			}

			ILaunchConfigurationWorkingCopy wc = null;
			if (conf != null) {
				wc = conf.getWorkingCopy();
			} else {
				wc = type.newInstance(null, uniqueLaunchName);
			}
			
			// To stop from appearing in history lists
			// wc.setAttribute(IDebugUIConstants.ATTR_PRIVATE, true);
			// TODO: IDebugUIConstants

			wc.setAttribute(
					IJavaLaunchConfigurationConstants.ATTR_MAIN_TYPE_NAME,
					getServerDefinition().getResolver()
							.resolveProperties(
									this.getServerDefinition().getStop()
											.getMainClass()));

			GenericServerRuntime runtime = getRuntimeDelegate();

			IVMInstall vmInstall = runtime.getVMInstall();
			wc.setAttribute(
					IJavaLaunchConfigurationConstants.ATTR_VM_INSTALL_TYPE,
					runtime.getVMInstallTypeId());
			wc.setAttribute(
					IJavaLaunchConfigurationConstants.ATTR_VM_INSTALL_NAME,
					vmInstall.getName());

			setupLaunchClasspath(wc, vmInstall, getPublishClasspath());
			// Use the last module in the path
			wc.setAttribute(
					IJavaLaunchConfigurationConstants.ATTR_WORKING_DIRECTORY,
					getServerDefinition().getResolver().resolveProperties(
							getServerDefinition().getStop()
									.getWorkingDirectory()));

			String programArgs=getServerDefinition().getResolver().resolveProperties(
					getServerDefinition().getStop().getProgramArguments())
					+ " "
					+ command;
				
			wc.setAttribute(
					IJavaLaunchConfigurationConstants.ATTR_PROGRAM_ARGUMENTS,
					programArgs);

			wc.setAttribute(
					IJavaLaunchConfigurationConstants.ATTR_VM_ARGUMENTS,
					getServerDefinition().getResolver().resolveProperties(
							getServerDefinition().getStop().getVmParameters()));
			// wc.setAttribute(ATTR_STOP, "true");

			ILaunch launched = wc.launch(ILaunchManager.RUN_MODE,
					new NullProgressMonitor());
			
			if(blockForCompletion)
			{
				IProcess[] processes = launched.getProcesses();
				for (int i = 0; i < processes.length; i++) {
					int maxDelay=15000;
					while(!processes[i].isTerminated() && maxDelay>0)
					{
						Thread.sleep(100);
						maxDelay -= 100;
					}
					rc=processes[i].getExitValue();
				}				
			}

		} catch (Exception e) {
			Trace.trace(Trace.SEVERE, "Error launching publish action", e);
			rc=-1;
		}
		
		return rc;
	}

	private File createJarFile(IPath location) {

		try {

			String rootFilename = location.toOSString();

			File rootDir = new File(rootFilename);
			String zipFilePrefix=rootDir.getName();
			if(zipFilePrefix.length()<3)
				zipFilePrefix+="123";
			File zipFile = File.createTempFile(zipFilePrefix, null);

			if (zipFile.exists())
				zipFile.delete();

			FileOutputStream fos = new FileOutputStream(zipFile);
			JarOutputStream jos = new JarOutputStream(fos);

			addToJar("", rootDir, jos);

			jos.close();
			fos.close();
			
			zipFile.deleteOnExit();

			return zipFile;
			
		} catch (IOException e) {
			Trace.trace(Trace.SEVERE, "Error creating zip file", e);
			return null;
		}
	}

	private void addToJar(String namePrefix, File dir,
			JarOutputStream jos) throws IOException {
		File[] contents = dir.listFiles();
		for (int i = 0; i < contents.length; i++) {
			File f = contents[i];
			if (f.isDirectory()) {
				// Recurse into the directory
				addToJar(namePrefix + f.getName() + "/", f, jos);
			} else {
				JarEntry entry = new JarEntry(namePrefix + f.getName());
				jos.putNextEntry(entry);

				byte[] buffer = new byte[10000];
				FileInputStream fis = new FileInputStream(f);
				int bytesRead = 0;
				while (bytesRead != -1) {
					bytesRead = fis.read(buffer);
					if (bytesRead > 0)
						jos.write(buffer, 0, bytesRead);
				}
			}
		}
	}

	private List getPublishClasspath() {
		String cpRef = getServerDefinition().getStop().getClasspathReference();
		return serverClasspath(cpRef);
	}

	public Map getServerInstanceProperties() {
		return getRuntimeDelegate().getServerInstanceProperties();
	}

}