/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - Initial API and implementation
 *******************************************************************************/
package org.eclipse.jst.server.geronimo.core.internal.model;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;

import org.eclipse.jst.server.geronimo.core.internal.xml.IMemento;
import org.eclipse.jst.server.geronimo.core.internal.xml.XMLMemento;
/**
 * Helper class to access a geronimo-jetty.xml file.
 */
public class WebAppDeploymentPlan {
	private static final String PARENT_ID = "parentId";
	private static final String CONTEXT_PRIORITY_CLASSLOADER = "context-priority-classloader";
	private static final String CONTEXT_ROOT = "context-root";
	private static final String CONFIG_ID = "configId";
	private static final String WEB_APP = "web-app";
	protected XMLMemento root;

	/**
	 * Loads a deploymentplan.xml from the given URL, and creates one if none exists now
	 *
	 * @param is
	 * @throws Exception if anything goes wrong
	 */
	public WebAppDeploymentPlan(InputStream is) throws Exception {
		root = XMLMemento.loadMemento(is);
		if (root == null) {
			root = XMLMemento.createWriteRoot(WEB_APP);
			root.putString("xmlns", "http://geronimo.apache.org/xml/ns/web/jetty");
		}
	}

	/**
	 * Loads a deploymentplan.xml from the given URL, and creates one if none exists now
	 *
	 * @throws Exception if anything goes wrong
	 */
	public WebAppDeploymentPlan() throws Exception {
		root = XMLMemento.createWriteRoot(WEB_APP);
		root.putString("xmlns", "http://geronimo.apache.org/xml/ns/web/jetty");
	}

	/**
	 * Returns the config Id
	 *
	 * @return java.util.List
	 */
	public String getConfigId() {
		if (root == null)
			return "";
		
		return root.getString(CONFIG_ID);
	}

	/**
	 * Returns the config Id
	 *
	 * @param configId
	 */
	public void setConfigId(String configId) {
		if (configId != null && configId.length() == 0)
			configId = null;
		root.putString(CONFIG_ID, configId);
	}

	/**
	 * Returns the config Id
	 *
	 * @return java.util.List
	 */
	public String getParentId() {
		if (root == null)
			return "";
		
		return root.getString(PARENT_ID);
	}

	/**
	 * Returns the config Id
	 *
	 * @param parentId
	 */
	public void setParentId(String parentId) {
		if (parentId != null && parentId.length() == 0)
			parentId = null;
		root.putString(PARENT_ID, parentId);
	}
	
	/**
	 * Returns the context root
	 *
	 * @return java.util.List
	 */
	public String getContextRoot() {
		if (root == null)
			return "";
		
		IMemento contextRootMemento = root.getChild(CONTEXT_ROOT);
		if (contextRootMemento == null)
			return "";
		return contextRootMemento.getTextData();
	}

	public void setContextRoot(String contextRoot) {
		IMemento contextRootMemento = root.getChild(CONTEXT_ROOT);
		if (contextRootMemento == null)
			contextRootMemento = root.createChild(CONTEXT_ROOT);
			
		if (contextRootMemento instanceof XMLMemento)
			((XMLMemento)contextRootMemento).setTextData(contextRoot);
		else
			throw new IllegalStateException();
	}

	/**
	 * Returns the context root
	 *
	 * @return java.util.List
	 */
	public boolean getContextPriorityClassloader() {
		if (root == null)
			return false;
		
		IMemento contextRootMemento = root.getChild(CONTEXT_PRIORITY_CLASSLOADER);
		if (contextRootMemento == null)
			return false;
		String data = contextRootMemento.getTextData();
		return (new Boolean(data)).booleanValue();
	}

	public void setContextPriorityClassloader(boolean contextPriorityClassloader) {
		IMemento memento=root.getChild(CONTEXT_PRIORITY_CLASSLOADER);
		if (memento==null)
			memento=root.createChild(CONTEXT_PRIORITY_CLASSLOADER);
		
		if (memento instanceof XMLMemento) {
			((XMLMemento)memento).setTextData(String.valueOf(contextPriorityClassloader));
		} else
			throw new IllegalStateException();
	}

	public ResourceRef[] getResourceRefs() {
		if (root == null)
			return new ResourceRef[0];
		
		ArrayList refList = new ArrayList();
		
		// Load the elements of type resource-ref
		IMemento[] children=root.getChildren(ResourceRef.RESOURCE_REF);
		for(int i=0; i<children.length; i++) {
			ResourceRef ref=new ResourceRef((XMLMemento)children[i]);
			refList.add(ref);
		}
		
		// Load the elements of type naming:resource-ref
		children=root.getChildren(ResourceRef.RESOURCE_REF_QUALIFIED);
		for(int i=0; i<children.length; i++) {
			ResourceRef ref=new ResourceRef((XMLMemento)children[i]);
			refList.add(ref);
		}
		
		return (ResourceRef[])refList.toArray(new ResourceRef[refList.size()]);
	}

	public void removeResourceRef(ResourceRef refToRemove) {
		root.removeChild(refToRemove.getMemento());
	}

	public ResourceEnvRef[] getResourceEnvRefs() {
		if (root == null)
			return new ResourceEnvRef[0];

		ArrayList envRefList = new ArrayList();
		
		// Load the elements of type resource-env-ref
		IMemento[] children=root.getChildren(ResourceEnvRef.RESOURCE_ENV_REF);
		for(int i=0; i<children.length; i++) {
			ResourceEnvRef ref=new ResourceEnvRef((XMLMemento)children[i]);
			envRefList.add(ref);
		}
		
		// Load the elements of type naming:resource-env-ref
		children=root.getChildren(ResourceEnvRef.RESOURCE_ENV_REF_QUALIFIED);
		for(int i=0; i<children.length; i++) {
			ResourceEnvRef ref=new ResourceEnvRef((XMLMemento)children[i]);
			envRefList.add(ref);
		}
		
		return (ResourceEnvRef[])envRefList.toArray(new ResourceEnvRef[envRefList.size()]);
	}

	public void removeResourceEnvRef(ResourceEnvRef refToRemove) {
		root.removeChild(refToRemove.getMemento());
	}

	public InputStream saveToStream() throws IOException {
		return root.getInputStream();
	}
	
	public byte[] getContents() throws IOException {
		return root.getContents();
	}

	public XMLMemento getRootMemento() {
		return root;
	}
}
