/*******************************************************************************
 * Copyright (c) 2003, 2004 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 * IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.wst.common.navigator.internal.provisional.views;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.StructuredViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.wst.common.navigator.internal.views.extensions.IExtensionActivationListener;
import org.eclipse.wst.common.navigator.internal.views.extensions.INavigatorContentServiceListener;
import org.eclipse.wst.common.navigator.internal.views.extensions.NavigatorActivationService;
import org.eclipse.wst.common.navigator.internal.views.extensions.NavigatorContentDescriptor;
import org.eclipse.wst.common.navigator.internal.views.extensions.NavigatorContentDescriptorInstance;
import org.eclipse.wst.common.navigator.internal.views.extensions.NavigatorContentDescriptorRegistry;
import org.eclipse.wst.common.navigator.internal.views.extensions.NavigatorViewerDescriptor;
import org.eclipse.wst.common.navigator.internal.views.extensions.NavigatorViewerDescriptorRegistry;
import org.eclipse.wst.common.navigator.internal.views.extensions.StructuredViewerManager;

/**
 * <p>
 * Provides centralized access to the information provided by NavigatorContentExtensions. Can be
 * instantiated as needed, but should be cached for active viewers. Information specific to a given
 * viewer will be cached by the NavigatorContentService, not including ContentProviders and Label
 * Providers created by {@see #createCommonContentProvider()}and
 * {@see #createCommonLabelProvider()}respectively.
 * </p>
 * 
 * <p>
 * The following class is experimental until fully documented.
 * </p>
 */
public class NavigatorContentService implements IExtensionActivationListener {

	private static final NavigatorActivationService NAVIGATOR_ACTIVATION_SERVICE = NavigatorActivationService.getInstance();
	private static final NavigatorContentDescriptorRegistry CONTENT_DESCRIPTOR_REGISTRY = NavigatorContentDescriptorRegistry.getInstance();
	private static final NavigatorViewerDescriptorRegistry VIEWER_DESCRIPTOR_REGISTRY = NavigatorViewerDescriptorRegistry.getInstance();

	private static final NavigatorContentDescriptorInstance[] NO_DESCRIPTOR_INSTANCES = new NavigatorContentDescriptorInstance[0];
	private static final ITreeContentProvider[] NO_CONTENT_PROVIDERS = new ITreeContentProvider[0];
	private static final ILabelProvider[] NO_LABEL_PROVIDERS = new ILabelProvider[0];

	private final NavigatorViewerDescriptor viewerDescriptor;
	private final List listeners = new ArrayList();

	/*
	 * A map of (String-based-Navigator-Content-Extension-IDs,
	 * NavigatorContentDescriptorInstance-objects)-pairs
	 */
	private final Map contentExtensions = new HashMap();
	private StructuredViewerManager structuredViewerManager;

	private ITreeContentProvider[] rootContentProviders;
	private Collection exclusions = new ArrayList();

	/**
	 *  
	 */
	public NavigatorContentService(String aViewerId) {
		super();
		aViewerId = aViewerId != null ? aViewerId : ""; //$NON-NLS-1$
		viewerDescriptor = VIEWER_DESCRIPTOR_REGISTRY.getNavigatorViewerDescriptor(aViewerId);
		NavigatorActivationService.getInstance().addExtensionActivationListener(viewerDescriptor.getViewerId(), this);
	}

	/**
	 *  
	 */
	public NavigatorContentService(String aViewerId, StructuredViewer aViewer) {
		this(aViewerId);
		structuredViewerManager = new StructuredViewerManager(aViewer);
	}

	public ITreeContentProvider createCommonContentProvider() {
		return new NavigatorContentServiceContentProvider(this);
	}

	/**
	 * @return
	 */
	public ILabelProvider createCommonLabelProvider() {
		return new NavigatorContentServiceLabelProvider(this);
	}

	/**
	 *  
	 */
	public void dispose() {
		for (Iterator contentItr = contentExtensions.values().iterator(); contentItr.hasNext();)
			((NavigatorContentDescriptorInstance) contentItr.next()).dispose();
		NavigatorActivationService.getInstance().removeExtensionActivationListener(viewerDescriptor.getViewerId(), this);
	}

	protected void updateService(Viewer aViewer, Object anOldInput, Object aNewInput) {

		synchronized (this) {

			if (structuredViewerManager == null) {
				structuredViewerManager = new StructuredViewerManager(aViewer);
				structuredViewerManager.inputChanged(anOldInput, aNewInput);
			} else
				structuredViewerManager.inputChanged(aViewer, anOldInput, aNewInput);

			for (Iterator contentItr = contentExtensions.values().iterator(); contentItr.hasNext();)
				structuredViewerManager.initialize(((NavigatorContentDescriptorInstance) contentItr.next()).getContentProvider());

			NavigatorContentDescriptorInstance[] resultInstances = findRootContentDescriptors(aNewInput);
			rootContentProviders = extractContentProviders(resultInstances);
		}
	}

	/**
	 * @param element
	 * @return
	 */
	public ITreeContentProvider[] findParentContentProviders(Object anElement) {
		NavigatorContentDescriptorInstance[] resultInstances = findRelevantContentDescriptorInstances(anElement);
		return extractContentProviders(resultInstances);
	}

	/**
	 * <p>
	 * Return all of the content providers that are relevant for the viewer. The viewer is
	 * determined by the ID used to create the NavigatorContentService.
	 * </p>
	 * 
	 * @return
	 */
	public ITreeContentProvider[] findRootContentProviders(Object anElement) {
		if (rootContentProviders != null)
			return rootContentProviders;
		synchronized (this) {
			if (rootContentProviders == null) {
				NavigatorContentDescriptorInstance[] resultInstances = findRootContentDescriptors(anElement);
				if (resultInstances.length > 0)
					rootContentProviders = extractContentProviders(resultInstances);
				else {
					resultInstances = findRootContentDescriptors(anElement, false);
					rootContentProviders = extractContentProviders(resultInstances);
				}
			}
		}
		return rootContentProviders;
	}

	/**
	 * <p>
	 * Return all of the content providers that are enabled for the given parameter 'element'.
	 * 
	 * @param anElement
	 * @return
	 */
	public ITreeContentProvider[] findRelevantContentProviders(Object anElement) {
		NavigatorContentDescriptorInstance[] resultInstances = findRelevantContentDescriptorInstances(anElement);
		return extractContentProviders(resultInstances);
	}


	/**
	 * <p>
	 * Return all of the label providers that are enabled for the given parameter 'element'.
	 * 
	 * @param anElement
	 * @return
	 */
	
	public ILabelProvider[] findRelevantLabelProviders(Object anElement) {
		NavigatorContentDescriptorInstance[] resultInstances = findRelevantContentDescriptorInstances(anElement,false);
		return extractLabelProviders(resultInstances);
	}
	
	public NavigatorContentDescriptorInstance[] findRelevantContentDescriptorInstances(Object anElement) {
		List enabledDescriptors = getEnabledDescriptors(anElement);
		return extractDescriptorInstances(enabledDescriptors);		
	}
	
	public NavigatorContentDescriptorInstance[] findRelevantContentDescriptorInstances(Object anElement, boolean toLoadIfNecessary) {
		List enabledDescriptors = getEnabledDescriptors(anElement);
		return extractDescriptorInstances(enabledDescriptors,toLoadIfNecessary);		
	}
	
	

	public NavigatorContentDescriptorInstance[] findRelevantContentDescriptorInstances(IStructuredSelection aSelection) {
		List contentDescriptors = getEnabledDescriptors(aSelection);
		if(contentDescriptors.size() == 0)
			return NO_DESCRIPTOR_INSTANCES;
		NavigatorContentDescriptorInstance[] contentDescriptorInstances = new NavigatorContentDescriptorInstance[contentDescriptors.size()];
		NavigatorContentDescriptor descriptor; 		
		for(int i=0; i<contentDescriptors.size(); i++ ) {
			descriptor = (NavigatorContentDescriptor) contentDescriptors.get(i);
			contentDescriptorInstances[i] = getDescriptorInstance(descriptor);
		}
		return contentDescriptorInstances;
		
	}


	/**
	 * @param anElement
	 * @return
	 */
	private List getEnabledDescriptors(Object anElement) {
		return filterDescriptors(CONTENT_DESCRIPTOR_REGISTRY.getEnabledContentDescriptors(anElement, viewerDescriptor)); 
	}

	/**
	 * @param aSelection
	 * @return
	 */
	private List getEnabledDescriptors(IStructuredSelection aSelection) {
		return filterDescriptors(CONTENT_DESCRIPTOR_REGISTRY.getEnabledContentDescriptors(aSelection));
	}
	

	
	private List filterDescriptors(List contentDescriptors) {
		List result = new ArrayList();
		for (int x=0; x< contentDescriptors.size(); ++x) {
			NavigatorContentDescriptor descriptor = (NavigatorContentDescriptor) contentDescriptors.get(x);
			if(!exclusions.contains(descriptor.getId()))
				result.add(descriptor);			
		}
		return result;
		
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.wst.common.navigator.internal.views.extensions.IInitializationManager#initialize(org.eclipse.jface.viewers.IStructuredContentProvider)
	 */
	public boolean initialize(IStructuredContentProvider aContentProvider) {
		return structuredViewerManager.initialize(aContentProvider);
	}

	/**
	 * @param viewerId
	 * @param navigatorExtensionId
	 * @param toEnable
	 */
	public void onExtensionActivation(String aViewerId, String aNavigatorExtensionId, boolean toEnable) {
		update();
	}

	public void update() {
		rootContentProviders = null;
		structuredViewerManager.safeRefresh();
	}

	/**
	 * @return Returns the viewerId.
	 */
	public final String getViewerId() {
		return viewerDescriptor.getViewerId();
	}
	/**
	 * @param object
	 * @return
	 */
	public final NavigatorContentDescriptorInstance getDescriptorInstance(NavigatorContentDescriptor aDescriptorKey) {
		return getDescriptorInstance(aDescriptorKey,true);
	}
	
	/**
	 * @param object
	 * @return
	 */
	public final NavigatorContentDescriptorInstance getDescriptorInstance(NavigatorContentDescriptor aDescriptorKey , boolean toLoadIfNecessary) {
		/* Query and return the relevant descriptor instance */
		NavigatorContentDescriptorInstance descriptorInstance = (NavigatorContentDescriptorInstance) contentExtensions.get(aDescriptorKey);
		if (descriptorInstance != null || !toLoadIfNecessary)
			return descriptorInstance;

		/*
		 * If the descriptor instance hasn't been created yet, then we need to (1) verify that it
		 * wasn't added by another thread, (2) create and add the result into the map
		 */
		synchronized (this) {
			descriptorInstance = (NavigatorContentDescriptorInstance) contentExtensions.get(aDescriptorKey);
			if (descriptorInstance == null) {
				contentExtensions.put(aDescriptorKey, (descriptorInstance = new NavigatorContentDescriptorInstance(aDescriptorKey, this, viewerDescriptor.getViewerId())));
				notifyListeners(descriptorInstance);
			}
		}
		return descriptorInstance;
		
	}
	
	
	
	

	/**
	 * 
	 * @return The ViewerDescriptor for tihs Content Service instance. 
	 */
	public NavigatorViewerDescriptor getViewerDescriptor() {
		return viewerDescriptor;
	}
	
	
	public void addExclusion(String anExtensionId) {
		exclusions.add(anExtensionId);
	}
	
	public void removeExclusion(String anExtensionId) {
		exclusions.remove(anExtensionId);
	}

	protected final NavigatorContentDescriptorInstance[] findRootContentDescriptors(Object anElement) {
		return findRootContentDescriptors(anElement, true);
	}

	protected final NavigatorContentDescriptorInstance[] findRootContentDescriptors(Object anElement, boolean toRespectViewerRoots) {
		String[] rootDescriptorIds = viewerDescriptor.getRootContentExtensionIds();
		NavigatorContentDescriptorInstance[] resultInstances = null;

		if (toRespectViewerRoots && rootDescriptorIds.length > 0 && rootDescriptorIds != NavigatorViewerDescriptor.DEFAULT_VIEWER_ID) {
			List resultInstancesList = new ArrayList();

			NavigatorContentDescriptor descriptor = null;
			NavigatorContentDescriptorInstance descriptorInstance = null;
			for (int i = 0; i < rootDescriptorIds.length; i++) {
				if (isActive(rootDescriptorIds[i])) {
					descriptor = CONTENT_DESCRIPTOR_REGISTRY.getContentDescriptor(rootDescriptorIds[i]);
					descriptorInstance = getDescriptorInstance(descriptor);
					if (!descriptorInstance.hasLoadingFailed())
						resultInstancesList.add(descriptorInstance);
				}
			}
			resultInstancesList.toArray((resultInstances = new NavigatorContentDescriptorInstance[resultInstancesList.size()]));
		} else
			resultInstances = findRelevantContentDescriptorInstances(anElement);
		return resultInstances;
	}

	protected boolean isActive(String anExtensionId) {
		return !exclusions.contains(anExtensionId) && NAVIGATOR_ACTIVATION_SERVICE.isNavigatorExtensionActive(getViewerId(), anExtensionId);
	}	

	protected final Collection getDescriptorInstances() {
		return (contentExtensions.size() > 0) ? Collections.unmodifiableCollection(contentExtensions.values()) : Collections.EMPTY_LIST;
	}

	public void addListener(INavigatorContentServiceListener aListener) {
		listeners.add(aListener);
	}

	public void removeListener(INavigatorContentServiceListener aListener) {
		listeners.remove(aListener);
	}

	private void notifyListeners(NavigatorContentDescriptorInstance aDescriptorInstance) {
		
		if (listeners.size() == 0)
			return;
		INavigatorContentServiceListener listener = null;
		List failedListeners = null;
		for (Iterator listenersItr = listeners.iterator(); listenersItr.hasNext();) {
			try {
				listener = (INavigatorContentServiceListener) listenersItr.next();
				listener.onLoad(aDescriptorInstance);
			} catch (RuntimeException re) {
				if (failedListeners == null)
					failedListeners = new ArrayList();
				failedListeners.add(listener);				
			}
		}
		if (failedListeners != null) {
			listeners.removeAll(failedListeners);
		}
	}

	/**
	 * @param theDescriptorInstances
	 * @return
	 */
	private ITreeContentProvider[] extractContentProviders(NavigatorContentDescriptorInstance[] theDescriptorInstances) {
		if (theDescriptorInstances.length == 0)
			return NO_CONTENT_PROVIDERS;
		List resultProvidersList = new ArrayList();
		for (int i = 0; i < theDescriptorInstances.length; i++)
			if (theDescriptorInstances[i].getContentProvider() != null)
				resultProvidersList.add(theDescriptorInstances[i].getContentProvider());
		return (ITreeContentProvider[]) resultProvidersList.toArray(new ITreeContentProvider[resultProvidersList.size()]);
	}

	/**
	 * @param theDescriptors
	 *            a List of NavigatorContentDescriptor objects
	 * @return
	 */
	private NavigatorContentDescriptorInstance[] extractDescriptorInstances(List theDescriptors) {
		return extractDescriptorInstances(theDescriptors,true);
	}
	
	
	/**
	 * @param theDescriptors
	 *            a List of NavigatorContentDescriptor objects
	 * @return
	 */
	private NavigatorContentDescriptorInstance[] extractDescriptorInstances(List theDescriptors, boolean toLoadAllIfNecessary) {
		if (theDescriptors.size() == 0)
			return NO_DESCRIPTOR_INSTANCES;
		List  resultInstances = new ArrayList();
		for (int i = 0; i < theDescriptors.size(); i++) {
			NavigatorContentDescriptorInstance descriptorInstance = getDescriptorInstance((NavigatorContentDescriptor) theDescriptors.get(i), toLoadAllIfNecessary);
			if (descriptorInstance != null) {
				resultInstances.add(descriptorInstance);
				
			}
		}
		return (NavigatorContentDescriptorInstance[]) resultInstances.toArray(new NavigatorContentDescriptorInstance[resultInstances.size()]);
	}

	/**
	 * @param theDescriptorInstances
	 * @return
	 */
	private ILabelProvider[] extractLabelProviders(NavigatorContentDescriptorInstance[] theDescriptorInstances) {
		if (theDescriptorInstances.length == 0)
			return NO_LABEL_PROVIDERS;
		List resultProvidersList = new ArrayList();
		for (int i = 0; i < theDescriptorInstances.length; i++)
			if (theDescriptorInstances[i].getLabelProvider() != null)
				resultProvidersList.add(theDescriptorInstances[i].getLabelProvider());
		return (ILabelProvider[]) resultProvidersList.toArray(new ILabelProvider[resultProvidersList.size()]);
	}
}