/*******************************************************************************
 * Copyright (c) 2001, 2004 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.wst.rdb.data.internal.ui.editor;

import org.eclipse.core.runtime.*;
import org.eclipse.jface.action.*;
import org.eclipse.jface.dialogs.*;
import org.eclipse.jface.viewers.*;
import org.eclipse.swt.*;
import org.eclipse.swt.layout.*;
import org.eclipse.swt.widgets.*;
import org.eclipse.ui.*;
import org.eclipse.ui.part.*;
import org.eclipse.wst.rdb.data.internal.core.editor.*;
import org.eclipse.wst.rdb.data.internal.ui.*;

/**
 * This editor allows borwsing and editing the data stored in a SQL table.
 * The logic for accessing and altering the data is implemented in TableDataImpl.
 * @author groux
 */
public class TableDataEditor extends EditorPart
{
    /** JFace table viewer */
    protected TableViewer tableViewer;
    
    /** Application model for the table data */
    protected ITableData tableData;
    
    /** SQLObject for the table */
    protected org.eclipse.wst.rdb.internal.models.sql.tables.Table sqlTable;
    
    /** Editor dirty */
    protected boolean dirty = false;
    
    /** Spreadsheet-like cursor */
    protected TableDataTableCursor cursor;
    
    
    public TableDataEditor() {

    }
    
    public void init(IEditorSite site, IEditorInput input) throws PartInitException {
        if (!(input instanceof TableDataEditorInput))
            throw new PartInitException("Invalid Input: Must be TableEditorInput"); //$NON-NLS-1$
        
        setSite(site);
        setInput(input);
        sqlTable = ((TableDataEditorInput)input).getTable();
        
        this.setPartName(sqlTable.getName());
    }
    
    public void createPartControl(Composite parent) {
        
        tableViewer = new TableViewer(parent, SWT.HIDE_SELECTION);
        tableViewer.getTable().setHeaderVisible(true);      
        tableViewer.getTable().setLinesVisible(true);
        tableViewer.getTable().setLayoutData(new GridData(GridData.FILL_BOTH));

        tableViewer.setLabelProvider( new TableDataLabelProvider() );
        tableViewer.setContentProvider( new TableDataContentProvider() );                
        
        try {
            tableData = new TableDataImpl(sqlTable);
            configureTable();
            tableViewer.setInput(tableData);            
        } catch (Exception ex) {      
            DataUIPlugin.getDefault().writeLog(IStatus.ERROR, 0, ex.getMessage(), ex);
            displayException(Messages.getString("TableDataEditor.ErrorInitializingEditor"), ex); //$NON-NLS-1$
        }
        
        cursor = new TableDataTableCursor(tableViewer);
        if (tableViewer.getTable().getItemCount()>0)
        	cursor.setSelection(0,0);

        createContextMenu();
    }
    
    public void dispose() {
        if (tableData!=null)
            tableData.dispose();
        super.dispose();
    }
    
    public void doSave(IProgressMonitor monitor) {
        try {
	        boolean ok = tableData.save();
	        if (ok)
	            setDirty(false);
	        else
	            monitor.setCanceled(true);
        } catch (Exception ex) {
            DataUIPlugin.getDefault().writeLog(IStatus.ERROR, 0, ex.getMessage(), ex);
            displayException(Messages.getString("TableDataEditor.ErrorWhileSaving"), ex); //$NON-NLS-1$
            monitor.setCanceled(true);
        }        
    }

    public boolean isSaveAsAllowed() {
        return false;
    }
    
    public void doSaveAs() {
    }

    public boolean isDirty() {
        return dirty;
    }
    
    protected void setDirty(boolean value) {
        dirty = value;
        firePropertyChange(PROP_DIRTY);
    }
    
    public void doRevert() {
        tableData.revert();
        tableViewer.refresh();
        cursor.redraw();
        setDirty(false);
    }
    
    public void doRefresh() {
        if (getEditorSite().getPage().saveEditor(this, true))
	        try {
	            tableData = new TableDataImpl(sqlTable);
	            tableViewer.setInput(tableData);          
	            cursor.redraw();
	            setDirty(false);
	        } catch (Exception ex) {
	            DataUIPlugin.getDefault().writeLog(IStatus.ERROR, 0, ex.getMessage(), ex);
	            displayException(Messages.getString("TableDataEditor.ErrorRefreshing"), ex); //$NON-NLS-1$
	        }
    }
    
    protected IRowData getRow()
    {
        Object row = cursor.getRow().getData();
        if (row instanceof IRowData)
            return (IRowData)row;
        else
            return null;
    }
    
    protected IRowData getOrCreateRow()
    {
        IRowData row = getRow();
        if (row==null) {
            IRowData newRow = tableData.insertRow();
            tableViewer.insert(newRow, tableViewer.getTable().getItemCount()-1);  
            cursor.setSelection(tableViewer.getTable().getItemCount()-2, cursor.getColumn());  
            cursor.redraw();
            return newRow;
        }
        return row;
    }
    
    public void doInsertRow() {
        cursor.setSelection(tableViewer.getTable().getItemCount()-1, 0);
        doUpdateValue();
    }
    
    public void doUpdateValue() {
        cursor.edit();
    }
    
    public void doSetNull() {
        IRowData row = getOrCreateRow();
        row.updateValue(cursor.getColumn(), null);
        tableViewer.refresh(row);
        cursor.redraw();
        setDirty(true);
    }
    
//    public void setDefault() {
//        Object row = cursor.getRow().getData();
//        if (row instanceof IRowData) {
//            Column col = (Column)sqlTable.getColumns().get(cursor.getColumn());
//            if (col.getDefaultValue()!=null) {
//                Object o = DataDeserializer.deserialize(col.getDefaultValue(), tableData.getColumnType(cursor.getColumn()));
//    	        ((IRowData)row).updateValue(cursor.getColumn(), o);
//    	        tableViewer.refresh(row);
//    	        cursor.redraw();
//    	        setDirty(true);
//            }
//        }
//    }
//    
//    public boolean getColumnHasDefault()
//    {
//        Column col = (Column)sqlTable.getColumns().get(cursor.getColumn());
//        return col.getDefaultValue()!=null;
//    }
    
    public void doDelete() {
        IRowData row = getRow();
        if (row!=null) {
	        tableData.deleteRow(row);
	        tableViewer.remove(row);
	        setDirty(true);
        }
    }
    
    protected void createContextMenu() {
        
        final TableDataEditorActionBarContributor contributor = DataUIPlugin.getDefault().getTableDataEditorContributor();
       
        // Create menu manager.
        MenuManager menuMgr = new MenuManager();
        menuMgr.setRemoveAllWhenShown(true);
        menuMgr.addMenuListener(new IMenuListener() {
            public void menuAboutToShow(IMenuManager mgr) {
                mgr.add(contributor.revertAction);
                mgr.add(contributor.refreshAction);
                mgr.add(new Separator());
                mgr.add(contributor.updateAction);
                mgr.add(contributor.setNullAction);
//                mgr.add(contributor.setDefaultAction);
                mgr.add(contributor.insertAction);
                mgr.add(contributor.deleteAction);
                mgr.add(new Separator());
                mgr.add(contributor.saveAction);
 //               mgr.add( getEditorSite().getActionBars().getGlobalActionHandler(ActionFactory.REFRESH.getId()) );
            }
        });

        // Create menu.
        Menu menu = menuMgr.createContextMenu(tableViewer.getTable());
        tableViewer.getTable().setMenu(menu);
        cursor.setMenu(menu);
    }

    public void setFocus() {
        cursor.setFocus();
        
    }

    protected void configureTable()
    {
        TextCellEditor textEditor = new TableDataCellEditor(this, tableViewer.getTable());
        
        CellEditor[] editors = new CellEditor[sqlTable.getColumns().size()];
        String[] properties = new String[sqlTable.getColumns().size()];

        for (int i=0; i<tableData.getColumnCount(); ++i) {
            TableColumn col = new TableColumn(tableViewer.getTable(), SWT.NONE);
            col.setWidth(100);
            col.setText(tableData.getColumnHeader(i));
            col.pack();
            editors[i] = textEditor;
            properties[i] = tableData.getColumnName(i);
        }

        tableViewer.setCellEditors(editors);
        tableViewer.setColumnProperties(properties);
        tableViewer.setCellModifier(new TableDataCellModifier(this, tableViewer));
        
        tableViewer.getTable().pack();
    }


    public ITableData getTableData() {
        return tableData;
    }
    
    public org.eclipse.wst.rdb.internal.models.sql.tables.Table getSqlTable() {
        return sqlTable;
    }
    
    public TableDataTableCursor getCursor() {
        return cursor;
    }
    
    public boolean isReadonly() {
        return tableData.isReadonly();
    }
    
    protected void displayException(String msg, Exception ex) {
        IStatus warning = new Status(IStatus.ERROR, DataUIPlugin.PLUGIN_ID, 1, ex.toString(), ex);
        ErrorDialog.openError(tableViewer.getControl().getShell(), msg, null, warning);
    }
}


