/*******************************************************************************
 * Copyright (c) 2001, 2004 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.wst.rdb.data.internal.core.extract;

import java.io.*;
import java.sql.*;
import java.util.*;

import org.eclipse.core.runtime.*;
import org.eclipse.wst.rdb.data.internal.core.*;
import org.eclipse.wst.rdb.data.internal.core.common.*;
import org.eclipse.wst.rdb.internal.core.rte.*;
import org.eclipse.wst.rdb.internal.models.sql.tables.*;
import org.eclipse.wst.rdb.internal.outputview.*;

public class ExtractData
{

    protected Connection connection;
    protected Table table;
    protected String filePath;
    
    protected String colDelim = ","; //$NON-NLS-1$
    protected static final String ENDL = System.getProperty("line.separator"); //$NON-NLS-1$
    protected String stringDelim = "\""; //$NON-NLS-1$
    
    protected HashSet errorColumns = new HashSet();
    protected int rowsExtracted = 0;
   
    public ExtractData(Table table, String filePath)
    {
        this.connection = ((ICatalogObject)table).getConnection();
        this.table = table;
        this.filePath = filePath;
    }
    
    public void setDelims(String colDelim, String stringDelim)
    {
        this.colDelim = colDelim;
        this.stringDelim = stringDelim;
    }
    
    public int doExtract(OutputItem item)
    {                
        OutputViewAPI.getInstance().showMessage(item, Messages.getString("ExtractData.Extracting") + " " + getFullyQualifiedName() + "...", true); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
        
        try {
            doExtract1();
        } catch (Exception ex) {
            DataCorePlugin.getDefault().writeLog(IStatus.ERROR, 0, ex.getMessage(), ex);
            OutputViewAPI.getInstance().showMessage(item, ex.toString(), true);
            OutputViewAPI.getInstance().showMessage(item, Messages.getString("ExtractData.DataExtractionFailed"), true); //$NON-NLS-1$
            OutputViewAPI.getInstance().updateStatus(item,OutputItem.STATUS_FAILURE, true);
            return OutputItem.STATUS_FAILURE;
        }
        
        OutputViewAPI.getInstance().showMessage(item, Messages.getString("ExtractData.DataExtractionSuccessful"), true); //$NON-NLS-1$
        OutputViewAPI.getInstance().showMessage(item, String.valueOf(rowsExtracted) + Messages.getString("ExtractData.RowsExtracted"), true);         //$NON-NLS-1$
        
        if (errorColumns.size()>0) {
            StringBuffer sb = new StringBuffer();
            sb.append(Messages.getString("ExtractData.ErrorRetrieving")); //$NON-NLS-1$
            Iterator it = errorColumns.iterator();
            while (it.hasNext()) {
                sb.append(it.next().toString());
                if (it.hasNext())
                    sb.append(", "); //$NON-NLS-1$
            }
            OutputViewAPI.getInstance().showMessage(item, sb.toString(), true);
            return OutputItem.STATUS_WARNING;
        } else {
            return OutputItem.STATUS_SUCCESS;
        }
    }
    
    protected void doExtract1() throws SQLException, IOException
    {
        Statement st = null;
        ResultSet rs = null;
        BufferedWriter fw = null;
        
		
		try {
		    st = connection.createStatement();
			rs = st.executeQuery( getQuery() );
			fw = new BufferedWriter(new FileWriter(filePath));
		    writeData(rs, fw);
		} finally {
		    if (rs!=null)
		        rs.close();
		    if (st!=null)
		        st.close();
		    if (fw!=null)
		        fw.close();
		}
    }
    
    protected void writeData(ResultSet rs, Writer fw) throws SQLException, IOException
    {
        ResultSetMetaData md = rs.getMetaData();

        while (rs.next()) {
			for (int i=0; i<md.getColumnCount(); ++i) {			    
			    if (i!=0)
					fw.write(colDelim);
			    String s = getVal(rs, i);
			    if (s!=null)
			        fw.write(s);
			}
			rowsExtracted++;
			fw.write(ENDL);			
		}

    }
    
    public String getVal(ResultSet rs, int column) throws SQLException, IOException
    {
        ResultSetMetaData md = rs.getMetaData();
        try {
            Object o = ResultSetReader.read(rs, column);
            String s = DataSerializer.write(o, md.getColumnType(column+1), stringDelim);
            return s;
        } catch (Exception ex) {
            errorColumns.add(md.getColumnName(column+1));
            return null;
        }
    }
    
    protected String getQuery()
    {
        return "select * from " + getFullyQualifiedName(); //$NON-NLS-1$
    }
    
    protected String getFullyQualifiedName() {
    	return DataCorePlugin.getQualifiedTableName(table);
    }
    
}
