/*******************************************************************************
 * Copyright (c) 2000, 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial implementation
 *******************************************************************************/
package org.eclipse.wst.rdb.server.extensions.internal.drop;

import java.util.ArrayList;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.IExtensionRegistry;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Platform;
import org.eclipse.wst.rdb.internal.core.connection.ConnectionInfo;
import org.eclipse.wst.rdb.internal.core.rte.DDLGenerator;
import org.eclipse.wst.rdb.internal.core.rte.EngineeringOption;
import org.eclipse.wst.rdb.internal.models.sql.schema.SQLObject;
import org.eclipse.wst.rdb.internal.outputview.OutputItem;
import org.eclipse.wst.rdb.internal.outputview.OutputViewAPI;
import org.eclipse.wst.rdb.server.extensions.internal.ServerExtensionsPlugin;
import org.eclipse.wst.rdb.server.extensions.internal.util.Utility;
import org.eclipse.wst.rdb.server.internal.ui.query.execute.QueryOutputHelper;

public class DropObject  
{
   protected SQLObject mysqlObj = null;
	
	public DropObject(SQLObject theSQLObj) 
	{
	   mysqlObj = theSQLObj;
	}
	
	/**
	 * Handle drop action
	 */
	public void run(ConnectionInfo conInfo) 
	{        	       
      IDropSQLObject dropSQLObject = null;
      
      IExtensionRegistry pluginRegistry = Platform.getExtensionRegistry();
      IExtensionPoint extensionPoint = pluginRegistry.getExtensionPoint("org.eclipse.wst.rdb.server.extensions", "DropSQLObjects"); //$NON-NLS-1$ //$NON-NLS-2$
      IExtension[] extensions = extensionPoint.getExtensions();
      for(int i=0; i<extensions.length; ++i) {
         IConfigurationElement[] configElements = extensions[i].getConfigurationElements();
         for(int j=0; j<configElements.length; ++j) {
            if(configElements[j].getName().equals("DropProvider")) { //$NON-NLS-1$
               String objectType = configElements[j].getAttribute("objectClassname"); //$NON-NLS-1$
					
					if ((objectType.equals(mysqlObj.getClass().getName()))) 
					{
					   try {
					      dropSQLObject = (IDropSQLObject) configElements[j].createExecutableExtension("class"); //$NON-NLS-1$
					      break;
					   }
					   catch (CoreException e) {
					      ServerExtensionsPlugin.getDefault().writeLog(IStatus.ERROR, 0, e.getMessage(), e);
					   }            
					}                  
            }
         }
         if (dropSQLObject != null)
            break;
      }
      
      if (dropSQLObject == null) {
         dropSQLObject(mysqlObj, Utility.getSchemaQualifiedName(mysqlObj), conInfo);         
      }
      else 
         dropSQLObject.dropSQLObject(mysqlObj);
	}           
	
	/**
	 * Drop a SQL object by creating the DROP DDL statement using generate DDL
	 * @param sqlObject
	 * @param qualifiedName
	 * @param conInfo
	 * @return true when the sqlObject could be dropped
	 */
	public static boolean dropSQLObject(SQLObject sqlObject, String qualifiedName, ConnectionInfo conInfo)
	{
	   boolean dropResult = true;
	   try 
		{       
	      OutputViewAPI outputView = (OutputViewAPI)OutputViewAPI.getInstance();	    	
	      OutputItem outItem = outputView.findOutputItem(qualifiedName, OutputItem.ACTION_DROP );
	      if (outItem == null)
	         outItem = new OutputItem(OutputItem.STATUS_START, OutputItem.ACTION_DROP, 
	                  					 Utility.getSchemaQualifiedName(sqlObject), qualifiedName);
	      else
	         outputView.resetOutputItem(outItem, true);
	      outputView.addOutputItem(outItem,true);
	      OutputViewAPI.getInstance().showMessage(outItem, 
	               ServerExtensionsPlugin.getString("DROP_STARTED", //$NON-NLS-1$
	                        								new Object[]{qualifiedName}), true); 	      
	      
       	DDLGenerator ddlProvider = conInfo.getDatabaseDefinition().getDDLGenerator();
       	SQLObject[] elements = new SQLObject []{sqlObject};
       	
         /* save engineering options of ddlGenerator and set all
          * options to true in order to create DDL for all elements
          */ 
         EngineeringOption[] options = ddlProvider.getOptions();
         ArrayList optionsSettings = new ArrayList(options.length);   	          
         for (int i = 0; i < options.length; i++){
            EngineeringOption option = (EngineeringOption) options[i];
            optionsSettings.add(new Boolean(option.getBoolean()));
            option.setBoolean(true);
         }

       	String[] dropObjects = ddlProvider.dropSQLObjects( elements, true, true, new NullProgressMonitor());
        	
         // restore engineering options which were set before
         for (int i = 0; i < options.length; i++){
            EngineeringOption option = (EngineeringOption) options[i];
            option.setBoolean(((Boolean)optionsSettings.get(i)).booleanValue());
         }

        	QueryOutputHelper outputHelper = new QueryOutputHelper(conInfo.getSharedConnection());
        	outputHelper.setOutputItem(outItem);
        	outputHelper.setUpdateOutputItemStatus(false);
        	boolean warningMsg = false;
        	for(int i = 0; i < dropObjects.length; i++)
        	{
        	   outputHelper.setStatement(dropObjects[i]);
        		boolean executeResult = outputHelper.executeDDL(
        		         ServerExtensionsPlugin.getString("DDL_GEN",  //$NON-NLS-1$
        		                  				new Object[]{Utility.getSchemaQualifiedName(sqlObject)}));
        		if (!warningMsg && !executeResult)
        		   warningMsg = true;
        		if (i==dropObjects.length && !executeResult)
        		   dropResult = false;
        	}                
        	
         if (dropResult && !warningMsg) {
            OutputViewAPI.getInstance().updateStatus(outItem, OutputItem.STATUS_SUCCESS, true);                          
            OutputViewAPI.getInstance().showMessage(outItem, 
                     ServerExtensionsPlugin.getString("DROP_COMPLETED", //$NON-NLS-1$
                              								new Object[]{qualifiedName}), true);
         }
         else if (dropResult && warningMsg) {
            OutputViewAPI.getInstance().updateStatus(outItem, OutputItem.STATUS_WARNING, true);
            OutputViewAPI.getInstance().showMessage(outItem, 
                     ServerExtensionsPlugin.getString("DROP_COMPLETED_WITH_WARNINGS", //$NON-NLS-1$
                              								new Object[]{qualifiedName}), true);
         }
         else {
            OutputViewAPI.getInstance().updateStatus(outItem, OutputItem.STATUS_FAILURE, true);
            OutputViewAPI.getInstance().showMessage(outItem, 
                     ServerExtensionsPlugin.getString("DROP_FAILED", //$NON-NLS-1$
                              								new Object[]{qualifiedName}), true);
         }
		} 
	   catch (Exception e) 
	   {
	      ServerExtensionsPlugin.getDefault().writeLog(IStatus.ERROR, 0, e.getMessage(), e);
	      dropResult = false;
	   }
	   
	   return dropResult;
	}
            
}

    
