/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 * IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.wst.rdb.server.extensions.internal.editorlaunch;

import java.text.MessageFormat;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.jface.dialogs.IMessageProvider;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IFileEditorInput;
import org.eclipse.ui.dialogs.SaveAsDialog;
import org.eclipse.ui.part.FileEditorInput;
import org.eclipse.ui.texteditor.IDocumentProvider;
import org.eclipse.wst.rdb.sqleditor.internal.ISQLEditorInput;
import org.eclipse.wst.rdb.sqleditor.internal.SQLEditor;
import org.eclipse.wst.rdb.sqleditor.internal.SQLEditorFileEditorInput;
import org.eclipse.wst.rdb.sqleditor.internal.SQLEditorResources;
import org.eclipse.wst.rdb.sqleditor.internal.SQLEditorStorageEditorInput;


/**
 * This class extends SQLEditor for use in the Server Explorer.  It overrides 
 * the "save" and "save as" behavior of the editor to make it possible to save 
 * storage-based inputs as files.
 */
public class SQLEditorForServerExplorer extends SQLEditor {

    /**
     * Constructs an instance of this class. This is the default constructor.
     */
    public SQLEditorForServerExplorer() {
        super();
    }

    /**
     * Saves the content of this editor.
     * 
     * @param progressMonitor the progress monitor for communicating result
     *            state or <code>null</code>
     * @see org.eclipse.ui.ISaveablePart#doSave(org.eclipse.core.runtime.IProgressMonitor)
     */
    public void doSave( IProgressMonitor monitor ) {
        IEditorInput editorInput = getEditorInput();
        if (editorInput instanceof SQLEditorStorageEditorInput) {
            super.doSaveAs();
        }
        else {
            super.doSave( monitor );
        }
    }

    /**
     * Performs a save as and reports the result state back to the 
     * given progress monitor.  This implementation calls the superclass 
     * implementation when the editor input is a <code>IFileEditorInput</code>
     * but has a custom implementation when the editor input is a 
     * <code>IStorageEditorInput</code>.  This is done to set the initial file
     * name from the name in the <code>IStorageEditorInput</code>.
     * 
     * @param progressMonitor the progress monitor to be used
     * @see org.eclipse.ui.texteditor.TextEditor#performSaveAs(org.eclipse.core.runtime.IProgressMonitor)
     */
    protected void performSaveAs(IProgressMonitor progressMonitor) {
        IEditorInput input = getEditorInput();
        String inputName = input.getName();
        
        /* Begin code copied from TextEditor.performSaveAs */
        Shell shell= getSite().getShell();
        
        SaveAsDialog dialog= new SaveAsDialog(shell);
        
        /* Begin local modifications */
        /* When file input, behave as usual. */
        if (input instanceof IFileEditorInput) {
            IFile original= (input instanceof IFileEditorInput) ? ((IFileEditorInput) input).getFile() : null;
            if (original != null) {
                dialog.setOriginalFile(original);
            }
        }
        /* When we have a SQLEditorStorageEditorInput, initialize the SaveAsDialog
         * with the input name, since that's all we have
         */
        else if (input instanceof SQLEditorStorageEditorInput){
            if (inputName != null && inputName.length() > 0) {
                if (!inputName.endsWith(".sql") && !inputName.endsWith(".ddl")) { //$NON-NLS-1$ //$NON-NLS-2$
                    inputName = inputName + ".sql"; //$NON-NLS-1$
                }
                dialog.setOriginalName( inputName );
            }
        }
        /* End local modifications */
        
        dialog.create();
        
        IDocumentProvider provider= getDocumentProvider();
        if (provider == null) {
            // editor has programmatically been  closed while the dialog was open
            return;
        }
        
        
        if (provider.isDeleted(input)) {
            /* Begin local modifications */
            /* We don't have access to the TextEditorMessages class, since it's
             * "package private" to the org.eclipse.ui.editors.text package. So
             * we have copied the messages into our own resources file. 
             */
            // String message= MessageFormat.format(TextEditorMessages.getString("Editor.warning.save.delete"), new Object[] { inputName }); //$NON-NLS-1$
            String message= MessageFormat.format(SQLEditorResources.getString("SQLEditor.warning.save.delete"), new Object[] { inputName }); //$NON-NLS-1$
            /* End local modifications */
            dialog.setErrorMessage(null);
            dialog.setMessage(message, IMessageProvider.WARNING);
        }
        
        if (dialog.open() == Window.CANCEL) {
            if (progressMonitor != null)
                progressMonitor.setCanceled(true);
            return;
        }
        
        IPath filePath= dialog.getResult();
        if (filePath == null) {
            if (progressMonitor != null)
                progressMonitor.setCanceled(true);
            return;
        }
        
        IWorkspace workspace= ResourcesPlugin.getWorkspace();
        IFile file= workspace.getRoot().getFile(filePath);
        /* Begin local modifications */
        /* We need to preserve the connection and other information that we got
         * from the original SQL editor input.  So we need to create a new 
         * SQL Editor input and copy the information to it. 
         */
        // final IEditorInput newInput= new FileEditorInput(file);
        IEditorInput newInput = null;
        if (input instanceof ISQLEditorInput) {
            ISQLEditorInput sqlEditorInput = (ISQLEditorInput) input;
            newInput = new SQLEditorFileEditorInput( file );
            SQLEditorFileEditorInput newSQLEditorInput = (SQLEditorFileEditorInput) newInput;
            newSQLEditorInput.setConnectionInfo( sqlEditorInput.getConnectionInfo() );
            newSQLEditorInput.setDatabase( sqlEditorInput.getDatabase() );
            newSQLEditorInput.setDefaultSchemaName( sqlEditorInput.getDefaultSchemaName() );
        }
        else {
            newInput = new FileEditorInput( file );
        }
        /* End local modifications */
        
        boolean success= false;
        try {
            
            provider.aboutToChange(newInput);
            provider.saveDocument(progressMonitor, newInput, provider.getDocument(input), true);            
            success= true;
            
        } catch (CoreException x) {
            IStatus status= x.getStatus();
            if (status == null || status.getSeverity() != IStatus.CANCEL) {
                /* Begin local modifications */
                // String title= TextEditorMessages.getString("Editor.error.save.title"); //$NON-NLS-1$
                // String msg= MessageFormat.format(TextEditorMessages.getString("Editor.error.save.message"), new Object[] { x.getMessage() }); //$NON-NLS-1$
                String title= SQLEditorResources.getString("SQLEditor.error.save.title"); //$NON-NLS-1$
                String msg= MessageFormat.format(SQLEditorResources.getString("SQLEditor.error.save.message"), new Object[] { x.getMessage() }); //$NON-NLS-1$
                /* End local modifications */
                
                if (status != null) {
                    switch (status.getSeverity()) {
                        case IStatus.INFO:
                            MessageDialog.openInformation(shell, title, msg);
                        break;
                        case IStatus.WARNING:
                            MessageDialog.openWarning(shell, title, msg);
                        break;
                        default:
                            MessageDialog.openError(shell, title, msg);
                    }
                } else {
                    MessageDialog.openError(shell, title, msg);
                }
            }
        } finally {
            provider.changed(newInput);
            if (success)
                setInput(newInput);
        }
        
        if (progressMonitor != null)
            progressMonitor.setCanceled(!success);
        /* End code copied from TextEditor.performSaveAs */
    }

}
