/*******************************************************************************
 * Copyright (c) 2000, 2004 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 * IBM Corporation - initial API and implementation
 ******************************************************************************/

package org.eclipse.wst.internet.internal.proxy;

import java.util.Properties;

import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.preference.PreferencePage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPreferencePage;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.help.IWorkbenchHelpSystem;

public class InternetPreferencePage extends PreferencePage implements IWorkbenchPreferencePage
{
  private Button proxyCheckBox_;
  private Text   hostnameTextField_;
  private Text   portTextField_;
  private Button socksCheckBox_;
  private Button nameCheckBox_;
  private Text   userNameTextField_;
  private Text   passwordTextField_;
  
  private NonProxyHostsComposite hostsList_;


  /**
   * Creates preference page controls on demand.
   * @param parent The parent for the preference page.
   */
  protected Control createContents ( Composite superparent )
  {
    Composite parent = new Composite( superparent, SWT.NONE );	
    GridLayout layout = new GridLayout();
    layout.numColumns = 2;
    parent.setLayout( layout );

    proxyCheckBox_ = createCheckBox( parent, InternetPlugin.getMessage( "%CHECKBOX_PREFERENCE_ENABLE_PROXY" ) );
    GridData gd = new GridData();
    gd.horizontalSpan = 2;
    proxyCheckBox_.setLayoutData(gd);
    
    proxyCheckBox_.addSelectionListener(new SelectionAdapter()
      {
        //
        // Enable/disable the text fields based on the checkbox state.
        //
        public void widgetSelected( SelectionEvent evt )
        {
          boolean selected = proxyCheckBox_.getSelection();
          hostnameTextField_.setEnabled( selected );
          portTextField_.setEnabled( selected );
          hostsList_.setEnabled( selected );
          socksCheckBox_.setEnabled(selected);
          socksCheckBox_.setSelection(false);
          nameCheckBox_.setEnabled(selected);          
          nameCheckBox_.setSelection(false);                    
          userNameTextField_.setEnabled( false );
          passwordTextField_.setEnabled( false );          
        }
      }
    );

    createLabel( parent, InternetPlugin.getMessage( "%LABEL_PREFERENCE_HOSTNAME" ) );
    hostnameTextField_ = createTextField( parent );

    createLabel( parent, InternetPlugin.getMessage( "%LABEL_PREFERENCE_PORT" )  );
    portTextField_ = createTextField( parent );
    
    createLabel( parent, InternetPlugin.getMessage( "%LABEL_PREFERENCE_NON_PROXY_HOSTNAMES" )  );  
    hostsList_ = new NonProxyHostsComposite( parent, SWT.NONE );
    
    GridData data = new GridData(GridData.FILL_HORIZONTAL);
    hostsList_.setLayoutData(data);
    
    createLabel( parent, "" );
    createLabel( parent, "" );    
    
    socksCheckBox_ = createCheckBox( parent, InternetPlugin.getMessage( "%CHECKBOX_PREFERENCE_USE_SOCKS" ) );
    gd = new GridData();
    gd.horizontalSpan = 2;
    socksCheckBox_.setLayoutData(gd);
    
    socksCheckBox_.addSelectionListener(new SelectionAdapter()
    {
      public void widgetSelected( SelectionEvent evt )
      {
        boolean selected = socksCheckBox_.getSelection();
        // We only support SOCKSV4. Authentication should be disabled if it isn't already.
        nameCheckBox_.setEnabled(!selected);
        
        if (!nameCheckBox_.getEnabled())
        {
          nameCheckBox_.setSelection(false);
          userNameTextField_.setEnabled(false);
          passwordTextField_.setEnabled(false);
        }
      }
    });

    Label label = createLabel( parent, "");        
    gd = new GridData();
    gd.horizontalSpan = 2;
    label.setLayoutData(gd);
       
    nameCheckBox_ = createCheckBox( parent, InternetPlugin.getMessage( "%CHECKBOX_PREFERENCE_ENABLE_NAME" ) );
    gd = new GridData();
    gd.horizontalSpan = 2;
    nameCheckBox_.setLayoutData(gd);
    
    nameCheckBox_.addSelectionListener(new SelectionAdapter()
      {
        //
        // Enable/disable the text fields based on the checkbox state.
        //
        public void widgetSelected( SelectionEvent evt )
        {
          boolean selected = nameCheckBox_.getSelection();
          userNameTextField_.setEnabled( selected );
          passwordTextField_.setEnabled( selected );
        }
      }
    );

    createLabel( parent, InternetPlugin.getMessage( "%LABEL_PREFERENCE_USERNAME" )  );
    userNameTextField_ = createTextField( parent );

    createLabel( parent, InternetPlugin.getMessage( "%LABEL_PREFERENCE_PASSWORD" )  );
    passwordTextField_ = createTextField( parent );
    passwordTextField_.setEchoChar('*');

    initializeValues();

    IWorkbenchHelpSystem helpSystem = PlatformUI.getWorkbench().getHelpSystem();
    helpSystem.setHelp(proxyCheckBox_, ContextIds.INTERNET_PREF_PROXY);
    helpSystem.setHelp(hostnameTextField_, ContextIds.INTERNET_PREF_PROXY_HOST);
    helpSystem.setHelp(portTextField_, ContextIds.INTERNET_PREF_PROXY_PORT);
    helpSystem.setHelp(socksCheckBox_, ContextIds.INTERNET_PREF_SOCKS);
    helpSystem.setHelp(nameCheckBox_, ContextIds.INTERNET_PREF_AUTHENTICATION);
    helpSystem.setHelp(userNameTextField_, ContextIds.INTERNET_PREF_USERNAME);
    helpSystem.setHelp(passwordTextField_, ContextIds.INTERNET_PREF_PASSWORD);
    helpSystem.setHelp(hostsList_, ContextIds.INTERNET_PREF_NON_PROXY_HOSTS );

    org.eclipse.jface.dialogs.Dialog.applyDialogFont(superparent);

    return parent;
  }

  //
  // Utility method to create a text field.
  //
  private Text createTextField ( Composite parent )
  {
     Text text = new Text(parent, SWT.SINGLE | SWT.BORDER);
     GridData data = new GridData();
     data.verticalAlignment = GridData.FILL;
     data.horizontalAlignment = GridData.FILL;
     data.grabExcessHorizontalSpace = true;
     text.setLayoutData(data);
     return text;
  }

  //
  // Utility method to create a label.
  //
  private Label createLabel ( Composite parent, String text )
  {
    Label label = new Label(parent, SWT.LEFT);
    label.setText(text);
    GridData data = new GridData();
    data.verticalAlignment = GridData.BEGINNING;
    data.horizontalAlignment = GridData.FILL;
    label.setLayoutData(data);
    return label;
  }

  //
  // Utility method to create a checkbox.
  //
  private Button createCheckBox ( Composite parent, String text )
  {
    Button button = new Button(parent, SWT.CHECK);
    button.setText(text);
    return button;
  }

  /**
   * Processes the "Defaults" button.
   */
  protected void performDefaults()
  {
    super.performDefaults();
    initializeDefaults();
  }

  /**
   * Processes the "OK" button.
   * @return True if the page can be dismissed.
   */
  public boolean performOk ()
  {
    storeValues();
    InternetPlugin.getInstance().savePluginPreferences();
    return true;
  }

  /**
   * @see IWorkbenchPreferencePage
   * @param workbench The workbench.
   */
  public void init ( IWorkbench workbench )
  {
  }

  /**
   * Returns the preference store where the proxy information is kept.
   * @return The preference store.
   */
  public IPreferenceStore getPreferenceStore ()
  {
    return InternetPlugin.getInstance().getPreferenceStore();
  }

  /**
   * Initializes states of the controls.
   */
  private void initializeDefaults ()
  {
    hostnameTextField_.setText( "" );
    portTextField_.setText( "" );
    userNameTextField_.setText( "" );
    passwordTextField_.setText( "" );
    hostsList_.setList( "localhost" );

    proxyCheckBox_.setSelection( false );
    socksCheckBox_.setSelection( false );
    nameCheckBox_.setSelection( false );
    hostnameTextField_.setEnabled( false );
    portTextField_.setEnabled( false );
    socksCheckBox_.setEnabled( false );
    nameCheckBox_.setEnabled( false );
    userNameTextField_.setEnabled( false );
    passwordTextField_.setEnabled( false );
    hostsList_.setEnabled( false );
  }

  /**
   * Initializes states of the controls from the preference store.
   */
  private void initializeValues ()
  {
    Properties       sysProps       = System.getProperties();
    String           host           = "";
    String           port           = "";
    boolean          socksSet       = false;
    IPreferenceStore store          = getPreferenceStore();
    String           storeUserName  = store.getString( InternetPlugin.PREFERENCE_USERNAME );
    String           storePassword  = store.getString( InternetPlugin.PREFERENCE_PASSWORD );
    String           storeHost      = store.getString( InternetPlugin.PREFERENCE_HOSTNAME );
    String           storePort      = store.getString( InternetPlugin.PREFERENCE_PORT );
    String           storeNonProxy  = store.getString( InternetPlugin.PREFERENCE_NON_PROXY_HOSTS );
    
    
    if (sysProps.containsKey("proxyHost") && sysProps.containsKey("proxyPort"))
    {
      host = (String)sysProps.get("proxyHost");
      port = (String)sysProps.get("proxyPort");
    }
    else if (sysProps.containsKey("socksProxyHost") && sysProps.containsKey("socksProxyPort"))
    {
      host = (String)sysProps.get("socksProxyHost");
      port = (String)sysProps.get("socksProxyPort");
      socksSet = true;
    }
    else
    {
      host = storeHost;
      port = storePort;
    }
    
    hostnameTextField_.setText(host);
    portTextField_.setText(port);
    
    String nonProxyHosts = (String)sysProps.get( "http.nonProxyHosts" );
    
    if( nonProxyHosts == null || nonProxyHosts.equals("") ) 
    {
      nonProxyHosts = storeNonProxy;  
    }
    
    if( nonProxyHosts == null || nonProxyHosts.equals("") ) 
    {
      nonProxyHosts = "localhost";  
    }
    
    hostsList_.setList( nonProxyHosts );
    
    String  userName     = "";
    String  password     = "";
    boolean httpProxySet = false;
    
    if (sysProps.containsKey("http.proxyUserName") && sysProps.containsKey("http.proxyPassword"))
    {
      userName = (String)sysProps.get("http.proxyUserName");
      password =  (String)sysProps.get("http.proxyPassword");
      httpProxySet = true;
    }
    else
    {
      userName = storeUserName;
      password = storePassword;
    }
    
    userNameTextField_.setText( (Encoder.isEncoded(userName)?Encoder.decode(userName):userName) );
    passwordTextField_.setText( (Encoder.isEncoded(password)?Encoder.decode(password):password) );
    
    boolean checkBoxState = sysProps.containsKey("proxySet") ? Boolean.valueOf((String)sysProps.get("proxySet")).booleanValue() : false;
    boolean checkBoxState2 = socksSet;
    boolean checkBoxState3 = httpProxySet;

    proxyCheckBox_.setSelection( checkBoxState );
    socksCheckBox_.setSelection( checkBoxState2 );
    nameCheckBox_.setSelection( checkBoxState3 );    
    hostnameTextField_.setEnabled( checkBoxState );
    portTextField_.setEnabled( checkBoxState );
    socksCheckBox_.setEnabled( checkBoxState );
    nameCheckBox_.setEnabled( checkBoxState & !checkBoxState2 );    
    userNameTextField_.setEnabled( checkBoxState3 );
    passwordTextField_.setEnabled( checkBoxState3 );
    hostsList_.setEnabled( checkBoxState );
  }

  /**
   * Stores the values of the controls back to the preference store.
   */
  private void storeValues ()
  {
    IPreferenceStore store = getPreferenceStore();

    String userName  = store.getString( InternetPlugin.PREFERENCE_USERNAME );
    String password  = store.getString( InternetPlugin.PREFERENCE_PASSWORD );
    String oldHostsList = store.getString( InternetPlugin.PREFERENCE_NON_PROXY_HOSTS );
    String newHostsList = hostsList_.getList();
      
    boolean isProxyPropertiesUpdated = 
      (proxyCheckBox_.getSelection() != store.getBoolean(InternetPlugin.PREFERENCE_PROXYCHECKED)) ||
      !hostnameTextField_.getText().equals(store.getString(InternetPlugin.PREFERENCE_HOSTNAME)) ||
      !portTextField_.getText().equals(store.getString(InternetPlugin.PREFERENCE_PORT)) ||
      (socksCheckBox_.getSelection() != store.getBoolean(InternetPlugin.PREFERENCE_SOCKSCHECKED)) ||
      (nameCheckBox_.getSelection() != store.getBoolean(InternetPlugin.PREFERENCE_NAMECHECKED)) ||
      !userNameTextField_.getText().equals((Encoder.isEncoded(userName)?Encoder.decode(userName):userName)) ||
      !passwordTextField_.getText().equals((Encoder.isEncoded(password)?Encoder.decode(password):password)) ||
      !newHostsList.equals( oldHostsList ) ;

    if (isProxyPropertiesUpdated)
    {
      store.setValue( InternetPlugin.PREFERENCE_HOSTNAME , hostnameTextField_.getText() );
      store.setValue( InternetPlugin.PREFERENCE_PORT , portTextField_.getText() );
      store.setValue( InternetPlugin.PREFERENCE_USERNAME , Encoder.encode(userNameTextField_.getText() ));
      store.setValue( InternetPlugin.PREFERENCE_PASSWORD , Encoder.encode(passwordTextField_.getText() ));
      store.setValue( InternetPlugin.PREFERENCE_PROXYCHECKED , proxyCheckBox_.getSelection() );
      store.setValue( InternetPlugin.PREFERENCE_SOCKSCHECKED , socksCheckBox_.getSelection() );
      store.setValue( InternetPlugin.PREFERENCE_NAMECHECKED , nameCheckBox_.getSelection() );
      store.setValue( InternetPlugin.PREFERENCE_NON_PROXY_HOSTS, newHostsList );
      InternetPlugin.getInstance().updateProxyProperties();
    }
  }
  
 /*
  * @see PreferencePage#createControl(Composite)
  */
  public void createControl(Composite parent) 
  {
	super.createControl(parent);
	PlatformUI.getWorkbench().getHelpSystem().setHelp(getControl(), ContextIds.INTERNET_PREF_PROXY);
  }
}
