/*******************************************************************************
 * Copyright (c) 2001, 2004 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.wst.rdb.internal.derby.catalog;

import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.ResultSet;
import java.sql.Statement;
import java.util.Iterator;
import java.util.Map;
import java.util.StringTokenizer;
import java.util.TreeMap;

import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.wst.rdb.internal.core.RDBCorePlugin;
import org.eclipse.wst.rdb.internal.core.definition.DataModelElementFactory;
import org.eclipse.wst.rdb.internal.core.definition.DatabaseDefinition;
import org.eclipse.wst.rdb.internal.core.rte.ICatalogObject;
import org.eclipse.wst.rdb.internal.core.rte.RefreshManager;
import org.eclipse.wst.rdb.internal.models.dbdefinition.PredefinedDataTypeDefinition;
import org.eclipse.wst.rdb.internal.models.sql.constraints.Constraint;
import org.eclipse.wst.rdb.internal.models.sql.constraints.ForeignKey;
import org.eclipse.wst.rdb.internal.models.sql.constraints.IncrementType;
import org.eclipse.wst.rdb.internal.models.sql.constraints.Index;
import org.eclipse.wst.rdb.internal.models.sql.constraints.IndexMember;
import org.eclipse.wst.rdb.internal.models.sql.constraints.SQLConstraintsPackage;
import org.eclipse.wst.rdb.internal.models.sql.constraints.UniqueConstraint;
import org.eclipse.wst.rdb.internal.models.sql.datatypes.PredefinedDataType;
import org.eclipse.wst.rdb.internal.models.sql.schema.Database;
import org.eclipse.wst.rdb.internal.models.sql.schema.ReferentialActionType;
import org.eclipse.wst.rdb.internal.models.sql.schema.Schema;
import org.eclipse.wst.rdb.internal.models.sql.statements.SQLStatement;
import org.eclipse.wst.rdb.internal.models.sql.statements.SQLStatementDefault;
import org.eclipse.wst.rdb.internal.models.sql.statements.SQLStatementsPackage;
import org.eclipse.wst.rdb.internal.models.sql.tables.ActionGranularityType;
import org.eclipse.wst.rdb.internal.models.sql.tables.ActionTimeType;
import org.eclipse.wst.rdb.internal.models.sql.tables.BaseTable;
import org.eclipse.wst.rdb.internal.models.sql.tables.Column;
import org.eclipse.wst.rdb.internal.models.sql.tables.SQLTablesPackage;
import org.eclipse.wst.rdb.internal.models.sql.tables.Table;
import org.eclipse.wst.rdb.internal.models.sql.tables.impl.PersistentTableImpl;

public class DerbyCatalogTable extends PersistentTableImpl implements ICatalogObject {
	private static final long serialVersionUID = 3257854259579074868L;
	public void refresh() {
		this.columnsLoaded = false;
		if (this.constraintLoaded) {
			super.getConstraints().clear();
			this.constraintLoaded = false;
		}
		if (this.indexLoaded) {
			super.getIndex().clear();
			this.indexLoaded = false;
		}
		this.triggerLoaded = false;
		RefreshManager.getInstance().referesh(this);
	}

	public boolean isSystemObject() {
		return false;
	}

	public Connection getConnection() {
		Database database = this.getCatalogDatabase();
		return ((DerbyCatalogDatabase) database).getConnection();
	}
	
	public Database getCatalogDatabase() {
		return this.getSchema().getDatabase();		
	}
	
	public EList getColumns() {
		if(!this.columnsLoaded) this.loadColumns();
		return this.columns;
	}

	public EList getConstraints() {
		if(!this.constraintLoaded) this.loadConstraints();
		return this.constraints;
	}
	
	public EList getTriggers() {
		if(!this.triggerLoaded) this.loadTriggers();
		return this.triggers;
	}
	
	public EList getIndex() {
		if(!this.indexLoaded) this.loadIndexes();
		return this.index;
	}
	
	public boolean eIsSet(EStructuralFeature eFeature) {
		int id = eDerivedStructuralFeatureID(eFeature);
		if(id == SQLTablesPackage.TABLE__COLUMNS) {
			this.getColumns();
		}
		else if(id == SQLConstraintsPackage.TABLE_CONSTRAINT) {
			this.getConstraints();
		}
		else if(id == SQLTablesPackage.TABLE__INDEX) {
			this.getIndex();
		}
		else if(id == SQLTablesPackage.TABLE__TRIGGERS) {
			this.getTriggers();
		}
		
		return super.eIsSet(eFeature);
	}
	
	
	private synchronized void loadColumns() {
		if(this.columnsLoaded) return;
		EList columnList = super.getColumns();
		Object[] list = columnList.toArray();
		columnList.clear();
		Connection connection = this.getConnection();
		
		boolean deliver = this.eDeliver();
		this.eSetDeliver(false);	
		
		try {
			DatabaseMetaData metaData = this.getConnection().getMetaData();
			ResultSet r = metaData.getColumns(null, this.getSchema().getName(), this.getName(), null);
			while(r.next()) {
				Column column;
				final String columnName = r.getString(4);

				Object element = DerbyCatalogSchema.findElement(list, columnName,SQLTablesPackage.eINSTANCE.getColumn());
				
				if (element != null){
					column = (Column) element;
					((ICatalogObject)element).refresh();
				} else {
					column = new DerbyCatalogColumn();
				}	
				
				column.setName(columnName);
				
				final String remarks = r.getString(12);
				column.setDescription(remarks);

//				String defaultValue = r.getString(13);
//				column.setDefaultValue(defaultValue);

				String typeName = r.getString(6);
				
				final DatabaseDefinition databaseDefinition = this.getDatabaseDefinition();
				PredefinedDataTypeDefinition typeDefinition = databaseDefinition.getPredefinedDataTypeDefinition(typeName);
				if(typeDefinition != null) {
					PredefinedDataType type = databaseDefinition.getPredefinedDataType(typeDefinition);
					if(typeDefinition.isLengthSupported()) {
						EStructuralFeature feature = type.eClass().getEStructuralFeature("length");  //$NON-NLS-1$
						type.eSet(feature, new Integer(r.getInt(7)));
					}
					else if(typeDefinition.isPrecisionSupported()) {
						EStructuralFeature feature = type.eClass().getEStructuralFeature("precision"); //$NON-NLS-1$
						type.eSet(feature, new Integer(r.getInt(7)));
					}
					
					if(typeDefinition.isScaleSupported()) {
						EStructuralFeature feature = type.eClass().getEStructuralFeature("scale"); //$NON-NLS-1$
						type.eSet(feature, new Integer(r.getInt(9)));
					}
					column.setContainedType(type);
				}
				else {
				}

				final String nulls = r.getString(18);
				if(nulls.equals("YES")) column.setNullable(true);  //$NON-NLS-1$
				else column.setNullable(false);

				columnList.add(column);
			}
			this.columnsLoaded = true;
			r.close();
		}
		catch (Exception e) {
			System.out.println(e.toString());
		}
		
		this.eSetDeliver(deliver);
	}
	
	private synchronized void loadConstraints() {
		if(this.constraintLoaded) return;
		Connection connection = this.getConnection();
		if(connection == null) return;
		
		boolean deliver = this.eDeliver();
		this.eSetDeliver(false);	
		
		try {
			DatabaseMetaData metaData = connection.getMetaData();
			
			DerbyCatalogPrimaryKey  pk= this.loadPrimaryKey(metaData);
			
			DerbyCatalogTable.loadForeignKey(connection,super.getConstraints(),this);
			DerbyCatalogTable.loadCheckConstraint(connection, super.getConstraints(),this);
		}
		catch (Exception e) {
			System.out.println(e.toString());
		}
		this.constraintLoaded = true;
		this.eSetDeliver(deliver);
	}
		
	private synchronized DerbyCatalogPrimaryKey loadPrimaryKey(DatabaseMetaData metaData){
		DerbyCatalogPrimaryKey key = null;
		try {
			ResultSet r = metaData.getPrimaryKeys(null, this.getSchema().getName(),this.getName());
	
			DataModelElementFactory factory = RDBCorePlugin.getDefault().getDatabaseDefinitionRegistry().getDefinition(this.getCatalogDatabase()).getDataModelElementFactory();

			KeyColumnCollection columns = new KeyColumnCollection(this);  
			
			while(r.next()) {
				if(key == null) {
					String name = r.getString(6);
					key = new DerbyCatalogPrimaryKey();
					key.setName(name);
					super.getConstraints().add(key);
				}
				String columnName = r.getString(4);
				int seq = r.getInt(5);
				columns.add(seq, columnName);
			}
			
			Iterator it = columns.iterator();
			while(it.hasNext()) {
				Column column = (Column) it.next();
				key.getMembers().add(column); 
			}
			r.close();
		}catch (Exception e){
		}
		
		return key;
	}

	private synchronized void loadReferenceForeignKey(DatabaseMetaData metaData, UniqueConstraint uk){
		try {
			ResultSet r = metaData.getExportedKeys(null, this.getSchema().getName(),this.getName());
	
			DataModelElementFactory factory = RDBCorePlugin.getDefault().getDatabaseDefinitionRegistry().getDefinition(this.getCatalogDatabase()).getDataModelElementFactory();
			
			ForeignKey fk = null;
			Table fkTable=null;
			String fkTableName=""; //$NON-NLS-1$
			while(r.next()) {
				final String fkSchema_Name= r.getString(6);
				final String fkTable_Name = r.getString(7);
				if(!fkTableName.equals(fkTable_Name)) {
					fkTable= this.getTable(fkSchema_Name,fkTable_Name);
					fkTableName = fkTable_Name;
					final String fkName = r.getString(12);
					
					fk = this.getForeignkey(fkSchema_Name,fkTable_Name,fkName);
					
					fk.setUniqueConstraint(uk);

					short updateRule = r.getShort(10);
					switch(updateRule) {
						case DatabaseMetaData.importedKeyCascade:
							fk.setOnUpdate(ReferentialActionType.CASCADE_LITERAL);
							break;
						case DatabaseMetaData.importedKeyRestrict:
							fk.setOnUpdate(ReferentialActionType.RESTRICT_LITERAL);
							break;
						case DatabaseMetaData.importedKeySetDefault:
							fk.setOnUpdate(ReferentialActionType.SET_DEFAULT_LITERAL);
							break;
						case DatabaseMetaData.importedKeySetNull:
							fk.setOnUpdate(ReferentialActionType.SET_NULL_LITERAL);
							break;
						case DatabaseMetaData.importedKeyNoAction:
							default:
							fk.setOnUpdate(ReferentialActionType.NO_ACTION_LITERAL);
							break;
					}
					short deleteRule = r.getShort(11);
					switch(deleteRule) {
						case DatabaseMetaData.importedKeyCascade:
							fk.setOnDelete(ReferentialActionType.CASCADE_LITERAL);
							break;
						case DatabaseMetaData.importedKeyRestrict:
							fk.setOnDelete(ReferentialActionType.RESTRICT_LITERAL);
							break;
						case DatabaseMetaData.importedKeySetDefault:
							fk.setOnDelete(ReferentialActionType.SET_DEFAULT_LITERAL);
							break;
						case DatabaseMetaData.importedKeySetNull:
							fk.setOnDelete(ReferentialActionType.SET_NULL_LITERAL);
							break;
						case DatabaseMetaData.importedKeyNoAction:
							default:
							fk.setOnDelete(ReferentialActionType.NO_ACTION_LITERAL);
							break;
					}
					short deferrability = r.getShort(14);
					switch(deferrability) {
						case DatabaseMetaData.importedKeyInitiallyDeferred:
							fk.setDeferrable(true);
							fk.setInitiallyDeferred(true);
							break;
						case DatabaseMetaData.importedKeyInitiallyImmediate:
							fk.setDeferrable(true);
							fk.setInitiallyDeferred(false);
							break;
						case DatabaseMetaData.importedKeyNotDeferrable:
						default:
							fk.setDeferrable(false);
							break;
					}
					((BaseTable)fkTable).getConstraints().add(fk);
				}
				
				String columnName = r.getString(8);
				Column column =DerbyCatalogTable.getColumn(fkTable,columnName);
				fk.getMembers().add(column);
			}
			r.close();
		}catch (Exception e){
		}
	}

	public static void loadForeignKey(Connection connection, EList constraints, Table table){
		try {
			final String query = "SELECT CONSTRAINTNAME,DELETERULE,UPDATERULE" + //$NON-NLS-1$
			" FROM SYS.SYSCONSTRAINTS FK,SYS.SYSFOREIGNKEYS R, SYS.SYSTABLES T, SYS.SYSSCHEMAS S"+ //$NON-NLS-1$
			" WHERE T.SCHEMAID=S.SCHEMAID" + //$NON-NLS-1$
			" AND S.SCHEMANAME='"+ table.getSchema().getName()+"'" + //$NON-NLS-1$ //$NON-NLS-2$
			" AND T.TABLEID= FK.TABLEID" + //$NON-NLS-1$
			" AND T.TABLENAME='"+ table.getName()+"'" + //$NON-NLS-1$ //$NON-NLS-2$
			" AND FK.CONSTRAINTID=R.CONSTRAINTID"; //$NON-NLS-1$
			
			Statement s = connection.createStatement();
			ResultSet r = s.executeQuery(query); 
			while(r.next()) {
				DerbyCatalogForeignKey fk = new DerbyCatalogForeignKey();
				final String fkName = r.getString("CONSTRAINTNAME"); //$NON-NLS-1$
				fk.setName(fkName);
				String updateRule = r.getString("UPDATERULE"); //$NON-NLS-1$
				if (updateRule.equals("S")) //$NON-NLS-1$
					fk.setOnUpdate(ReferentialActionType.RESTRICT_LITERAL);
				else
					fk.setOnUpdate(ReferentialActionType.NO_ACTION_LITERAL);

				String deleteRule = r.getString("DELETERULE"); //$NON-NLS-1$
				if (updateRule.equals("C"))  //$NON-NLS-1$
					fk.setOnDelete(ReferentialActionType.CASCADE_LITERAL);
				else if (updateRule.equals("S")) //$NON-NLS-1$
					fk.setOnDelete(ReferentialActionType.RESTRICT_LITERAL);
				else if (updateRule.equals("U")) //$NON-NLS-1$
					fk.setOnDelete(ReferentialActionType.SET_NULL_LITERAL);
				else
					fk.setOnDelete(ReferentialActionType.NO_ACTION_LITERAL);
				
				constraints.add(fk);
			}
			r.close();
			s.close();
		}catch (Exception e){
		}
	}
	
	public static void loadCheckConstraint(Connection connection, EList constraints, Table table){
		final Database database = table.getSchema().getDatabase();
		final DatabaseDefinition databaseDefinition = RDBCorePlugin.getDefault().getDatabaseDefinitionRegistry().getDefinition(database);
		final DataModelElementFactory factory = databaseDefinition.getDataModelElementFactory();
		final String query = "SELECT CONSTRAINTNAME,CHECKDEFINITION" + //$NON-NLS-1$
							" FROM SYS.SYSCONSTRAINTS A,SYS.SYSSCHEMAS B, SYS.SYSTABLES C, SYS.SYSCHECKS D"+ //$NON-NLS-1$
							" WHERE A.TYPE='C'" + //$NON-NLS-1$
							" AND A.SCHEMAID=B.SCHEMAID" + //$NON-NLS-1$
							" AND B.SCHEMANAME='"+ table.getSchema().getName()+"'" +  //$NON-NLS-1$//$NON-NLS-2$
							" AND A.TABLEID= C.TABLEID" + //$NON-NLS-1$
							" AND C.TABLENAME='"+ table.getName()+"'" +  //$NON-NLS-1$//$NON-NLS-2$
							" AND A.CONSTRAINTID=D.CONSTRAINTID"; //$NON-NLS-1$
		try {

			
			Statement s = connection.createStatement();
			ResultSet r = s.executeQuery(query); 
			while(r.next()) {
				final String check_name = r.getString("CONSTRAINTNAME"); //$NON-NLS-1$
				DerbyCatalogCheckConstraint check= new DerbyCatalogCheckConstraint();
				check.setName(check_name);
				constraints.add(check);
			}
			r.close();
			s.close();
		}catch(Exception e){
			System.out.println(e.toString());
		}
	}
	
	private synchronized void loadTriggers(){
		if(this.triggerLoaded) return;
		EList triggerList = super.getTriggers();
		Object[] list = triggerList.toArray();
		triggerList.clear();
		Connection connection = this.getConnection();
		
		boolean deliver = this.eDeliver();
		this.eSetDeliver(false);	

		try {
			DataModelElementFactory factory = RDBCorePlugin.getDefault().getDatabaseDefinitionRegistry().getDefinition(this.getCatalogDatabase()).getDataModelElementFactory();

			Statement s = connection.createStatement();
			final String query = "SELECT TRIGGERNAME, SCHEMANAME,EVENT,FIRINGTIME,TYPE,TRIGGERDEFINITION" + //$NON-NLS-1$
								",REFERENCINGOLD,REFERENCINGNEW,OLDREFERENCINGNAME,NEWREFERENCINGNAME,REFERENCEDCOLUMNS" + //$NON-NLS-1$
								" FROM SYS.SYSTRIGGERS A, SYS.SYSTABLES B, SYS.SYSSCHEMAS C"+ //$NON-NLS-1$
								" WHERE A.TABLEID=B.TABLEID"+ //$NON-NLS-1$
								" AND B.TABLENAME='"+ this.getName()+"'"+  //$NON-NLS-1$//$NON-NLS-2$
								" AND A.SCHEMAID=C.SCHEMAID"; //$NON-NLS-1$
			ResultSet r = s.executeQuery(query); 
			while(r.next()) {
				DerbyCatalogTrigger trigger;
				final String trigName = r.getString("TRIGGERNAME"); //$NON-NLS-1$

				EClass metaclass= SQLTablesPackage.eINSTANCE.getTrigger();

				Object element = DerbyCatalogSchema.findElement(list, trigName,metaclass);
				
				if (element != null){
					trigger = (DerbyCatalogTrigger) element;
					((ICatalogObject)element).refresh();
				} else {
					trigger = new DerbyCatalogTrigger();
					trigger.setName(trigName);
				}	
				
				
				
				trigger.setName(trigName);
				
				final String trigSchema = r.getString("SCHEMANAME"); //$NON-NLS-1$
				trigger.setSchema(this.getSchema(trigSchema));

				final String trigTime = r.getString("FIRINGTIME"); //$NON-NLS-1$
				if(trigTime.equals("A")) trigger.setActionTime(ActionTimeType.AFTER_LITERAL); //$NON-NLS-1$
				else if(trigTime.equals("B")) trigger.setActionTime(ActionTimeType.BEFORE_LITERAL); //$NON-NLS-1$
				else if(trigTime.equals("I")) trigger.setActionTime(ActionTimeType.INSTEADOF_LITERAL); //$NON-NLS-1$
				else continue;
				
				final String trigEvent = r.getString("EVENT"); //$NON-NLS-1$
				if(trigEvent.equals("I")) trigger.setInsertType(true); //$NON-NLS-1$
				if(trigEvent.equals("D")) trigger.setDeleteType(true); //$NON-NLS-1$
				if(trigEvent.equals("U")) trigger.setUpdateType(true); //$NON-NLS-1$

				final String granularity = r.getString("TYPE"); //$NON-NLS-1$
				if(granularity.equals("S")) trigger.setActionGranularity(ActionGranularityType.STATEMENT_LITERAL); //$NON-NLS-1$
				else if(granularity.equals("R")) trigger.setActionGranularity(ActionGranularityType.ROW_LITERAL); //$NON-NLS-1$
				else continue;
				
				final boolean referencingOld = r.getBoolean("REFERENCINGOLD"); //$NON-NLS-1$
				if (referencingOld) {
					trigger.setOldRow(r.getString("OLDREFERENCINGNAME")); //$NON-NLS-1$
				}

				final boolean referencingNew = r.getBoolean("REFERENCINGNEW"); //$NON-NLS-1$
				if (referencingNew) {
					trigger.setNewRow(r.getString("NEWREFERENCINGNAME")); //$NON-NLS-1$
				}

				String columns = r.getString("REFERENCEDCOLUMNS"); //$NON-NLS-1$
				if (columns != null){
					EList updateColumns = trigger.getTriggerColumn();
		            EList columnList = this.getColumns();
		            columns = columns.substring(columns.indexOf("(")+1,columns.indexOf(")")); //$NON-NLS-1$ //$NON-NLS-2$
			        StringTokenizer tokenizer = new StringTokenizer(columns, ","); //$NON-NLS-1$
			        byte previousToken = 0;
			        while (tokenizer.hasMoreTokens()) {
			            String token = tokenizer.nextToken().trim();
			            Column column = (Column) columnList.get(Integer.parseInt(token)-1);
			            updateColumns.add(column);
			        }
				}
				
				//body
				final String text = r.getString("TRIGGERDEFINITION"); //$NON-NLS-1$
				SQLStatement body = (SQLStatement) factory.create(SQLStatementsPackage.eINSTANCE.getSQLStatementDefault());
				((SQLStatementDefault)body).setSQL(text);
				trigger.getActionStatement().add(body);
				
				triggerList.add(trigger);
			}

			this.triggerLoaded = true;
			r.close();
			s.close();
			
		}catch (Exception e){
			System.out.println(e.toString());
		}
		
		this.eSetDeliver(deliver);
		
	}
	
	private synchronized void loadIndexes() {
		if(this.indexLoaded) return;
		EList indexList = super.getIndex();
		Connection connection = this.getConnection();
		
		boolean deliver = this.eDeliver();
		this.eSetDeliver(false);	
		
		try {
			DataModelElementFactory factory = this.getDatabaseDefinition().getDataModelElementFactory();
			
			DatabaseMetaData metaData = connection.getMetaData();
			ResultSet r = metaData.getIndexInfo(null,null,this.getName(),false,false);
			Index index = null;
			String indexName=""; //$NON-NLS-1$
			while(r.next()) {
				final String indName = r.getString(6);
				if (!indName.equals(indexName)){
					indexName = indName;
					index = new DerbyCatalogIndex();
					index.setName(indName);
	
					final String indSchema = r.getString(2);
					index.setSchema(this.getSchema(indSchema));

					index.setUnique(r.getBoolean(4)? false:true);

					final short type = r.getShort(7);
					if (type == DatabaseMetaData.tableIndexClustered) {
						index.setClustered(true);
					}
					
					indexList.add(index);
				}
				
				final String column_name= r.getString(9);
				if (column_name !=null){
					IndexMember member = (IndexMember) factory.create(SQLConstraintsPackage.eINSTANCE.getIndexMember()); 
					member.setColumn(DerbyCatalogTable.getColumn(this,column_name));
					final String order = r.getString(10);
					if(order.equals("A"))  //$NON-NLS-1$
						member.setIncrementType(IncrementType.ASC_LITERAL);
					else if(order.equals("D"))  //$NON-NLS-1$
						member.setIncrementType(IncrementType.DESC_LITERAL);
					
					index.getMembers().add(member);
				}
				
			}
			this.indexLoaded = true;
			r.close();
		}
		catch (Exception e) {
			System.out.println(e.toString());
		}
		
		this.eSetDeliver(deliver);
	}
	
	
	private UniqueConstraint getRefrencedUniqueConstraint(String pkSchema,String pkTable, String pkName){
		BaseTable table = (BaseTable) this.getTable(pkSchema, pkTable);
		Iterator it = table.getConstraints().iterator();
		while(it.hasNext()) {
			Constraint constraint = (Constraint) it.next();
			if(constraint instanceof UniqueConstraint && constraint.getName().equals(pkName)) return (UniqueConstraint) constraint;			
		}
		return null;		
	}

	
	private DatabaseDefinition getDatabaseDefinition() {
		Database d = this.getSchema().getDatabase();
		return RDBCorePlugin.getDefault().getDatabaseDefinitionRegistry().getDefinition(d);
	}
	
	static Column getColumn(Table table,String columnName) {
		Iterator it = table.getColumns().iterator();
		while(it.hasNext()) {
			Column c = (Column) it.next();
			if(c.getName().equals(columnName)) return c;
		}
		return null;
	}
	
	private Schema getSchema(String schemaName) {
		Schema s = this.getSchema();
		if(s.getName().equals(schemaName)) return s;
		Database d = s.getDatabase();
		Iterator it = d.getSchemas().iterator();
		while(it.hasNext()) {
			s = (Schema) it.next();
			if(s.getName().equals(schemaName)) return s;
		}
		Schema schema = new DerbyCatalogSchema();
		schema.setName(schemaName);
		schema.setDatabase(d);
		
		return schema;
	}

	private Table getTable(String schemaName, String tableName) {
		Schema schema = this.getSchema(schemaName);
		Iterator it = schema.getTables().iterator();
		while(it.hasNext()) {
			Table table = (Table) it.next();
			if(table.getName().equals(tableName)) return table;			
		}
		Table table = new DerbyCatalogTable();
		table.setName(tableName);
		table.setSchema(schema);
		return table;		
	}
	
	private ForeignKey getForeignkey(String schemaName, String tableName, String constraintName) {
		BaseTable table = (BaseTable) this.getTable(schemaName, tableName);
		Iterator it = table.getConstraints().iterator();
		while(it.hasNext()) {
			Constraint constraint = (Constraint) it.next();
			if(constraint.getName().equals(constraintName)) return (ForeignKey) constraint;			
		}
		
		ForeignKey fk = new DerbyCatalogForeignKey();
		fk.setName(constraintName);
		fk.setBaseTable(table);
		
		return fk;		
	}
	
	private static class KeyColumnCollection {
		public KeyColumnCollection(BaseTable table) {
			this.table = table;
		}
		
		public void add(int seq, String name) {
			Column column = this.getColumn(name);
			String key = "k" + seq; //$NON-NLS-1$
			this.keyMap.put(key, column);
		}
		
		public Iterator iterator() {
			return keyMap.values().iterator();
		}
		
		private Column getColumn(String name) {
			Iterator it = this.table.getColumns().iterator();
			while(it.hasNext()) {
				Column column = (Column) it.next();
				if(column.getName().equals(name)) return column;
			}
			return null;
		}
		
		private Map keyMap = new TreeMap();
		private BaseTable table;
	}
	
	private boolean columnsLoaded=false;
	private boolean constraintLoaded=false;
	private boolean triggerLoaded = false;
	private boolean indexLoaded = false;
	
	
}
