/*******************************************************************************
 * Copyright (c) 2001, 2004 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.wst.rdb.data.internal.core.editor;

import java.io.*;
import java.sql.*;

import org.eclipse.wst.rdb.data.internal.core.common.*;


public class RowDataImpl implements IRowData {
    
    // Possible state transitions ( OLD STATE - action - NEW STATE)
    // load - ORIGINAL
    // insert - INSERTED
    // ORIGINAL - update - UPDATED
    // ORIGINAL - delete - DELETED
    // UPDATED - delete - DELETED
    // INSERTED - delete
    // UPDATED - save - ORIGINAL
    // INSERTED - save - ORIGINAL
    // INSERTED - revert
    // UPDATED - revert - ORIGINAL
    // DELETED - revert - ORIGINAL
    
    protected TableDataImpl table;
    
    protected int state;    
    public static final int STATE_ORIGINAL = 0;
    public static final int STATE_UPDATED = 1;
    public static final int STATE_DELETED = 2;
    public static final int STATE_INSERTED = 3;
    
    protected Object[] newData;
    protected Object[] oldData;
    
    public RowDataImpl(TableDataImpl table, int type, Object[] data)
    {
        this.table = table;
        this.state = type;
        this.newData = data;
        
        if (type==STATE_ORIGINAL)
            oldData = (Object[])data.clone();
    }
    
    public ITableData getTable()
    {
        return table;
    }
    
    public Object getValue(int col) {
        return newData[col];
    }
    
    public int getState() {
        return state;
    }
    
    public void setState(int state) {
        this.state = state;
    }
    
    public void updateValue(int col, Object value)
    {
        if (state==STATE_ORIGINAL)
            state = STATE_UPDATED;
        newData[col] = value;
    }
    
    public void save(TableDataSaveStatus status) throws SQLException, IOException
    {
        switch (state) {
	        case STATE_UPDATED:
	            doUpdate(status);
	            break;
	        case STATE_INSERTED:
	            doInsert(status);
	            break;
	        case STATE_DELETED:
	            doDelete(status);
	            break;
        	case STATE_ORIGINAL:
        	default:        	    
        }
    }
    
    protected void doInsert(TableDataSaveStatus status) throws SQLException, IOException
    {
        // Write query
        String q = "insert into " + table.getQualifiedTableName(); //$NON-NLS-1$
        q += " " + computeValuesClause(); //$NON-NLS-1$
        
        // Set query arguments
        PreparedStatement pst = table.getConnection().prepareStatement(q);
        setValuesClauseArguments(pst, 0);
        
        // Execute query        
        pst.executeUpdate();      
        pst.close();
        
        status.inserted += 1;
    }
    
    protected void doUpdate(TableDataSaveStatus status) throws SQLException, IOException
    {
        // Write query
        String q = "update " + table.getQualifiedTableName(); //$NON-NLS-1$
        q += " " + computeSetClause(); //$NON-NLS-1$
        q += " " + computeWhereClause(); //$NON-NLS-1$
        
        // Set query arguments
        PreparedStatement pst = table.getConnection().prepareStatement(q);
        int arg = 0;
        arg = setSetClauseArguments(pst, arg);
        arg = setWhereClauseArguments(pst, arg);
        
        // Execute query        
        int n = pst.executeUpdate();      
        pst.close();
        
        status.updated += n;
        if (n!=1)
            status.duplicateRow = true;
    }
    
    protected void doDelete(TableDataSaveStatus status) throws SQLException, IOException
    {
        // Write query
        String q = "delete from " + table.getQualifiedTableName(); //$NON-NLS-1$
        q += " " + computeWhereClause(); //$NON-NLS-1$
        
        // Set query arguments
        PreparedStatement pst = table.getConnection().prepareStatement(q);
        setWhereClauseArguments(pst, 0);
        
        // Execute query        
        int n = pst.executeUpdate();      
        pst.close();
        
        status.deleted += n;
        if (n!=1)
            status.duplicateRow = true;
    }
    
    protected String computeValuesClause()
    {
        String q = "values("; //$NON-NLS-1$
        for (int col=0; col<newData.length; ++col)
            q += (col==0)?"?":",?"; //$NON-NLS-1$ //$NON-NLS-2$
        q += ")"; //$NON-NLS-1$
        return q;
    }

    protected int setValuesClauseArguments(PreparedStatement pst, int arg) throws SQLException, IOException
    {
        for (int col=0; col<newData.length; ++col) {
            PreparedStatementWriter.write(pst, arg, table.getColumnType(col), newData[col]);
            arg++;
        }
        return arg;
    }
    
    protected String computeSetClause()
    {
        String q = "set"; //$NON-NLS-1$
        boolean first = true;
        for (int col=0; col<newData.length; ++col) {
            if (oldData[col]!=newData[col]) {
                q += (first)?" ":", "; //$NON-NLS-1$ //$NON-NLS-2$
                q += table.getQuotedColumnName(col) + "=?"; //$NON-NLS-1$ //$NON-NLS-2$
                first = false;
            }
        }
        return q;
    }
    
    protected int setSetClauseArguments(PreparedStatement pst, int arg) throws SQLException, IOException
    {
        for (int col=0; col<newData.length; ++col)
            if (oldData[col]!=newData[col]) {
                PreparedStatementWriter.write(pst, arg, table.getColumnType(col), newData[col]);
                arg++;
            }
        return arg;
    }
    
    protected String computeWhereClause()
    {
        String q = "where"; //$NON-NLS-1$
        int[] keyColumns = table.getKeyColumns();
        for (int i=0; i<keyColumns.length; ++i) {
            q += (i==0)?" ":" AND "; //$NON-NLS-1$ //$NON-NLS-2$
            q += table.getQuotedColumnName( keyColumns[i] );
            Object val = oldData[ keyColumns[i] ];
            if (val==null)
                q += " is null"; //$NON-NLS-1$
            else
                q += "=?"; //$NON-NLS-1$
        }
        return q;
    }
    
    protected int setWhereClauseArguments(PreparedStatement pst, int arg) throws SQLException, IOException
    {
        int[] keyColumns = table.getKeyColumns();
        for (int i=0; i<keyColumns.length; ++i)
            if (oldData[ keyColumns[i] ]!=null) {
                PreparedStatementWriter.write(pst, arg, table.getColumnType(i), oldData[ keyColumns[i] ]);
                arg++;
            }
        return arg;
    }
    
    /**
     * When the modifications have been saved to the database, this method is called on rows of type
     * TYPE_UPDATED or TYPE_INSERTED to reflect the state of the databse.
     */
    public void resetToOriginal()
    {
        state = STATE_ORIGINAL;
        oldData = (Object[])newData.clone();
    }
    
    public void revertToOriginal()
    {
        state = STATE_ORIGINAL;
        newData = (Object[])oldData.clone();
    }
}
