/*******************************************************************************
 * Copyright (c) 2000, 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 * IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.wst.rdb.connection.internal.ui.wizards;

import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.window.Window;
import org.eclipse.jface.wizard.IWizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Event;
import org.eclipse.wst.rdb.connection.internal.ui.util.resource.ResourceLoader;
import org.eclipse.wst.rdb.connection.internal.ui.wizards.shared.ExistingConnectionsWizardPage;
import org.eclipse.wst.rdb.connection.internal.ui.wizards.shared.UserIdentification;
import org.eclipse.wst.rdb.internal.core.connection.ConnectionInfo;
import org.eclipse.wst.rdb.internal.models.sql.schema.Database;


/**
 * Overrides ExistingConnectionsWizardPage to allow user to pick an existing 
 * connection for which a resource is to be saved to
 * @author Quy V. On, quyon@us.ibm.com
 */
public class ConnectionSelectionExistingWizardPage extends ExistingConnectionsWizardPage
    implements SelectionListener {
 
    private static final ResourceLoader resourceLoader = ResourceLoader.INSTANCE;
    protected boolean myFirstTime = true;
    protected Button reconnectButton;
    private boolean connected;
    
    /**
     * Creates an instance of ConnectionSelectionExistingWizardPage
     * @param aName the name for this page
     */
    public ConnectionSelectionExistingWizardPage(String aName) {
    	super(aName);
    }
    
    /**
     * Creates an instance of ConnectionSelectionExistingWizardPage
     * @param aName the name for this page
     * @param aTitle the title of this page
     * @param titleImage the image for this page
     */
    public ConnectionSelectionExistingWizardPage(String aName, String aTitle, 
            ImageDescriptor titleImage) {
        super(aName);
        setTitle(aTitle);
        setImageDescriptor(titleImage);
    }
    
    public void createControl(Composite composite){
        super.createControl(composite);
        Composite control = (Composite)getControl();
        addReconnectButton(control);        
    }
        
    
    /**
     * Adds the reconnection button
     * @param parent the parent composite
     */
    private void addReconnectButton(Composite parent) {
        reconnectButton = new Button(parent, SWT.PUSH);
        reconnectButton.setText(resourceLoader.queryString(
                "CUI_CONN_SELECT_EXT_RECONNECT_BTN_LBL")); //$NON-NLS-1$
        GridData data = new GridData();
        data.horizontalSpan = 2;
        data.horizontalAlignment = GridData.BEGINNING;
        reconnectButton.setLayoutData(data);
        reconnectButton.addSelectionListener(this);
    }
    
    protected void setConnectionButton() {
        ConnectionInfo[] allInfos = getConnectionsToDisplay();
        for (int index = allInfos.length - 1; index >= 0; index--) {
        	if (allInfos[index].getDatabaseDefinition().getProduct().indexOf("DB2") > -1) {  //$NON-NLS-1$
        		setDefaultConnection(allInfos[index].getName());
        		break;
        	}
        }
    }
    
    /**
     * Gets whether or not the connectionInfo is connected
     * @return true if connected, false if not
     */
    public boolean isConnected() {
    	return connected;
    }
    
    
    /**
     * Calls when default selection occurs (eg return in Text)
     * @param evt the Selection event
     */
    public void widgetDefaultSelected(SelectionEvent evt) {
        // ignore
    }

    /**
     * Calls when selection occurs
     * @param evt the selection event
     */
    public void widgetSelected(SelectionEvent evt) {
        Object source = evt.getSource();
        if (source == reconnectButton) {            
            ConnectionInfo info = getSelectedConnection();
            UserIdentification userDialog = new UserIdentification(info.getUserName());
            if (userDialog.open() == Window.OK) {
                String us = userDialog.getUserNameInformation();
                String pa = userDialog.getPasswordInformation();
                info.setUserName(us);
                info.setPassword(pa);
                // try to connect
                ConnectionSelectionWizard wizard = (ConnectionSelectionWizard)getWizard();
                try {
                	wizard.reconnect(info);
                    reconnectButton.setEnabled(false);
                    connected = true;                    
                    wizard.getContainer().updateButtons();
                }
                catch (Exception ex) {
                	reconnectButton.setEnabled(true);
                    connected = false;
                    MessageDialog.openError(this.getShell(), 
                            resourceLoader.queryString("CUI_CONN_SELECT_EXT_ERROR_DIALOG_TITLE"), //$NON-NLS-1$
                            ex.getMessage());                    
                }                
            }
        }
    }
    
    /**
     * Overrides super to enable and disable reconnect button as needed
     * @param event the selection that trigger this call
     */
    public void handleEvent(Event event) {
        super.handleEvent(event);
        if (isNewConnectionSelected()) {
            reconnectButton.setEnabled(false);
        }
        else {             
            ConnectionSelectionWizard wizard = (ConnectionSelectionWizard)getWizard();
            wizard.getContainer().updateButtons();
        }        
    }
    
    /**
     * Refreshes this page
     */
    public void refresh() {
        Database db = null;
        ConnectionInfo connInfo = getSelectedConnection();
        if (connInfo != null) {
            db = connInfo.getSharedDatabase();
        }
        if (db == null) {
            // connection is not established            
            reconnectButton.setEnabled(true);
            connected = false;
        }
        else {
            reconnectButton.setEnabled(false);  
            connected = true;
        }        
    }
    
    /**
     * Enable the next page only when Create New Connection is selected
     * <p>
     * @see org.eclipse.jface.wizard.IWizardPage#getNextPage()
     */
    public IWizardPage getNextPage() {
        IWizardPage page = null;
        ConnectionSelectionWizard conSelectWiz = (ConnectionSelectionWizard)getWizard();
        if (isExistingConnectionSelected()) {
            ConnectionInfo conInfo = getSelectedConnection();
            conSelectWiz.setConInfo(conInfo); 
            refresh();
        }
        else {
            page = conSelectWiz.getPage(
                    ConnectionSelectionWizard.CONNECTION_SELECTION_JDBCPAGE_NAME);            
        }
        return page;
    }
    
    public void setVisible(boolean visible) {
    	super.setVisible(visible);
        
        if (visible) {
        	IWizardPage filterPage = getWizard().getPage(
                    ConnectionSelectionWizard.CONNECTION_SELECTION_FILTER_NAME);
            if (filterPage != null) {
        		filterPage.getControl().setData(ConnectionSelectionWizard.SKIP, Boolean.TRUE);
        	}            
        }
        
        if (myFirstTime) {
        	setConnectionButton();
        	myFirstTime = false;
        }
    }
}