/*******************************************************************************
 * Copyright (c) 2001, 2004 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.wst.rdb.connection.internal.ui.wizards.shared;

import java.util.Arrays;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.Vector;

import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.List;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.TableItem;
import org.eclipse.swt.widgets.Widget;
import org.eclipse.wst.rdb.connection.internal.ui.util.resource.ResourceLoader;
import org.eclipse.wst.rdb.connection.internal.ui.wizards.ConnectionDisplayProperty;
import org.eclipse.wst.rdb.internal.core.RDBCorePlugin;
import org.eclipse.wst.rdb.internal.core.connection.ConnectionInfo;

/**
 * @author ledunnel
 */
public class ExistingConnectionsWizardPage extends WizardPage implements
		Listener {
	private static final ResourceLoader resource = ResourceLoader.INSTANCE;
	
	private Button newConnectionRadioButton;

	private Button existingConnectionRadioButton;

	private Group existingConnectionsGroup;

	private List existingConnectionsList;

	private Label propertiesLabel;

	private Table connectionPropertiesTable;

	private Hashtable existingConnections;

	private int lastSelectedConnection = -1;

	/**
	 * Use this constructor when you want to provide your own graphic and title
	 * 
	 * @param pageName
	 * @param title
	 * @param titleImage
	 */
	public ExistingConnectionsWizardPage(String pageName, String title,
			ImageDescriptor titleImage) {
		super(pageName, title, titleImage);
	}

	/**
	 * @param pageName
	 */
	public ExistingConnectionsWizardPage(String pageName) {
		super(pageName);
		setTitle(resource.queryString("_UI_PAGE_TITLE_EXISTING_CONNECTIONS")); //$NON-NLS-1$
		this
				.setMessage(resource.queryString("_UI_PAGE_DESCRIPTION_EXISTING_CONNECTIONS")); //$NON-NLS-1$
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.swt.widgets.Listener#handleEvent(org.eclipse.swt.widgets.Event)
	 */
	public void handleEvent(Event event) {
		Widget source = event.widget;
		if (source == newConnectionRadioButton) {
			enableExistingConnectionsControls(existingConnectionRadioButton
					.getSelection());
		} else if (source == existingConnectionsList) {
			updateConnectionProperties();
			setPageComplete(true);
		}
	}

	private void updateConnectionProperties() {
		connectionPropertiesTable.removeAll();
		if (existingConnectionsList.getSelectionIndex() > -1) {
			ConnectionInfo selectedConnection = (ConnectionInfo) existingConnections
					.get((String) existingConnectionsList.getSelection()[0]);
			if (selectedConnection != null) {

			
			ConnectionDisplayProperty[] properties = getConnectionDisplayProperties(selectedConnection);
			if (properties != null){
				int propertyCount =  properties.length;
				for( int index = 0; index < propertyCount; index++){
					TableItem tableItem = new TableItem(connectionPropertiesTable,
							SWT.NONE);
					tableItem.setText(new String[] { properties[index].getPropertyName(),
							properties[index].getValue()});
				}
			}}
		}
	}
	
	private ConnectionDisplayProperty[] getConnectionDisplayProperties(ConnectionInfo connectionInfo){
		ConnectionDisplayProperty[] properties = null;	
		Vector propertiesCollection = new Vector();
		
		propertiesCollection.add(new ConnectionDisplayProperty(resource.queryString("_UI_DATABASE_CONNECTION_PROPERTY_NAME"), connectionInfo.getDatabaseDefinition().getProduct()+ " " + connectionInfo.getDatabaseDefinition().getVersion() )); //$NON-NLS-1$ //$NON-NLS-2$
		propertiesCollection.add(new ConnectionDisplayProperty(resource.queryString("_UI_JDBC_DRIVER_CLASS_CONNECTION_PROPERTY_NAME"), connectionInfo.getDriverClassName())); //$NON-NLS-1$
		propertiesCollection.add(new ConnectionDisplayProperty(resource.queryString("_UI_CLASS_LOCATION_CONNECTION_PROPERTY_NAME"), connectionInfo.getLoadingPath() )); //$NON-NLS-1$
		propertiesCollection.add(new ConnectionDisplayProperty(resource.queryString("_UI_URL_CONNECTION_PROPERTY_NAME"), connectionInfo.getURL() )); //$NON-NLS-1$
		propertiesCollection.add(new ConnectionDisplayProperty(resource.queryString("_UI_USER_ID_CONNECTION_PROPERTY_NAME"),connectionInfo.getUserName() )); //$NON-NLS-1$
		
		properties = new ConnectionDisplayProperty[propertiesCollection.size()];
		propertiesCollection.copyInto(properties);
		
		properties = updateConnectionDisplayProperties(connectionInfo, properties);
		return properties;	
	}

	protected ConnectionDisplayProperty[] updateConnectionDisplayProperties(ConnectionInfo connectionInfo, ConnectionDisplayProperty[] defaultDisplayProperties){
		return defaultDisplayProperties;
	}
	
	private void enableExistingConnectionsControls(boolean isEnabled) {
		existingConnectionsGroup.setEnabled(isEnabled);
		existingConnectionsList.setEnabled(isEnabled);
		propertiesLabel.setEnabled(isEnabled);
		connectionPropertiesTable.setEnabled(isEnabled);
		if (isEnabled) {
			existingConnectionsList.select(lastSelectedConnection);
			updateConnectionProperties();
		} else {
			lastSelectedConnection = existingConnectionsList
					.getSelectionIndex();
			existingConnectionsList.deselectAll();
			updateConnectionProperties();
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.jface.dialogs.IDialogPage#createControl(org.eclipse.swt.widgets.Composite)
	 */
	public void createControl(Composite parent) {
		Composite composite = new Composite(parent, SWT.NONE);
		GridLayout layout = new GridLayout();
		layout.numColumns = 1;
		layout.verticalSpacing = 5;
		composite.setLayout(layout);
		composite.setLayoutData(new GridData(GridData.FILL_BOTH));

		newConnectionRadioButton = new Button(composite, SWT.RADIO);
		newConnectionRadioButton.setText(resource.queryString("_UI_RADIO_BUTTON_NEW_CONNECTION")); //$NON-NLS-1$
		GridData gd = new GridData();
		gd.verticalAlignment = GridData.BEGINNING;
		newConnectionRadioButton.setLayoutData(gd);

		existingConnectionRadioButton = new Button(composite, SWT.RADIO);
		existingConnectionRadioButton.setText(resource.queryString("_UI_RADIO_BUTTON_EXISTING_CONNECTION")); //$NON-NLS-1$
		gd = new GridData();
		gd.verticalAlignment = GridData.BEGINNING;
		existingConnectionRadioButton.setLayoutData(gd);

		Composite indentationComposite = new Composite(composite, SWT.NONE);
		layout = new GridLayout();
		layout.numColumns = 2;
		layout.verticalSpacing = 5;
		indentationComposite.setLayout(layout);
		indentationComposite.setLayoutData(new GridData(GridData.FILL_BOTH));

		existingConnectionsGroup = new Group(indentationComposite, SWT.NONE);
		existingConnectionsGroup.setText(resource.queryString("_UI_GROUP_EXISTING_CONNECTIONS")); //$NON-NLS-1$
		layout = new GridLayout();
		layout.numColumns = 1;
		layout.verticalSpacing = 5;
		existingConnectionsGroup.setLayout(layout);
		gd = new GridData(GridData.FILL_BOTH);
		existingConnectionsGroup.setLayoutData(gd);

		existingConnectionsList = new List(existingConnectionsGroup, SWT.BORDER | SWT.V_SCROLL);
		gd = new GridData(GridData.GRAB_HORIZONTAL | GridData.FILL_HORIZONTAL);
		gd.heightHint = 100;
		existingConnectionsList.setLayoutData(gd);

		propertiesLabel = new Label(existingConnectionsGroup, SWT.NONE);
		propertiesLabel.setText(resource.queryString("_UI_LABEL_PROPERTIES")); //$NON-NLS-1$
		gd = new GridData();
		propertiesLabel.setLayoutData(gd);

		connectionPropertiesTable = new Table(existingConnectionsGroup,
				SWT.BORDER);
		gd = new GridData(GridData.FILL_BOTH);
		connectionPropertiesTable.setLayoutData(gd);
		connectionPropertiesTable.setLinesVisible(true);
		connectionPropertiesTable.setHeaderVisible(true);

		TableColumn tc1 = new TableColumn(connectionPropertiesTable, SWT.NONE);
		tc1.setText(resource.queryString("_UI_TABLE_COLUMN_PROPERTY")); //$NON-NLS-1$
		tc1.setResizable(true);
		tc1.setWidth(140);

		TableColumn tc2 = new TableColumn(connectionPropertiesTable, SWT.NONE);
		tc2.setText(resource.queryString("_UI_TABLE_COLUMN_VALUE")); //$NON-NLS-1$
		tc2.setResizable(true);
		tc2.setWidth(250);

		initializeDialogUnits(composite);
		setControl(composite);

		newConnectionRadioButton.addListener(SWT.Selection, this);
		newConnectionRadioButton.addListener(SWT.Selection, new Listener(){

			public void handleEvent(Event event) {
					onCreateNewConnectionSelectionChanged();			
				
			}
			});
		existingConnectionRadioButton.addListener(SWT.Selection, this);
		existingConnectionsList.addListener(SWT.Selection, this);

		newConnectionRadioButton.setSelection(true);
		enableExistingConnectionsControls(false);
		initializeValues();
		setPageComplete(true);
	}

	private void initializeValues() {
	    existingConnectionsList.removeAll();
		ConnectionInfo[] connInfos = getConnectionsToDisplay();
		if (connInfos != null) {
			existingConnections = new Hashtable();
			Iterator connections = Arrays.asList(connInfos).iterator();
			ConnectionInfo con;
			while (connections.hasNext()) {
				con = (ConnectionInfo) connections.next();
				existingConnections.put(con.getName(), con);
			}
			Enumeration existingConnectionNames = existingConnections.keys();
			while (existingConnectionNames.hasMoreElements()) {
				existingConnectionsList.add(existingConnectionNames
						.nextElement().toString());
			}
		}

		if (existingConnectionsList.getItemCount() > 0) {
			existingConnectionsList.select(0);
			updateConnectionProperties();
			enableExistingConnectionsControls(false);
			existingConnectionRadioButton.setEnabled(true);
		} else {
			enableExistingConnectionsControls(false);
			existingConnectionRadioButton.setEnabled(false);
		}
		newConnectionRadioButton.setSelection(true);
		existingConnectionRadioButton.setSelection(false);
	}

	/**
	 * Returns either the existing connection selected by the user. If the user
	 * has indicated they would like to use a new connection then this method
	 * returns null.
	 * 
	 * @return A ConnectionInfo object representing the users selection in the
	 *         wizard page
	 */
	public ConnectionInfo getSelectedConnection() {
		ConnectionInfo connection = null;
		if (existingConnectionRadioButton.getSelection()) {
			connection = (ConnectionInfo) existingConnections
					.get(existingConnectionsList.getSelection()[0]);
		}
		return connection;
	}

	/**
	 * Returns the list of existing connections to display to the user. Override
	 * this method to provide a filtered list of connections.
	 * 
	 * @return A array of ConnectionInfo objects that should be displayed in the
	 *         existing connections list
	 */
	protected ConnectionInfo[] getConnectionsToDisplay() {
		return RDBCorePlugin.getDefault().getConnectionManager()
				.getAllNamedConnectionInfo();
	}

	/**
	 * Use to determine whether the user wishes to create a new connection
	 * 
	 * @return A boolean indicating the user has chosen a new connection
	 */
	public boolean isNewConnectionSelected() {
		return newConnectionRadioButton.getSelection();
	}

	/**
	 * Use to determine whether the user has selected an existing connection
	 * 
	 * @return A boolean indicating the user has chosen an existing connection
	 */
	public boolean isExistingConnectionSelected() {
		return existingConnectionRadioButton.getSelection();
	}
	
	protected void onCreateNewConnectionSelectionChanged(){
		this.getContainer().updateButtons();
	}
	
	public void setDefaultConnection(String  connectionName){
	    if (existingConnectionsList.indexOf(connectionName) > -1){
	        existingConnectionRadioButton.setSelection(true);
	        existingConnectionRadioButton.setFocus();
	        newConnectionRadioButton.setSelection(false);
	        enableExistingConnectionsControls(true);
	        existingConnectionsList.setSelection(new String[]{connectionName});
            updateConnectionProperties();
            setPageComplete(true);
	    }
	}
	
	public void setVisible(boolean visible){
	    super.setVisible(visible);
	    if (visible){
	        initializeValues();
			if(this.newConnectionRadioButton.getSelection()) {
			    newConnectionRadioButton.setFocus();
    		}
    		else{
    		    existingConnectionRadioButton.setFocus();
    		}
	    }
	}
}