/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.wst.rdb.sqleditor.internal.utils;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.eclipse.jface.text.BadLocationException;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.ITypedRegion;
import org.eclipse.jface.util.Assert;
import org.eclipse.jface.util.IPropertyChangeListener;
import org.eclipse.jface.util.PropertyChangeEvent;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.texteditor.IDocumentProvider;
import org.eclipse.wst.rdb.sqleditor.internal.SQLEditor;
import org.eclipse.wst.rdb.sqleditor.internal.sql.SQLPartitionScanner;

/**
 * This class provides helpful methods for managing the SQL Editor statement 
 * terminator string.
 */
public class SQLStatementTerminatorSupport implements IPropertyChangeListener {
    /** A property name the the statement terminator property for use in 
     * property change notification. */
    public static final String STATEMENT_TERMINATOR = "statementTerminator"; //$NON-NLS-1$
    
    /** The SQL Editor with which this object is associated. */
    private SQLEditor fSQLEditor;
    /** The default statement terminator string. */
    private final String fDefaultStatementTerminator = ";"; //$NON-NLS-1$
    /** Defines the leading part of a tag to use to imbed the statement terminator
     *  string in the document. */
    private final String fStmtTermTag1 = "-- <ScriptOptions statementTerminator=\""; //$NON-NLS-1$
    /** Defines the trailing part of a tag to use to imbed the statement terminator
     *  string in the document. */
    private final String fStmtTermTag2 = "\" />\r"; //$NON-NLS-1$
    /** 
     * Defines a regular expression pattern so that we can locate the statement 
     * terminator string in an XML-like tag within a document.
     * This pattern can be interpreted as follows: 
     *   from the beginning of the input, 
     *   followed by zero or more spaces, 
     *   followed by "--", 
     *   followed by zero or more spaces, 
     *   followed by the string "<ScriptOptions", 
     *   followed by one or more spaces, 
     *   followed by the string "statementTerminator", 
     *   followed by zero or more spaces, 
     *   followed by "=",
     *   followed by zero or more spaces, 
     *   followed by a string of one or more characters surrounded by double-quote
     *     characters (which is the statement terminator), 
     *   followed by any number of characters, 
     *   followed by zero or more "whitespace" characters (including \r and \n).
     * The pattern is not case-sensitive.
     * The statement terminator string pattern itself is in parens in the pattern so that
     * it will be a "group" that we can extract from the matcher.
     */
    private final Pattern fStmtTermTagPattern = Pattern.compile( "\\A[ ]*--[ ]*<ScriptOptions[ ]+statementTerminator[ ]*=[ ]*\"(.+)\".*[\\s]*", Pattern.CASE_INSENSITIVE ); //$NON-NLS-1$
    
    /**
     * This inner class contains information for locating and updating a statement
     * terminator tag within a document.
     */
    private class StatementTerminatorTagInfo {
        IDocument doc;
        ITypedRegion tagRegion;
        Matcher tagMatcher;
    }
    
    /**
     * Constructs an instance of this class and associates it with the given
     * SQL Editor.
     * 
     * @param sqlEditor the SQL Editor to associate with this object
     */
    public SQLStatementTerminatorSupport( SQLEditor sqlEditor ) {
        super();
        fSQLEditor = sqlEditor;
        Assert.isNotNull( fSQLEditor );
    }

    /**
     * Gets the statement terminator string contained in a special "XML tag" 
     * imbedded in the document, if any, using the given document and partition
     * region array.
     *
     * @return the statement terminator string imbedded in the document, or 
     * <code>null</code> if none
     */
    protected StatementTerminatorTagInfo getStatementTerminatorTagInfo() {
        StatementTerminatorTagInfo stmtTermTagInfo = null;

        // Get the document from the SQL Editor.
        IDocument doc = getDocument();
        
        // Get an array of the document's regions.
        ITypedRegion[] regions = SQLPartitionScanner.getDocumentRegions( doc );

        // Process each region in turn, looking for the special XML-like tag that
        // contains a statement terminator.
        ITypedRegion region = null;
        int regionCount = regions.length;
        for (int i=0; i < regionCount && stmtTermTagInfo == null; i++) {
            // Get the next region.
            region = regions[ i ];
            
            // We're only interested in comment regions.
            if (region.getType().equals( SQLPartitionScanner.SQL_COMMENT )) {
                // Get the string represented by the region.
                int regionOffset = region.getOffset();
                int regionLength = region.getLength();
                String regionStr = ""; //$NON-NLS-1$
                try {
                   regionStr = doc.get( regionOffset, regionLength );
                }
                catch (BadLocationException e ) {
                    // ignore, shouldn't happen
                }
                
                // Match the region string against the pattern.
                Matcher stmtTermTagMatcher = fStmtTermTagPattern.matcher( regionStr );
                boolean matches = stmtTermTagMatcher.matches(); 
                
                // If we have a match, get the "group" from the matched pattern
                // that should be the statement terminator string.
                if (matches == true) {
                    stmtTermTagInfo = new StatementTerminatorTagInfo();
                    stmtTermTagInfo.doc = doc;
                    stmtTermTagInfo.tagRegion = region;
                    stmtTermTagInfo.tagMatcher = stmtTermTagMatcher;
                }
            }
        }

        return stmtTermTagInfo;
    }

    /**
     * Gets the default statement terminator string for this editor.
     *  
     * @return the default statement terminator
     */
    public String getDefaultStatementTerminator() {
        return fDefaultStatementTerminator;
    }

    /**
     * Gets the current document being edited by the SQL Editor associated with this object.
     * 
     * @return the current document 
     */
    protected IDocument getDocument() {
        SQLEditor sqlEditor = getSQLEditor();
        IEditorInput editorInput = sqlEditor.getEditorInput();
        IDocumentProvider docProvider = sqlEditor.getDocumentProvider();
        IDocument doc = docProvider.getDocument( editorInput );

        return doc;
    }
    
    /**
     * Gets the SQL Editor which is associated with this object.
     * 
     * @return the SQL Editor associated with this object
     */
    protected SQLEditor getSQLEditor() {
        return fSQLEditor;
    }
    
    /**
     * Gets the current statement terminator string for this editor.
     *  
     * @return the current statement terminator
     */
    public String getStatementTerminator() {
        // Get the default statement terminator.
        String statementTerminator = getDefaultStatementTerminator();
        
        // Determine if there is any statement terminator tag in the document.  If
        // there is, get the statement terminator string from that.
        StatementTerminatorTagInfo stmtTermTagInfo = getStatementTerminatorTagInfo();
        if (stmtTermTagInfo != null) {
            int groupCount = stmtTermTagInfo.tagMatcher.groupCount();
            if (groupCount == 1) {
                statementTerminator = stmtTermTagInfo.tagMatcher.group( 1 );
            }
        }
        
        return statementTerminator;
    }
    
    /**
     * Handles notifications to the object that a property has changed.
     * 
     * @param event the property change event object describing which property
     *            changed and how
     */
    public void propertyChange( PropertyChangeEvent event ) {
        if (event.getProperty().equals( STATEMENT_TERMINATOR )) {
            String statementTerminator = (String) event.getNewValue();
            setStatementTerminator( statementTerminator );
        }
    }

    /**
     * Sets the current statement terminator string for this editor to the given
     * string.  If the given statement terminator is different from the default,
     * an XML-like tag containing the statement terminator will be added to the
     * beginning of the document.  If there already is such a tag, it will be updated
     * with the new statement terminator value.
     *  
     * @param newStatementTerminator the statement terminator to use for this editor
     */
    public void setStatementTerminator( String newStatementTerminator ) {
        // Get the default statement terminator.
        String defaultStatementTerminator = getDefaultStatementTerminator();
        
        // Determine if there is any statement terminator tag in the document.  If
        // there is, get the statement terminator string from that.
        String docStatementTerminator = null;
        StatementTerminatorTagInfo stmtTermTagInfo = getStatementTerminatorTagInfo();
        if (stmtTermTagInfo != null) {
            int groupCount = stmtTermTagInfo.tagMatcher.groupCount();
            if (groupCount == 1) {
                docStatementTerminator = stmtTermTagInfo.tagMatcher.group( 1 );
            }
        }

        // If we have a statement terminator tag in the document, and the new
        // statement terminator being set is different from the one in the tag,  
        // then update the tag.
        if (docStatementTerminator != null && !newStatementTerminator.equals( docStatementTerminator )) {
            IDocument doc = getDocument();
            int tagOffset = stmtTermTagInfo.tagRegion.getOffset();
            int tagLength = stmtTermTagInfo.tagRegion.getLength();

            // If the new statement terminator is the default terminator, remove
            // the tag from the document.
            if (newStatementTerminator.equals( defaultStatementTerminator)) {
                try {
                    doc.replace( tagOffset, tagLength, "" );
                }
                catch(BadLocationException e) {
                    // ignore
                }
            }
            // Otherwise change the tag in the document to contain the new 
            // statement terminator.
            else {
                int stmtTermStartIndex = stmtTermTagInfo.tagMatcher.start( 1 );
                int stmtTermEndIndex = stmtTermTagInfo.tagMatcher.end( 1 );
                int stmtTermLength = stmtTermEndIndex - stmtTermStartIndex;
                try {
                    doc.replace( tagOffset + stmtTermStartIndex, stmtTermLength, newStatementTerminator );
                }
                catch(BadLocationException e) {
                    // ignore
                }
            }
        }
        // We don't have a statement terminator tag in the document.  If the
        // new tag is not the default terminator, add a tag to the document
        // containing the new statement terminator at the start of the document.
        else if (!newStatementTerminator.equals( defaultStatementTerminator)) {
            String stmtTermTag = fStmtTermTag1 + newStatementTerminator + fStmtTermTag2;
            
            IDocument doc = getDocument();
            try {
                doc.replace( 0, 0, stmtTermTag );
            }
            catch(BadLocationException e) {
                // ignore
            }
        }
    }


}
