/*******************************************************************************
* Copyright (c) 2000, 2005 IBM Corporation and others.
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
*
* Contributors:
*     IBM Corporation - initial API and implementation
*******************************************************************************/

package org.eclipse.wst.rdb.internal.outputview;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;

import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.swt.SWT;
import org.eclipse.swt.browser.Browser;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.MessageBox;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.TabFolder;
import org.eclipse.swt.widgets.TabItem;
import org.eclipse.swt.widgets.Text;
import org.eclipse.core.runtime.IStatus;

/**
 * Dialog to show a Browser Composite with rendered XML output
 * @author quyon
 */
public class DialogShowXMLField extends Dialog
{
    protected String xmlText;
    protected String dialogTitle;
    protected Browser myBrowser;
    protected Button btnClose;
    protected Button btnSaveAs;
    private File temporaryFile;
    
    /**
     * Creates an instance of DialogShowXMLField
     * @param shell the parent Shell
     * @param title the dialog title
     * @param text the XML text
     */
    public DialogShowXMLField(Shell shell, String title, String text)
    {
        super(shell);
        xmlText = text;
        dialogTitle = title;        
    }
    
    /**
     * Sets the window title when the shell is configured.
     * @param newShell The shell.
     */
    protected void configureShell(Shell newShell)
    {
       super.configureShell(newShell);
       newShell.setText(dialogTitle);
    }
    
    /** 
     * Sets the shell style when the dialog is opened.
     */
    public int open()
    {
       setShellStyle(getShellStyle() | SWT.RESIZE);
       return super.open();
    }
    
    /**
     * Creates the elements of the dialog area.
     * @param parent The parent composite.
     * @return The dialog area.
     */
    protected Control createDialogArea(Composite parent)
    {
        Composite composite = (Composite) super.createDialogArea(parent);
        GridData gridData;
        
        GridLayout gridLayout = new GridLayout();
        gridLayout.numColumns = 1;
        composite.setLayout(gridLayout);
        
        gridData = new GridData();
        gridData.grabExcessHorizontalSpace = true;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessVerticalSpace = true;
        gridData.verticalAlignment = GridData.FILL;
        
        gridData.widthHint = 480;
        gridData.heightHint = 300;         
        
        TabFolder tabFolder = new TabFolder(composite, SWT.BOTTOM);
        tabFolder.setLayoutData(gridData);
        createRenderedTextTabItem(tabFolder);
        createSourceTabItem(tabFolder);
        
        return composite;
    }
    
    /**
     * Creates the Source TabItem
     * @param parent the parent TabFolder
     */
    private void createSourceTabItem(TabFolder parent)
    {
        Text sourceText = new Text(parent, SWT.MULTI |
                SWT.READ_ONLY | SWT.WRAP | SWT.V_SCROLL);
        sourceText.setText(xmlText);
        TabItem item = new TabItem(parent, SWT.NONE);
        item.setText(OutputViewPlugin.getString("XML_DIALOG_SOURCE"));
        item.setControl(sourceText);
    }
    
    /**
     * Creates the Rendered text TabItem
     * @param parent the parent TabFolder
     */
    private void createRenderedTextTabItem(TabFolder parent)
    {
        myBrowser = new Browser(parent, SWT.READ_ONLY);
        // Ideally, browser.setText() should be used.  But, that method is
        // currently contain many bugs.  To work around the short commings, the
        // xmlText is saved to a temporary file and use the file created to call
        // browser.setURL()        
        temporaryFile = getTempFile();
        if (temporaryFile != null) {
        	try {
        		writeStringToFile(xmlText, temporaryFile.getAbsolutePath());
                String url = temporaryFile.toURL().toString();
                myBrowser.setUrl(url);                
            }
            catch (Exception ex) {
            	myBrowser.setText(xmlText);
            }
        }  
        else {
        	myBrowser.setText(xmlText);  
        }
        TabItem item = new TabItem(parent, SWT.NONE);
        item.setText(OutputViewPlugin.getString("XML_DIALOG_BROWSER"));
        item.setControl(myBrowser);
    }
    
    /**
     * Creates Save As and Close buttons to replace the Ok and Cancel buttons.
     * Save As will save the XML source to a file and Close will use the
     * Cancel-Button-behavior to reuse cancelPressed().
     * @param parent Our parent composite.
     */
    protected void createButtonsForButtonBar(Composite parent)
    {
        //  Overrides Ok button to save XML source when clicked
        btnSaveAs = createButton(parent, IDialogConstants.OK_ID,
             OutputViewPlugin.getString("OV_MESSAGE_SAVE_AS"), false);
        
        btnClose = createButton(parent, IDialogConstants.CANCEL_ID, 
            IDialogConstants.CLOSE_LABEL, false);       
    }
    
    /**
     * Called when Ok(Save As...) button is pressed
     */
    protected void okPressed()
    {
        saveOutput();               
    }
    
    /**
     * Called when Close button is pressed
     */
    protected void cancelPressed() {
    	// remove the temporary file
        if (temporaryFile != null) {
        	temporaryFile.delete();
        }
        super.cancelPressed();
    }
    
    /**
     * Called when user closes dialog some other way rather than with Close button
     */
    protected void handleShellCloseEvent(){
    	super.handleShellCloseEvent();
    	// remove the temporary file
        if (temporaryFile != null) {
            temporaryFile.delete();
        }
    }
    
    /**
     * Displays an error message box to let a user know that the file that is selected
     * is a directory.  This is needed for Linux because the fileChooser allows
     * the directory name as a file.
     * @param aShell the parent shell
     * @param aFile the file name
     */
    protected void displayDirectoryError(Shell aShell, String aFile)
    {
       // display error message because a directory was chosen
       Object[] obj = {aFile};
       String message = OutputViewPlugin.getString(
                "OV_MESSAGE_FILE_ISDIRECTORY_DESC", obj);
       MessageBox box = new MessageBox(aShell, SWT.ICON_ERROR | SWT.OK);
       box.setText(OutputViewPlugin.getString("OV_MESSAGE_FILE_ISDIRECTORY_TITLE"));
       box.setMessage(message);
       box.open();
       // reopen save dialog again
       saveOutput();
    }
    
    /**
     * Launches file chooser to let a user save results to a file.
     */
    private void saveOutput()
    {
        FileDialog fileDialog = new FileDialog(this.getShell(), SWT.SAVE);
        String fileName = fileDialog.open();
        if (fileName != null)
        {
            try
            {
                File file = new File(fileName);
                if (file.isDirectory())
                {
                    displayDirectoryError(this.getShell(), fileName);
                    // return is needed because displayDirectoryError(...) makes
                    // a recursive call to this method.
                    return;
                }
                if (file.exists())
                {
                    // launch question dialog
                    Object[] obj = {fileName};
                    String message = OutputViewPlugin.getString(
                         "OV_MESSAGE_FILE_EXISTS_DESC", obj);
                    MessageBox box = new MessageBox(this.getShell(),
                         SWT.YES | SWT.NO);
                    box.setText(OutputViewPlugin.getString("OV_MESSAGE_FILE_EXISTS_TITLE"));
                    box.setMessage(message);
                    if (box.open() != SWT.YES)
                    {
                        return;
                    }
                }
                BufferedWriter bw = new BufferedWriter(new FileWriter(file));
                bw.write(xmlText);             
                bw.flush();
                bw.close();
            }
            catch (IOException ex)
            {
                MessageBox box = new MessageBox(this.getShell(),
                    SWT.ICON_ERROR);
                box.setText(OutputViewPlugin.getString("OV_STATUS_ERROR"));
                box.setMessage(ex.getMessage());
                box.open();
            }
        }
    }
    
    /**
     * Writes a string to a file
     * @param content the String content
     * @param fileName the system-dependent file name
     */
    private void writeStringToFile(String content, String fileName) throws IOException {
    	BufferedWriter writer = new BufferedWriter(new FileWriter(fileName));
        writer.write(content);
        writer.flush();
        writer.close();
    }
    
    /**
     * Gets the a temporary file.
     * <p>
     * This file will be in the workspace specified by the user when eclipse is launched
     * @return the temporary file
     */
    private File getTempFile() {
        IWorkspaceRoot root = ResourcesPlugin.getWorkspace().getRoot();
        if (root != null) {
            try {
            	File rootFile = root.getLocation().toFile();
                StringBuffer sb = new StringBuffer(rootFile.getAbsolutePath());
                sb.append(System.getProperty("file.separator"))
                  .append("XML_")
                  .append(System.currentTimeMillis())
                  .append(".html");
                File tempFile = new File(sb.toString());                
                return tempFile;
            }
            catch (Exception ex) {
            	OutputViewPlugin.getPlugin().writeLog(IStatus.ERROR, 0, 
                        "DialogShowXMLField:getTempFile()", ex); //$NON-NLS-1$
                return null;
            }
        }
        return null;        
    }    
}