/*******************************************************************************
* Copyright (c) 2000, 2004 IBM Corporation and others.
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
*
* Contributors:
* IBM Corporation - initial API and implementation
*******************************************************************************/
package org.eclipse.wst.rdb.internal.outputview;

import java.util.HashMap;
import org.eclipse.swt.graphics.Image;

/**
 *  The OutputItem represents an entry in the action list (table on left side of the
 *  Output View) and it contains the object name against which an action is performed,
 *  the actions name and the status and optionally a unique name.
 *  To use the Output View, you must first instantiate an OuputItem object.
 *  <p>
 *  The OutputItem is used as input to all methods against the Output View to provide
 *  the action list item for which a messages is to be displayed, replaced or cleared,
 *  for status changes, and to display parameters and results when applicable for
 *  actions against database objects (see {@link #OutputViewAPI} for details).
 *  <p>
 *  When an OutputItem is passed as input, it is added as a new entry to the
 *  action list if it doesn't already exist.   If it already exists,
 *  the information (such as message, paramaters) is added to the already existing OutputItem.
 *
 *  <p>
 */
public class OutputItem
{
   /** Status of an action that has been started. */
   public static final int STATUS_START =	1;
   /** Status of an action that has succeeded. */
   public static final int STATUS_SUCCESS =	2;
   /** Status of an action that has succeeded with a warning. */
   public static final int STATUS_WARNING =	3;
   /** Status of an action that has failed. */
   public static final int STATUS_FAILURE =	4;
   /** Status of an action that is in progress. */
   public static final int STATUS_IN_PROGRESS =	5;
   /** Status of an action that has failed with a critical error. */
   public static final int STATUS_CRITICAL_ERROR =	6;

   /** Code for the run action. */
   public static final int ACTION_RUN =	20;
   /** Code for the build action. */
   public static final int ACTION_BUILD =	21;
   /** Code for the call action. */
   public static final int ACTION_CALL =	22;
   /** Code for the execute action. */
   public static final int ACTION_EXECUTE =	23;
   /** Code for the open action. */
   public static final int ACTION_OPEN =	24;
   /** Code for the save action. */
   public static final int ACTION_SAVE =	25;
   /** Code for the close action. */
   public static final int ACTION_CLOSE =	26;
   /** Code for the select action. */
   public static final int ACTION_SELECT =	27;
   /** Code for the create action. */
   public static final int ACTION_CREATE =	28;
   /** Code for the print action. */
   public static final int ACTION_PRINT =	29;
   /** Code for the view action. */
   public static final int ACTION_VIEW =	30;
   /** Code for the edit action. */
   public static final int ACTION_EDIT =	31;
   /** Code for the delete action. */
   public static final int ACTION_DELETE =	32;
   /** Code for the drop action. */
   public static final int ACTION_DROP =	33;
   /** Code for the build for debug action. */
   public static final int ACTION_BUILD_FOR_DEBUG =	34;
   /** Code for the debug action. */
   public static final int ACTION_DEBUG =	35;
   /** Code for the deploy action. */
   public static final int ACTION_DEPLOY =	36;
   /** Code for the export action. */
   public static final int ACTION_EXPORT =	37;
   /** Code for the import action. */
   public static final int ACTION_IMPORT =	38;
   /** Code for the generate DDL action. */
   public static final int ACTION_GENERATE_DDL = 39;
   /** Code for the Alter action. */
   public static final int ACTION_ALTER = 40;
   /** Code for the Before Run action. */
   public static final int ACTION_BEFORE_RUN = 41;
   /** Code for the After Run action. */
   public static final int ACTION_AFTER_RUN = 42;
   
   /** Code for the next user-supplied action. */
   public static int nextUserAction = 100;
   /** Map of actions, mapping String to int. */
   protected static HashMap actionMap;

   protected String objName;
   protected String action;
   protected String status;
   protected String messages;
   protected String nameQualifier;
   protected String signature;
   protected String uniqueName;
   protected boolean hasParms;
   protected boolean hasResults;
   protected int actionCode;
   protected int statusCode;

   /**
    * Gets the actionCode for a user-supplied action.
    * @param action The display string for the action.
    * @return The action code.
    */
   public static int getActionCode(String action)
   {
      if (actionMap == null)
      {
         return -1;
      }
      else
      {
         Integer mappedAct = (Integer)actionMap.get(action);
         if (mappedAct == null)
         {
            return -1;
         }
         else
         {
            return mappedAct.intValue();
         }   
      }
   }
      
   /**
    * <p>Instantiate an OutputItem for use in the DB Output View.
    * <p>
    * @param statusCode  The int status code as declared in OutputItem (e.g. OutputItem.STATUS_SUCCESS)
    * @param actionCode  The int action code as declared in OutputItem (e.g. OutputItem.ACTION_BUILD)
    * @param objectName  The name of the object against which an action is performed
    * @param uniqueName  The name that uniquely identifies the object. If the objectName/actionCode
    *                    pair may not be unique (e.g. stored procedures objects), you must give
    *                    a unique name (such as the complete SP signature and qualifier) so that
    *                    the entry can be uniquely identified within the Output View.
    */
   public OutputItem(int statusCode, int actionCode, String objectName, String uniqueName)
   {
      setStatus(statusCode);
      setAction(actionCode);
      setObjName(objectName);
      setUniqueName(uniqueName);
      init();
   }

   /**
    * <p>Instantiate an OutputItem for use in the DB Output View.
    * <p>
    * @param statusCode  The int status code as declared in OutputItem (e.g. OutputItem.STATUS_SUCCESS)
    * @param actionCode  The int action code as declared in OutputItem (e.g. OutputItem.ACTION_BUILD)
    * @param objectName  The name of the object against which an action is performed
    *                    If the objectName/actionCode may not be unique (e.g. stored procedures objects), you must give
    *                    you should use the instantiation with the additional uniqueName
    *                    parameter (such as the complete SP signature and qualifier) so that
    *                    the entry can be uniquely identified within the Output View.
    */
   public OutputItem(int statusCode, int actionCode, String objectName)
   {
      setStatus(statusCode);
      setAction(actionCode);
      setObjName(objectName);
      init();
   }

   /**
    * <p>Instantiate an OutputItem for use in the DB Output View.
    * <p>
    * @param statusCode  The int status code as declared in OutputItem (e.g. OutputItem.STATUS_SUCCESS)
    * @param actionCode  The int action code as declared in OutputItem (e.g. OutputItem.ACTION_BUILD)
    * @param objectName  The name of the object against which an action is performed
    * @param message     The message string associated with the action against the object objectName.
    * @param uniqueName  The name that uniquely identifies the object. If the objectName/actionCode
    *                    pair may not be unique (e.g. stored procedures objects), you must give
    *                    a unique name (such as the complete SP signature and qualifier) so that
    *                    the entry can be uniquely identified within the Output View.
    */
   public OutputItem(int statusCode, int actionCode, String objectName, String message, String uniqueName)
   {
      setStatus(statusCode);
      setAction(actionCode);
      setObjName(objectName);
      setMessage(message);
      setUniqueName(uniqueName);
      init();
   }

/**
    * <p>Instantiate an OutputItem for use in the DB Output View.
    * <p>
    * @param statusCode  The int status code as declared in OutputItem (e.g. OutputItem.STATUS_SUCCESS)
    * @param action      The action string (e.g., "Print"). These supplement predefined actionCodes
    *                    and must be translated.
    * @param objectName  The name of the object against which an action is performed
    * @param uniqueName  The name that uniquely identifies the object. If the objectName/actionCode
    *                    pair may not be unique (e.g. stored procedures objects), you must give
    *                    a unique name (such as the complete SP signature and qualifier) so that
    *                    the entry can be uniquely identified within the Output View.
    */
   public OutputItem(int statusCode, String action, String objectName, String uniqueName)
   {
      setStatus(statusCode);
      setObjName(objectName);
      setUniqueName(uniqueName);
      init();
      setAction(action);
   }

   /**
    * <p>Instantiate an OutputItem for use in the DB Output View.
    * <p>
    * @param statusCode  The int status code as declared in OutputItem (e.g. OutputItem.STATUS_SUCCESS)
    * @param action      The action string (e.g., "Print"). These supplement predefined actionCodes
    *                    and must be translated.
    * @param objectName  The name of the object against which an action is performed
    *                    If the objectName/actionCode may not be unique (e.g. stored procedures objects), you must give
    *                    you should use the instantiation with the additional uniqueName
    *                    parameter (such as the complete SP signature and qualifier) so that
    *                    the entry can be uniquely identified within the Output View.
    */
   public OutputItem(int statusCode, String action, String objectName)
   {
      setStatus(statusCode);
      setObjName(objectName);
      init();
      setAction(action);
   }

   /**
    * <p>Instantiate an OutputItem for use in the DB Output View.
    * <p>
    * @param statusCode  The int status code as declared in OutputItem (e.g. OutputItem.STATUS_SUCCESS)
    * @param action      The action string (e.g., "Print"). These supplement predefined actionCodes
    *                    and must be translated.
    * @param objectName  The name of the object against which an action is performed
    * @param message     The message string associated with the action against the object objectName.
    * @param uniqueName  The name that uniquely identifies the object. If the objectName/actionCode
    *                    pair may not be unique (e.g. stored procedures objects), you must give
    *                    a unique name (such as the complete SP signature and qualifier) so that
    *                    the entry can be uniquely identified within the Output View.
    */
   public OutputItem(int statusCode, String action, String objectName, String message, String uniqueName)
   {
      setStatus(statusCode);
      setObjName(objectName);
      setMessage(message);
      setUniqueName(uniqueName);
      init();
      setAction(action);
   }

   /**
    * Initializes variables.
    */
   private void init()
   {
      messages = "";
      hasParms = false;
      hasResults = false;
      if (actionMap == null)
      {
         actionMap = new HashMap();
         actionMap.put(OutputViewPlugin.getString("OV_ACTION_RUN"), new Integer(ACTION_RUN));
         actionMap.put(OutputViewPlugin.getString("OV_ACTION_BUILD"), new Integer(ACTION_BUILD));
         actionMap.put(OutputViewPlugin.getString("OV_ACTION_CREATE"), new Integer(ACTION_CREATE));
         actionMap.put(OutputViewPlugin.getString("OV_ACTION_OPEN"), new Integer(ACTION_OPEN));
         actionMap.put(OutputViewPlugin.getString("OV_ACTION_SAVE"), new Integer(ACTION_SAVE));
         actionMap.put(OutputViewPlugin.getString("OV_ACTION_CLOSE"), new Integer(ACTION_CLOSE));
         actionMap.put(OutputViewPlugin.getString("OV_ACTION_CALL"), new Integer(ACTION_CALL));
         actionMap.put(OutputViewPlugin.getString("OV_ACTION_EXECUTE"), new Integer(ACTION_EXECUTE));
         actionMap.put(OutputViewPlugin.getString("OV_ACTION_SELECT"), new Integer(ACTION_SELECT));
         actionMap.put(OutputViewPlugin.getString("OV_ACTION_PRINT"), new Integer(ACTION_PRINT));
         actionMap.put(OutputViewPlugin.getString("OV_ACTION_VIEW"), new Integer(ACTION_VIEW));
         actionMap.put(OutputViewPlugin.getString("OV_ACTION_EDIT"), new Integer(ACTION_EDIT));
         actionMap.put(OutputViewPlugin.getString("OV_ACTION_DELETE"), new Integer(ACTION_DELETE));
         actionMap.put(OutputViewPlugin.getString("OV_ACTION_DROP"), new Integer(ACTION_DROP));
         actionMap.put(OutputViewPlugin.getString("OV_ACTION_DEBUG"), new Integer(ACTION_DEBUG));
         actionMap.put(OutputViewPlugin.getString("OV_ACTION_BUILD_FOR_DEBUG"), new Integer(ACTION_BUILD_FOR_DEBUG));
         actionMap.put(OutputViewPlugin.getString("OV_ACTION_DEPLOY"), new Integer(ACTION_DEPLOY));
         actionMap.put(OutputViewPlugin.getString("OV_ACTION_EXPORT"), new Integer(ACTION_EXPORT));
         actionMap.put(OutputViewPlugin.getString("OV_ACTION_IMPORT"), new Integer(ACTION_IMPORT));
         actionMap.put(OutputViewPlugin.getString("OV_ACTION_GENERATEDDL"), new Integer(ACTION_GENERATE_DDL));
         actionMap.put(OutputViewPlugin.getString("OV_ACTION_ALTER"), new Integer(ACTION_ALTER));
         actionMap.put(OutputViewPlugin.getString("OV_ACTION_BEFORE_RUN"), new Integer(ACTION_BEFORE_RUN));
         actionMap.put(OutputViewPlugin.getString("OV_ACTION_AFTER_RUN"), new Integer(ACTION_AFTER_RUN));
      }
   }

   /**
    * Gets the item's object name.
    * @return The item's object name.
    */
   protected String getObjName()
   {
      return objName;
   }

   /**
    * Gets the item's messages.
    * @return The item's messages.
    */
   protected String getMessages()
   {
      return messages;
   }

   /**
    * Gets the item's status.
    * @return The item's status.
    */
   protected String getStatus()
   {
      return status;
   }

   /**
    * Gets the item's status code.
    * @return The item's status code.
    */
   public int getStatusCode()
   {
      return statusCode;
   }

   /**
    * Gets the item's action.
    * @return The item's action.
    */
   protected String getAction()
   {
      return action;
   }

   /**
    * Gets the item's action code.
    * @return The item's action code.
    */
   protected int getActionCode()
   {
      return actionCode;
   }

   /**
    * Gets the item's unique name.
    * @return The item's unique name.
    */
   protected String getUniqueName()
   {
      if (uniqueName == null || uniqueName.equals("") )
         return objName;
      else
         return uniqueName;
   }

   /**
    * Determines if the action succeeded.
    * @return True if the status is success.
    */
   protected boolean isStatusSuccess()
   {
      if  (statusCode == STATUS_SUCCESS)
         return true;
      else
         return false;
   }

   /**
    * Determines if the item has parameters.
    * @return True if the item has parameters.
    */
   protected boolean hasParameters()
   {
      return hasParms;
   }

   /**
    * Determines if the item has results.
    * @return True if the item has results.
    */
   protected boolean hasResults()
   {
      return hasResults;
   }

   /**
    * Sets whether the item has parameters.
    * @param value True if the item has parameters.
    */
   protected void setHasParameters(boolean value)
   {
      hasParms = value;
   }

   /**
    * Sets whether the item has results.
    * @param value True if the item has results.
    */
   protected void setHasResults(boolean value)
   {
      hasResults = value;
   }

   /**
    * Sets the item's object name.
    * @param name The item's object name.
    */
   protected void setObjName(String name)
   {
      objName = name;
   }

   /**
    * Sets the item's name qualifier.
    * @param qualifier The item's name qualifier.
    */
   protected void setNameQualifier(String qualifier)
   {
      nameQualifier = qualifier;
   }

   /**
    * Sets the item's messages.
    * @param msg The item's messages.
    */
   protected void setMessage(String msg)
   {
      messages = msg;
   }

   /**
    * Sets the item's unique name.
    * @param uniqueName The item's unique name.
    */
   protected void setUniqueName(String uniqueName)
   {
      this.uniqueName = uniqueName;
   }

   /**
    * Appends to the item's messages.
    * @param msgs Messages to append to the item's messages.
    */
   protected void appendMessage(String msgs)
   {
      if (messages.length() == 0)
         messages = msgs ;
      else
         messages = messages + "\n\n" + msgs;
   }


   /**
    *  Sets the current status of the item.
    *  @param status The status of the current action.
    */
   public void setStatus(int statusCode)
   {
      this.statusCode = statusCode;
      status = statusFlagToString(statusCode);
   }

   /**
    *  Sets the action code in the item.
    *  @param action The action code of the item.
    */
   protected void setAction(int actionCode)
   {
      this.actionCode = actionCode;
      action = actionFlagToString(actionCode);
   }

   /**
    *  Sets the action in the item.
    *  @param action The displayable action for the item.
    */
   protected void setAction(String action)
   {
      this.action = action;
      Integer mappedAct = (Integer)actionMap.get(action);
      if (mappedAct == null)
      {
         actionCode = nextUserAction++;
         actionMap.put(action, new Integer(actionCode));
      }
      else
      {
         actionCode = mappedAct.intValue();
      }
   }

   /**
    *  Gets the status icon.
    *  @return  A Image object of the status.
    */
   public Image getStatusImage()
   {
      switch (statusCode)
      {
      case STATUS_START :
         return OutputViewPlugin.getPlugin().getImage("task_inprogress_misc");
      case STATUS_SUCCESS :
         return OutputViewPlugin.getPlugin().getImage("task_success_misc");
      case STATUS_WARNING :
         return OutputViewPlugin.getPlugin().getImage("task_warning_misc");
      case STATUS_FAILURE :
         return OutputViewPlugin.getPlugin().getImage("task_error_misc");
      case STATUS_IN_PROGRESS :
         return OutputViewPlugin.getPlugin().getImage("task_inprogress_misc");
      case STATUS_CRITICAL_ERROR :
         return OutputViewPlugin.getPlugin().getImage("task_error_misc");
      default:
         return OutputViewPlugin.getPlugin().getImage("task_inprogress_misc");
      }
   }

   /**
    * Converts the status flag from int to a String (translated).
    * @param statusFlag An integer value of the status to be converted.
    * @return The translated status.
    */
   private String statusFlagToString(int statusFlag)
   {
      switch (statusFlag)
      {
      case STATUS_START:
         return OutputViewPlugin.getString("OV_STATUS_START");
      case STATUS_SUCCESS:
         return OutputViewPlugin.getString("OV_STATUS_SUCCESS");
      case STATUS_WARNING:
         return OutputViewPlugin.getString("OV_STATUS_WARNING");
      case STATUS_FAILURE:
         return OutputViewPlugin.getString("OV_STATUS_FAILURE");
      case STATUS_IN_PROGRESS:
         return OutputViewPlugin.getString("OV_STATUS_IN_PROGRESS");
      case STATUS_CRITICAL_ERROR:
         return OutputViewPlugin.getString("OV_STATUS_CRITICAL_ERROR");
      default:
         return OutputViewPlugin.getString("OV_STATUS_UNKNOWN");
      }
   }

   /**
    * Converts the action flag from int to a String (translated).
    * @param statusFlag An integer value of the status to be converted
    * @return The translated action.
    */
   private String actionFlagToString(int actionFlag)
   {
      switch (actionFlag)
      {
      case ACTION_RUN:
         return OutputViewPlugin.getString("OV_ACTION_RUN");
      case ACTION_BUILD:
         return OutputViewPlugin.getString("OV_ACTION_BUILD");
      case ACTION_CREATE:
         return OutputViewPlugin.getString("OV_ACTION_CREATE");
      case ACTION_OPEN:
         return OutputViewPlugin.getString("OV_ACTION_OPEN");
      case ACTION_SAVE:
         return OutputViewPlugin.getString("OV_ACTION_SAVE");
      case ACTION_CLOSE:
         return OutputViewPlugin.getString("OV_ACTION_CLOSE");
      case ACTION_CALL:
         return OutputViewPlugin.getString("OV_ACTION_CALL");
      case ACTION_EXECUTE:
         return OutputViewPlugin.getString("OV_ACTION_RUN");
      case ACTION_SELECT:
         return OutputViewPlugin.getString("OV_ACTION_SELECT");
      case ACTION_PRINT:
         return OutputViewPlugin.getString("OV_ACTION_PRINT");
      case ACTION_VIEW:
         return OutputViewPlugin.getString("OV_ACTION_VIEW");
      case ACTION_EDIT:
         return OutputViewPlugin.getString("OV_ACTION_EDIT");
      case ACTION_DELETE:
         return OutputViewPlugin.getString("OV_ACTION_DELETE");
      case ACTION_DROP:
         return OutputViewPlugin.getString("OV_ACTION_DROP");
      case ACTION_DEBUG:
         return OutputViewPlugin.getString("OV_ACTION_DEBUG");
      case ACTION_BUILD_FOR_DEBUG:
         return OutputViewPlugin.getString("OV_ACTION_BUILD_FOR_DEBUG");
      case ACTION_DEPLOY:
         return OutputViewPlugin.getString("OV_ACTION_DEPLOY");
      case ACTION_EXPORT:
         return OutputViewPlugin.getString("OV_ACTION_EXPORT");
      case ACTION_IMPORT:
         return OutputViewPlugin.getString("OV_ACTION_IMPORT");
      case ACTION_GENERATE_DDL:
         return OutputViewPlugin.getString("OV_ACTION_GENERATEDDL");
      case ACTION_ALTER:
         return OutputViewPlugin.getString("OV_ACTION_ALTER");
      case ACTION_BEFORE_RUN:
        return OutputViewPlugin.getString("OV_ACTION_BEFORE_RUN");
      case ACTION_AFTER_RUN:
        return OutputViewPlugin.getString("OV_ACTION_AFTER_RUN");
      default:
         return OutputViewPlugin.getString("OV_STATUS_UNKNOWN");
      }
   }

   /**
    * Compares two OutputItems.
    * @param obj An item to compare.
    * @return True if the given item is equal to this item.
    */
   public boolean equals(OutputItem obj)
   {

      if ( obj.getObjName().equals(getObjName())
      && obj.getUniqueName().equals(getUniqueName())
      && obj.getActionCode() == getActionCode() )
      {
         return true;
      }
      else
      {
         return false;
      }
   }

}
