/*******************************************************************************
* Copyright (c) 2000, 2004 IBM Corporation and others.
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
*
* Contributors:
*     IBM Corporation - initial API and implementation
*******************************************************************************/
package org.eclipse.wst.rdb.internal.outputview;

import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Types;
import java.sql.Clob;
import java.util.ArrayList;
import java.io.BufferedInputStream;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.wst.rdb.core.internal.ui.RDBCoreUIPlugin;
import org.eclipse.wst.rdb.internal.core.*;
import org.eclipse.wst.rdb.internal.core.util.*;

/**
 * A table model for a result set to be displayed in the output view.
 */
public class ResultTableModel
{
   /** Our column names. */
   protected String[] columnNames;
   /** Our table rows. */
   protected ArrayList tableRows;
   /** The row count. */
   protected int numRows;
   /** The column count. */
   protected int columnCount;
   /** True if we have result sets. */
   protected boolean haveRS = false;
   /** Our column type codes. */
   protected int[] columnTypes;
   /** Our column widths. */
   protected int[] columnLengths;
   /** Hexadecimal digits */
   protected char [] hexDigits = {'0', '1', '2', '3', '4', '5', '6', '7', '8',
   		'9', 'a', 'b', 'c', 'd', 'e', 'f' };      
   
   /** Constructs a default model. */
   public ResultTableModel()
   {
      haveRS = false;
      numRows = 0;
      columnCount = 0;
      tableRows = new ArrayList();
   }
   
   /**
    * Constructs a model for a given result set.
    * @param rs A ResultSet.
    * @throws SQLException
    */
   public ResultTableModel(ResultSet rs) throws SQLException
   {
      if (rs != null)
      {
         haveRS = true;
         int maxLOBLengthRetrieved = RDBCorePlugin.getDefault().getPluginPreferences(
             ).getInt(RDBCorePluginConstants.MAX_LOB_LENGTH);
         try
         {
            tableRows = new ArrayList();
            ResultSetMetaData rmeta = rs.getMetaData();
            columnCount = rmeta.getColumnCount();
            columnNames = new String[columnCount];
            columnTypes = new int[columnCount];
            columnLengths = new int[columnCount];
            
            for (int j=1;j<=columnCount;j++)
            {
               columnNames[j-1] = rmeta.getColumnName(j);
               columnTypes[j-1] = rmeta.getColumnType(j);
               try {
                  columnLengths[j-1] = rmeta.getPrecision(j);
               } catch (Exception ep) {
                  columnLengths[j-1] = 5;
               } 
            }
            numRows = 0;
            ResultTableRow row;
            while (rs.next() )
            {
               row = new ResultTableRow(columnCount);
               for (int i=1; i <= columnCount; i++)
               {
                String value = null;
                
                  int colType = rmeta.getColumnType(i);
                  if (colType == Types.BLOB)
                  {
                  	value = getValueForBLOB(rs, i, maxLOBLengthRetrieved);
                  }
                  else if (colType == Types.CLOB)
                  {
                  	value = getValueForCLOB(rs, i, maxLOBLengthRetrieved);
                  }
               	  else
                  {
               	  	try {
               	  		value = rs.getString(i);
               	  	} catch (Exception ev) {}
                  }
                  
                  if (value == null)
                     value = "-";
                  row.setColumn(value, i-1);
               }
               tableRows.add(row);
               numRows++;
            }
         }
         catch (SQLException sqlE) {
            throw sqlE;  // this error is handled properly by the calling methods.
         }
         catch (Exception e)
         {
            OutputViewPlugin.getPlugin().writeLog(IStatus.ERROR, 0, 
                    "ResultTableModel:constructor.",e); // $NON-NLS-1$
         }
      }
   }
   
    /**
     * Gets the value for CLOB column type.
     * The length of the value returned is from the user preference for LOB bytes retrieved.
     * <p>
     * The column type for the column is assumed to be CLOB
     * @param aResult the ResultSet to retrieve the data
     * @param index the column index to retrieve value from
     * @param maxLength the maximum length in bytes to retrieve
     * @return String of the column up to the length for LOB preference
     */
    private String getValueForCLOB(ResultSet aResult, int index, int maxLength)
    {
    	try
        {
   	    	Clob clob = aResult.getClob(index);            
            if (clob != null)
            {            	
                String stringValue = clob.getSubString(1, maxLength);                  
                return stringValue;
            }
        }
        catch (Exception ex)
        {
        	OutputViewPlugin.getPlugin().writeLog(IStatus.ERROR, 0, 
                    "ResultTableModel:getValueForCLOB.",ex); //$NON-NLS-1$
        }
   	    return null;        
   }
    
    /**
     * Gets the value for BLOB column type.
     * The length of the value returned is from the user preference for LOB bytes retrieved.
     * <p>
     * The column type for the column is assumed to be BLOB
     * @param aResult the ResultSet to retrieve the data
     * @param index the column index to retrieve value from
     * @param maxLength the maximum length in bytes to retrieve
     * @return String of the column up to the length for LOB preference
     */
    private String getValueForBLOB(ResultSet aResult, int index, int maxLength)
    {
        try
        {
            BufferedInputStream stream = 
                new BufferedInputStream(aResult.getBinaryStream(index));            
            byte [] byteArray = new byte[maxLength];
            int bytesRead = stream.read(byteArray, 0, maxLength);
            if (bytesRead > 0)
            {
                StringBuffer sb = new StringBuffer(bytesRead * 2);
                for (int i=0;i<bytesRead;i++)
                {
                	// high nibble
                    sb.append(hexDigits[(byteArray[i] & 0xF0) >>> 4]);
                    // low nibble
                    sb.append(hexDigits[byteArray[i] & 0x0F]);
                }                
                String stringValue = sb.toString();               
                
                return stringValue;
            }
        }
        catch (Exception ex)
        {
            OutputViewPlugin.getPlugin().writeLog(IStatus.ERROR, 0, 
                    "ResultTableModel:getValueForBLOB.",ex); //$NON-NLS-1$
        }
        return null;        
   }
   
   /**
    * Constructs a model for a procedure call.
    * @param proc The procedure all information.
    * @throws SQLException
    */
   public ResultTableModel(ProcedureCallCache proc) throws SQLException
   {
      if (proc != null)
      {
         haveRS = false;
         tableRows = new ArrayList();
         
         
         columnCount = proc.getColumnCount();
         columnNames = new String[columnCount];
         columnTypes = new int[columnCount];
         columnLengths = new int[columnCount];
         
         for (int j=1;j<=columnCount;j++)
         {
            columnNames[j-1] = proc.getColumnName(j);
            columnTypes[j-1] = proc.getColumnSQLType(j);
            columnLengths[j-1] = proc.getColumnLength(j);
         }
         
         numRows = proc.getNumRows();
         ResultTableRow row;
         for (int i=0;i<numRows;i++)
         {
            row = new ResultTableRow(columnCount);
            for (int j=1;j<=columnCount;j++)
            {
               row.setColumn(proc.getColumnValueToString(j),j-1);
            }
            tableRows.add(row);
            
            proc.nextRow();
            
         }
      } // proc != null
      
   }
   
   /**
    * Gets the row for a given index.
    * @param rowIndex The index.
    * @return The corresponding ResultTableRow.
    */
   public ResultTableRow getRow(int rowIndex)
   {
      return  (ResultTableRow)tableRows.get(rowIndex);
   }
   
   /**
    * Gets the table rows.
    * @return A list of rows.
    */
   public ArrayList getTableRows()
   {
      return tableRows;
   }
   
   /**
    * Gets the column count. 
    * @return The number of columns.
    */
   public int getColumnCount()
   {
      return columnCount;
   }
   
   /**
    * Gets the count of result sets.
    * @return The number of result sets.
    */
   protected int getResultSetRowCount()
   {
      if (haveRS)
         return numRows;
      else
         return -1;
   }
   
   /**
    * Gets the column names.
    * @return An array of column names.
    */
   public String[] getColumnNames()
   {
      return columnNames;
   }
   
   /** 
    * Gets the column type codes.
    * @param An array of column type codes.
    */
   public int[] getColumnTypes()
   {
      return columnTypes;
   }
   
   /** 
    * Gets the column widths.
    * @param An array of column widths.
    */
   public int[] getColumnLengths()
   {
      return columnLengths;
   }
   
   
}
